package docker // import "docker.io/go-docker"

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"testing"

	"docker.io/go-docker/api/types/swarm"
	"golang.org/x/net/context"
)

func TestTaskInspectError(t *testing.T) {
	client := &Client{
		client: newMockClient(errorMock(http.StatusInternalServerError, "Server error")),
	}

	_, _, err := client.TaskInspectWithRaw(context.Background(), "nothing")
	if err == nil || err.Error() != "Error response from daemon: Server error" {
		t.Fatalf("expected a Server Error, got %v", err)
	}
}

func TestTaskInspect(t *testing.T) {
	expectedURL := "/tasks/task_id"
	client := &Client{
		client: newMockClient(func(req *http.Request) (*http.Response, error) {
			if !strings.HasPrefix(req.URL.Path, expectedURL) {
				return nil, fmt.Errorf("Expected URL '%s', got '%s'", expectedURL, req.URL)
			}
			content, err := json.Marshal(swarm.Task{
				ID: "task_id",
			})
			if err != nil {
				return nil, err
			}
			return &http.Response{
				StatusCode: http.StatusOK,
				Body:       ioutil.NopCloser(bytes.NewReader(content)),
			}, nil
		}),
	}

	taskInspect, _, err := client.TaskInspectWithRaw(context.Background(), "task_id")
	if err != nil {
		t.Fatal(err)
	}
	if taskInspect.ID != "task_id" {
		t.Fatalf("expected `task_id`, got %s", taskInspect.ID)
	}
}
