/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::bessel::j0f::j1f_rsqrt;
use crate::common::f_fmla;
use crate::exponents::core_expf;
use crate::polyeval::{f_estrin_polyeval7, f_estrin_polyeval9, f_polyeval10};

/// Modified Bessel of the first kind order 1
///
/// Max ULP 0.5
pub fn f_i1f(x: f32) -> f32 {
    if (x.to_bits() & 0x0007_ffff) == 0 {
        if x == 0. {
            return 0.;
        }
        if x.is_infinite() {
            return if x.is_sign_positive() {
                f32::INFINITY
            } else {
                f32::NEG_INFINITY
            };
        }
        if x.is_nan() {
            return f32::NAN;
        }
    }

    let xb = x.to_bits() & 0x7fff_ffff;

    if xb > 0x42b7d001 {
        // 91.906261
        return if x.is_sign_negative() {
            f32::NEG_INFINITY
        } else {
            f32::INFINITY
        };
    }

    static SIGN: [f64; 2] = [1., -1.];

    let sign_scale = SIGN[x.is_sign_negative() as usize];

    if xb <= 0x40f80000u32 {
        // |x| <= 7.75
        return i1f_small(f32::from_bits(xb), sign_scale) as f32;
    }

    i1f_asympt(f32::from_bits(xb), sign_scale)
}

/**
Computes
I1(x) = x/2 * (1 + 1 * (x/2)^2 + (x/2)^4 * P((x/2)^2))

Generated by Woflram Mathematica:

```text
<<FunctionApproximations`
ClearAll["Global`*"]
f[x_]:=(BesselI[1,x]*2/x-1-1/2(x/2)^2)/(x/2)^4
g[z_]:=f[2 Sqrt[z]]
{err, approx}=MiniMaxApproximation[g[z],{z,{0.000000001,7.75},6,6},WorkingPrecision->60]
poly=Numerator[approx][[1]];
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
poly=Denominator[approx][[1]];
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
```
**/
#[inline]
fn i1f_small(x: f32, sign_scale: f64) -> f64 {
    let dx = x as f64;
    let x_over_two = dx * 0.5;
    let x_over_two_sqr = x_over_two * x_over_two;
    let x_over_two_p4 = x_over_two_sqr * x_over_two_sqr;

    let p_num = f_estrin_polyeval7(
        x_over_two_sqr,
        f64::from_bits(0x3fb5555555555555),
        f64::from_bits(0x3f706cdccca396c4),
        f64::from_bits(0x3f23f9e12bdbba92),
        f64::from_bits(0x3ec8e39208e926b2),
        f64::from_bits(0x3e62e53b433c42ff),
        f64::from_bits(0x3def7cb16d10fb46),
        f64::from_bits(0x3d6747cd73d9d783),
    );
    let p_den = f_estrin_polyeval7(
        x_over_two_sqr,
        f64::from_bits(0x3ff0000000000000),
        f64::from_bits(0xbfa2075f77b54885),
        f64::from_bits(0x3f438c6d797c29f5),
        f64::from_bits(0xbeda57e2a258c6da),
        f64::from_bits(0x3e677e777c569432),
        f64::from_bits(0xbdea9212a96babc1),
        f64::from_bits(0x3d5e183186d5d782),
    );
    let p = p_num / p_den;

    let p1 = f_fmla(0.5, x_over_two_sqr, 1.);
    let p2 = f_fmla(x_over_two_p4, p, p1);
    p2 * x_over_two * sign_scale
}

/**
Asymptotic expansion for I1.

Computes:
sqrt(x) * exp(-x) * I1(x) = Pn(1/x)/Qn(1/x)
hence:
I1(x) = Pn(1/x)/Qm(1/x)*exp(x)/sqrt(x)

Generated by Wolfram Mathematica:
```text
<<FunctionApproximations`
ClearAll["Global`*"]
f[x_]:=Sqrt[x] Exp[-x] BesselI[1,x]
g[z_]:=f[1/z]
{err, approx,err1}=MiniMaxApproximation[g[z],{z,{1/91.9,1/7.75},9,8},WorkingPrecision->60]
poly=Numerator[approx];
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
poly=Denominator[approx];
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
```
**/
#[inline]
fn i1f_asympt(x: f32, sign_scale: f64) -> f32 {
    let dx = x as f64;
    let recip = 1. / dx;
    let p_num = f_polyeval10(
        recip,
        f64::from_bits(0x3fd9884533d43711),
        f64::from_bits(0xc0309c047537243a),
        f64::from_bits(0x4073bdb14a29bf68),
        f64::from_bits(0xc0aaf9eca14d15af),
        f64::from_bits(0x40d6c629318a9e42),
        f64::from_bits(0xc0f7bee33088a4b0),
        f64::from_bits(0x410d018cef093ee2),
        f64::from_bits(0xc111f32b325d3fe4),
        f64::from_bits(0x4100dddad80e0b42),
        f64::from_bits(0xc0c96006c91a00e2),
    );
    let p_den = f_estrin_polyeval9(
        recip,
        f64::from_bits(0x3ff0000000000000),
        f64::from_bits(0xc044a11d10bae889),
        f64::from_bits(0x408843069497d993),
        f64::from_bits(0xc0c058710de4b9b9),
        f64::from_bits(0x40eb0d97f71420ae),
        f64::from_bits(0xc10b55d181ef9ea1),
        f64::from_bits(0x411f9413e1932a48),
        f64::from_bits(0xc1213bff5bc7d2d6),
        f64::from_bits(0x4105c53e92d9b9c0),
    );
    let z = p_num / p_den;
    let e = core_expf(x);
    let r_sqrt = j1f_rsqrt(dx);
    (z * r_sqrt * e * sign_scale) as f32
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_i1f() {
        assert!(f_i1f(f32::NAN).is_nan());
        assert!(f_i1f(f32::INFINITY).is_infinite());
        assert!(f_i1f(f32::NEG_INFINITY).is_infinite());
        assert_eq!(f_i1f(0.), 0.);
        assert_eq!(f_i1f(1.), 0.5651591);
        assert_eq!(f_i1f(-1.), -0.5651591);
        assert_eq!(f_i1f(9.), 1030.9147);
        assert_eq!(f_i1f(-9.), -1030.9147);
    }
}
