;; -*- mode: scheme; coding: utf-8 -*-
;; Copyright © 2018 Göran Weinholt <goran@weinholt.se>
;; SPDX-License-Identifier: MIT
#!r6rs

;; The tables are based on the SPDX license list at
;; https://github.com/spdx/license-list-data

(library (spdx ids)
  (export
    licenses
    license
    license-exceptions
    license-exception)
  (import
    (rnrs)
    (only (srfi :43 vectors) vector-binary-search)
    (srfi :67 compare-procedures))

(define (licenses)
  (vector->list (vector-map car *licenses*)))

(define (license name)
  (assert (string? name))
  (let ((idx (vector-binary-search *licenses* name
                                   (lambda (v n)
                                     (string-compare-ci (car v) n)))))
    (and idx
         (let ((entry (vector-ref *licenses* idx)))
           (cons (cons 'id (car entry)) (cdr entry))))))

(define (license-exceptions)
  (vector->list (vector-map car *exceptions*)))

(define (license-exception name)
  (assert (string? name))
  (let ((idx (vector-binary-search *exceptions* name
                                   (lambda (v n)
                                     (string-compare-ci (car v) n)))))
    (and idx
         (let ((entry (vector-ref *exceptions* idx)))
           (cons (cons 'id (car entry)) (cdr entry))))))

(define (identifier-list-version)
  "3.0")

;; XXX: (vector-sort string-ci<? *exceptions*)
(define *exceptions*
  '#(
     ("389-exception" (name . "389 Directory Server Exception"))
     ("Autoconf-exception-2.0" (name . "Autoconf exception 2.0"))
     ("Autoconf-exception-3.0" (name . "Autoconf exception 3.0"))
     ("Bison-exception-2.2" (name . "Bison exception 2.2"))
     ("Bootloader-exception" (name . "Bootloader Distribution Exception"))
     ("Classpath-exception-2.0" (name . "Classpath exception 2.0"))
     ("CLISP-exception-2.0" (name . "CLISP exception 2.0"))
     ("DigiRule-FOSS-exception" (name . "DigiRule FOSS License Exception"))
     ("eCos-exception-2.0" (name . "eCos exception 2.0"))
     ("Fawkes-Runtime-exception" (name . "Fawkes Runtime Exception"))
     ("FLTK-exception" (name . "FLTK exception"))
     ("Font-exception-2.0" (name . "Font exception 2.0"))
     ("freertos-exception-2.0" (name . "FreeRTOS Exception 2.0"))
     ("GCC-exception-2.0" (name . "GCC Runtime Library exception 2.0"))
     ("GCC-exception-3.1" (name . "GCC Runtime Library exception 3.1"))
     ("gnu-javamail-exception" (name . "GNU JavaMail exception"))
     ("i2p-gpl-java-exception" (name . "i2p GPL+Java Exception"))
     ("Libtool-exception" (name . "Libtool Exception"))
     ("Linux-syscall-note" (name . "Linux Syscall Note"))
     ("LZMA-exception" (name . "LZMA exception"))
     ("mif-exception" (name . "Macros and Inline Functions Exception"))
     ("Nokia-Qt-exception-1.1" (name . "Nokia Qt LGPL exception 1.1"))
     ("OCCT-exception-1.0" (name . "Open CASCADE Exception 1.0"))
     ("openvpn-openssl-exception" (name . "OpenVPN OpenSSL Exception"))
     ("Qwt-exception-1.0" (name . "Qwt exception 1.0"))
     ("u-boot-exception-2.0" (name . "U-Boot exception 2.0"))
     ("WxWindows-exception-3.1" (name . "WxWindows Library Exception 3.1")))
  )

;; XXX: (vector-sort string-ci<? *licenses*)
(define *licenses*
  '#(
     ("0BSD" . ((name . "BSD Zero Clause License") (osi-approved? . #f) (deprecated? . #f)))
     ("AAL" . ((name . "Attribution Assurance License") (osi-approved? . #t) (deprecated? . #f)))
     ("Abstyles" . ((name . "Abstyles License") (osi-approved? . #f) (deprecated? . #f)))
     ("Adobe-2006" . ((name . "Adobe Systems Incorporated Source Code License Agreement") (osi-approved? . #f) (deprecated? . #f)))
     ("Adobe-Glyph" . ((name . "Adobe Glyph List License") (osi-approved? . #f) (deprecated? . #f)))
     ("ADSL" . ((name . "Amazon Digital Services License") (osi-approved? . #f) (deprecated? . #f)))
     ("AFL-1.1" . ((name . "Academic Free License v1.1") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("AFL-1.2" . ((name . "Academic Free License v1.2") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("AFL-2.0" . ((name . "Academic Free License v2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("AFL-2.1" . ((name . "Academic Free License v2.1") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("AFL-3.0" . ((name . "Academic Free License v3.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Afmparse" . ((name . "Afmparse License") (osi-approved? . #f) (deprecated? . #f)))
     ("AGPL-1.0" . ((name . "Affero General Public License v1.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("AGPL-3.0" . ((name . "GNU Affero General Public License v3.0") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("AGPL-3.0-only" . ((name . "GNU Affero General Public License v3.0 only") (osi-approved? . #t) (deprecated? . #f)))
     ("AGPL-3.0-or-later" . ((name . "GNU Affero General Public License v3.0 or later") (osi-approved? . #t) (deprecated? . #f)))
     ("Aladdin" . ((name . "Aladdin Free Public License") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("AMDPLPA" . ((name . "AMD's plpa_map.c License") (osi-approved? . #f) (deprecated? . #f)))
     ("AML" . ((name . "Apple MIT License") (osi-approved? . #f) (deprecated? . #f)))
     ("AMPAS" . ((name . "Academy of Motion Picture Arts and Sciences BSD") (osi-approved? . #f) (deprecated? . #f)))
     ("ANTLR-PD" . ((name . "ANTLR Software Rights Notice") (osi-approved? . #f) (deprecated? . #f)))
     ("Apache-1.0" . ((name . "Apache License 1.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Apache-1.1" . ((name . "Apache License 1.1") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Apache-2.0" . ((name . "Apache License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("APAFML" . ((name . "Adobe Postscript AFM License") (osi-approved? . #f) (deprecated? . #f)))
     ("APL-1.0" . ((name . "Adaptive Public License 1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("APSL-1.0" . ((name . "Apple Public Source License 1.0") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #f)))
     ("APSL-1.1" . ((name . "Apple Public Source License 1.1") (osi-approved? . #t) (deprecated? . #f)))
     ("APSL-1.2" . ((name . "Apple Public Source License 1.2") (osi-approved? . #t) (deprecated? . #f)))
     ("APSL-2.0" . ((name . "Apple Public Source License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Artistic-1.0" . ((name . "Artistic License 1.0") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #f)))
     ("Artistic-1.0-cl8" . ((name . "Artistic License 1.0 w/clause 8") (osi-approved? . #t) (deprecated? . #f)))
     ("Artistic-1.0-Perl" . ((name . "Artistic License 1.0 (Perl)") (osi-approved? . #t) (deprecated? . #f)))
     ("Artistic-2.0" . ((name . "Artistic License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Bahyph" . ((name . "Bahyph License") (osi-approved? . #f) (deprecated? . #f)))
     ("Barr" . ((name . "Barr License") (osi-approved? . #f) (deprecated? . #f)))
     ("Beerware" . ((name . "Beerware License") (osi-approved? . #f) (deprecated? . #f)))
     ("BitTorrent-1.0" . ((name . "BitTorrent Open Source License v1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("BitTorrent-1.1" . ((name . "BitTorrent Open Source License v1.1") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Borceux" . ((name . "Borceux license") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-1-Clause" . ((name . "BSD 1-Clause License") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-2-Clause" . ((name . "BSD 2-Clause \"Simplified\" License") (osi-approved? . #t) (deprecated? . #f)))
     ("BSD-2-Clause-FreeBSD" . ((name . "BSD 2-Clause FreeBSD License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-2-Clause-NetBSD" . ((name . "BSD 2-Clause NetBSD License") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-2-Clause-Patent" . ((name . "BSD-2-Clause Plus Patent License") (osi-approved? . #t) (deprecated? . #f)))
     ("BSD-3-Clause" . ((name . "BSD 3-Clause \"New\" or \"Revised\" License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("BSD-3-Clause-Attribution" . ((name . "BSD with attribution") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-3-Clause-Clear" . ((name . "BSD 3-Clause Clear License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-3-Clause-LBNL" . ((name . "Lawrence Berkeley National Labs BSD variant license") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-3-Clause-No-Nuclear-License" . ((name . "BSD 3-Clause No Nuclear License") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-3-Clause-No-Nuclear-License-2014" . ((name . "BSD 3-Clause No Nuclear License 2014") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-3-Clause-No-Nuclear-Warranty" . ((name . "BSD 3-Clause No Nuclear Warranty") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-4-Clause" . ((name . "BSD 4-Clause \"Original\" or \"Old\" License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-4-Clause-UC" . ((name . "BSD-4-Clause (University of California-Specific)") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-Protection" . ((name . "BSD Protection License") (osi-approved? . #f) (deprecated? . #f)))
     ("BSD-Source-Code" . ((name . "BSD Source Code Attribution") (osi-approved? . #f) (deprecated? . #f)))
     ("BSL-1.0" . ((name . "Boost Software License 1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("bzip2-1.0.5" . ((name . "bzip2 and libbzip2 License v1.0.5") (osi-approved? . #f) (deprecated? . #f)))
     ("bzip2-1.0.6" . ((name . "bzip2 and libbzip2 License v1.0.6") (osi-approved? . #f) (deprecated? . #f)))
     ("Caldera" . ((name . "Caldera License") (osi-approved? . #f) (deprecated? . #f)))
     ("CATOSL-1.1" . ((name . "Computer Associates Trusted Open Source License 1.1") (osi-approved? . #t) (deprecated? . #f)))
     ("CC-BY-1.0" . ((name . "Creative Commons Attribution 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-2.0" . ((name . "Creative Commons Attribution 2.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-2.5" . ((name . "Creative Commons Attribution 2.5") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-3.0" . ((name . "Creative Commons Attribution 3.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-4.0" . ((name . "Creative Commons Attribution 4.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-1.0" . ((name . "Creative Commons Attribution Non Commercial 1.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-2.0" . ((name . "Creative Commons Attribution Non Commercial 2.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-2.5" . ((name . "Creative Commons Attribution Non Commercial 2.5") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-3.0" . ((name . "Creative Commons Attribution Non Commercial 3.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-4.0" . ((name . "Creative Commons Attribution Non Commercial 4.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-ND-1.0" . ((name . "Creative Commons Attribution Non Commercial No Derivatives 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-ND-2.0" . ((name . "Creative Commons Attribution Non Commercial No Derivatives 2.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-ND-2.5" . ((name . "Creative Commons Attribution Non Commercial No Derivatives 2.5") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-ND-3.0" . ((name . "Creative Commons Attribution Non Commercial No Derivatives 3.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-ND-4.0" . ((name . "Creative Commons Attribution Non Commercial No Derivatives 4.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-SA-1.0" . ((name . "Creative Commons Attribution Non Commercial Share Alike 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-SA-2.0" . ((name . "Creative Commons Attribution Non Commercial Share Alike 2.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-SA-2.5" . ((name . "Creative Commons Attribution Non Commercial Share Alike 2.5") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-SA-3.0" . ((name . "Creative Commons Attribution Non Commercial Share Alike 3.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-NC-SA-4.0" . ((name . "Creative Commons Attribution Non Commercial Share Alike 4.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-ND-1.0" . ((name . "Creative Commons Attribution No Derivatives 1.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-ND-2.0" . ((name . "Creative Commons Attribution No Derivatives 2.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-ND-2.5" . ((name . "Creative Commons Attribution No Derivatives 2.5") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-ND-3.0" . ((name . "Creative Commons Attribution No Derivatives 3.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-ND-4.0" . ((name . "Creative Commons Attribution No Derivatives 4.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-SA-1.0" . ((name . "Creative Commons Attribution Share Alike 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-SA-2.0" . ((name . "Creative Commons Attribution Share Alike 2.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-SA-2.5" . ((name . "Creative Commons Attribution Share Alike 2.5") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-SA-3.0" . ((name . "Creative Commons Attribution Share Alike 3.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CC-BY-SA-4.0" . ((name . "Creative Commons Attribution Share Alike 4.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("CC0-1.0" . ((name . "Creative Commons Zero v1.0 Universal") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("CDDL-1.0" . ((name . "Common Development and Distribution License 1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("CDDL-1.1" . ((name . "Common Development and Distribution License 1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("CDLA-Permissive-1.0" . ((name . "Community Data License Agreement Permissive 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CDLA-Sharing-1.0" . ((name . "Community Data License Agreement Sharing 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CECILL-1.0" . ((name . "CeCILL Free Software License Agreement v1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("CECILL-1.1" . ((name . "CeCILL Free Software License Agreement v1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("CECILL-2.0" . ((name . "CeCILL Free Software License Agreement v2.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("CECILL-2.1" . ((name . "CeCILL Free Software License Agreement v2.1") (osi-approved? . #t) (deprecated? . #f)))
     ("CECILL-B" . ((name . "CeCILL-B Free Software License Agreement") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("CECILL-C" . ((name . "CeCILL-C Free Software License Agreement") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("ClArtistic" . ((name . "Clarified Artistic License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("CNRI-Jython" . ((name . "CNRI Jython License") (osi-approved? . #f) (deprecated? . #f)))
     ("CNRI-Python" . ((name . "CNRI Python License") (osi-approved? . #t) (deprecated? . #f)))
     ("CNRI-Python-GPL-Compatible" . ((name . "CNRI Python Open Source GPL Compatible License Agreement") (osi-approved? . #f) (deprecated? . #f)))
     ("Condor-1.1" . ((name . "Condor Public License v1.1") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("CPAL-1.0" . ((name . "Common Public Attribution License 1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("CPL-1.0" . ((name . "Common Public License 1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("CPOL-1.02" . ((name . "Code Project Open License 1.02") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("Crossword" . ((name . "Crossword License") (osi-approved? . #f) (deprecated? . #f)))
     ("CrystalStacker" . ((name . "CrystalStacker License") (osi-approved? . #f) (deprecated? . #f)))
     ("CUA-OPL-1.0" . ((name . "CUA Office Public License v1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("Cube" . ((name . "Cube License") (osi-approved? . #f) (deprecated? . #f)))
     ("curl" . ((name . "curl License") (osi-approved? . #f) (deprecated? . #f)))
     ("D-FSL-1.0" . ((name . "Deutsche Freie Software Lizenz") (osi-approved? . #f) (deprecated? . #f)))
     ("diffmark" . ((name . "diffmark license") (osi-approved? . #f) (deprecated? . #f)))
     ("DOC" . ((name . "DOC License") (osi-approved? . #f) (deprecated? . #f)))
     ("Dotseqn" . ((name . "Dotseqn License") (osi-approved? . #f) (deprecated? . #f)))
     ("DSDP" . ((name . "DSDP License") (osi-approved? . #f) (deprecated? . #f)))
     ("dvipdfm" . ((name . "dvipdfm License") (osi-approved? . #f) (deprecated? . #f)))
     ("ECL-1.0" . ((name . "Educational Community License v1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("ECL-2.0" . ((name . "Educational Community License v2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("eCos-2.0" . ((name . "eCos license version 2.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("EFL-1.0" . ((name . "Eiffel Forum License v1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("EFL-2.0" . ((name . "Eiffel Forum License v2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("eGenix" . ((name . "eGenix.com Public License 1.1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("Entessa" . ((name . "Entessa Public License v1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("EPL-1.0" . ((name . "Eclipse Public License 1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("EPL-2.0" . ((name . "Eclipse Public License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("ErlPL-1.1" . ((name . "Erlang Public License v1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("EUDatagrid" . ((name . "EU DataGrid Software License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("EUPL-1.0" . ((name . "European Union Public License 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("EUPL-1.1" . ((name . "European Union Public License 1.1") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("EUPL-1.2" . ((name . "European Union Public License 1.2") (osi-approved? . #t) (deprecated? . #f)))
     ("Eurosym" . ((name . "Eurosym License") (osi-approved? . #f) (deprecated? . #f)))
     ("Fair" . ((name . "Fair License") (osi-approved? . #t) (deprecated? . #f)))
     ("Frameworx-1.0" . ((name . "Frameworx Open License 1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("FreeImage" . ((name . "FreeImage Public License v1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("FSFAP" . ((name . "FSF All Permissive License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("FSFUL" . ((name . "FSF Unlimited License") (osi-approved? . #f) (deprecated? . #f)))
     ("FSFULLR" . ((name . "FSF Unlimited License (with License Retention)") (osi-approved? . #f) (deprecated? . #f)))
     ("FTL" . ((name . "Freetype Project License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("GFDL-1.1" . ((name . "GNU Free Documentation License v1.1") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GFDL-1.1-only" . ((name . "GNU Free Documentation License v1.1 only") (osi-approved? . #f) (deprecated? . #f)))
     ("GFDL-1.1-or-later" . ((name . "GNU Free Documentation License v1.1 or later") (osi-approved? . #f) (deprecated? . #f)))
     ("GFDL-1.2" . ((name . "GNU Free Documentation License v1.2") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GFDL-1.2-only" . ((name . "GNU Free Documentation License v1.2 only") (osi-approved? . #f) (deprecated? . #f)))
     ("GFDL-1.2-or-later" . ((name . "GNU Free Documentation License v1.2 or later") (osi-approved? . #f) (deprecated? . #f)))
     ("GFDL-1.3" . ((name . "GNU Free Documentation License v1.3") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GFDL-1.3-only" . ((name . "GNU Free Documentation License v1.3 only") (osi-approved? . #f) (deprecated? . #f)))
     ("GFDL-1.3-or-later" . ((name . "GNU Free Documentation License v1.3 or later") (osi-approved? . #f) (deprecated? . #f)))
     ("Giftware" . ((name . "Giftware License") (osi-approved? . #f) (deprecated? . #f)))
     ("GL2PS" . ((name . "GL2PS License") (osi-approved? . #f) (deprecated? . #f)))
     ("Glide" . ((name . "3dfx Glide License") (osi-approved? . #f) (deprecated? . #f)))
     ("Glulxe" . ((name . "Glulxe License") (osi-approved? . #f) (deprecated? . #f)))
     ("gnuplot" . ((name . "gnuplot License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("GPL-1.0" . ((name . "GNU General Public License v1.0 only") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-1.0+" . ((name . "GNU General Public License v1.0 or later") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-1.0-only" . ((name . "GNU General Public License v1.0 only") (osi-approved? . #f) (deprecated? . #f)))
     ("GPL-1.0-or-later" . ((name . "GNU General Public License v1.0 or later") (osi-approved? . #f) (deprecated? . #f)))
     ("GPL-2.0" . ((name . "GNU General Public License v2.0 only") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("GPL-2.0+" . ((name . "GNU General Public License v2.0 or later") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("GPL-2.0-only" . ((name . "GNU General Public License v2.0 only") (osi-approved? . #t) (deprecated? . #f)))
     ("GPL-2.0-or-later" . ((name . "GNU General Public License v2.0 or later") (osi-approved? . #t) (deprecated? . #f)))
     ("GPL-2.0-with-autoconf-exception" . ((name . "GNU General Public License v2.0 w/Autoconf exception") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-2.0-with-bison-exception" . ((name . "GNU General Public License v2.0 w/Bison exception") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-2.0-with-classpath-exception" . ((name . "GNU General Public License v2.0 w/Classpath exception") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-2.0-with-font-exception" . ((name . "GNU General Public License v2.0 w/Font exception") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-2.0-with-GCC-exception" . ((name . "GNU General Public License v2.0 w/GCC Runtime Library exception") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-3.0" . ((name . "GNU General Public License v3.0 only") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("GPL-3.0+" . ((name . "GNU General Public License v3.0 or later") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("GPL-3.0-only" . ((name . "GNU General Public License v3.0 only") (osi-approved? . #t) (deprecated? . #f)))
     ("GPL-3.0-or-later" . ((name . "GNU General Public License v3.0 or later") (osi-approved? . #t) (deprecated? . #f)))
     ("GPL-3.0-with-autoconf-exception" . ((name . "GNU General Public License v3.0 w/Autoconf exception") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("GPL-3.0-with-GCC-exception" . ((name . "GNU General Public License v3.0 w/GCC Runtime Library exception") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("gSOAP-1.3b" . ((name . "gSOAP Public License v1.3b") (osi-approved? . #f) (deprecated? . #f)))
     ("HaskellReport" . ((name . "Haskell Language Report License") (osi-approved? . #f) (deprecated? . #f)))
     ("HPND" . ((name . "Historical Permission Notice and Disclaimer") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("IBM-pibs" . ((name . "IBM PowerPC Initialization and Boot Software") (osi-approved? . #f) (deprecated? . #f)))
     ("ICU" . ((name . "ICU License") (osi-approved? . #f) (deprecated? . #f)))
     ("IJG" . ((name . "Independent JPEG Group License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("ImageMagick" . ((name . "ImageMagick License") (osi-approved? . #f) (deprecated? . #f)))
     ("iMatix" . ((name . "iMatix Standard Function Library Agreement") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Imlib2" . ((name . "Imlib2 License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Info-ZIP" . ((name . "Info-ZIP License") (osi-approved? . #f) (deprecated? . #f)))
     ("Intel" . ((name . "Intel Open Source License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Intel-ACPI" . ((name . "Intel ACPI Software License Agreement") (osi-approved? . #f) (deprecated? . #f)))
     ("Interbase-1.0" . ((name . "Interbase Public License v1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("IPA" . ((name . "IPA Font License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("IPL-1.0" . ((name . "IBM Public License v1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("ISC" . ((name . "ISC License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("JasPer-2.0" . ((name . "JasPer License") (osi-approved? . #f) (deprecated? . #f)))
     ("JSON" . ((name . "JSON License") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("LAL-1.2" . ((name . "Licence Art Libre 1.2") (osi-approved? . #f) (deprecated? . #f)))
     ("LAL-1.3" . ((name . "Licence Art Libre 1.3") (osi-approved? . #f) (deprecated? . #f)))
     ("Latex2e" . ((name . "Latex2e License") (osi-approved? . #f) (deprecated? . #f)))
     ("Leptonica" . ((name . "Leptonica License") (osi-approved? . #f) (deprecated? . #f)))
     ("LGPL-2.0" . ((name . "GNU Library General Public License v2 only") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("LGPL-2.0+" . ((name . "GNU Library General Public License v2 or later") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("LGPL-2.0-only" . ((name . "GNU Library General Public License v2 only") (osi-approved? . #t) (deprecated? . #f)))
     ("LGPL-2.0-or-later" . ((name . "GNU Library General Public License v2 or later") (osi-approved? . #t) (deprecated? . #f)))
     ("LGPL-2.1" . ((name . "GNU Lesser General Public License v2.1 only") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("LGPL-2.1+" . ((name . "GNU Library General Public License v2 or later") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("LGPL-2.1-only" . ((name . "GNU Lesser General Public License v2.1 only") (osi-approved? . #t) (deprecated? . #f)))
     ("LGPL-2.1-or-later" . ((name . "GNU Lesser General Public License v2.1 or later") (osi-approved? . #t) (deprecated? . #f)))
     ("LGPL-3.0" . ((name . "GNU Lesser General Public License v3.0 only") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("LGPL-3.0+" . ((name . "GNU Lesser General Public License v3.0 or later") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #t)))
     ("LGPL-3.0-only" . ((name . "GNU Lesser General Public License v3.0 only") (osi-approved? . #t) (deprecated? . #f)))
     ("LGPL-3.0-or-later" . ((name . "GNU Lesser General Public License v3.0 or later") (osi-approved? . #t) (deprecated? . #f)))
     ("LGPLLR" . ((name . "Lesser General Public License For Linguistic Resources") (osi-approved? . #f) (deprecated? . #f)))
     ("Libpng" . ((name . "libpng License") (osi-approved? . #f) (deprecated? . #f)))
     ("libtiff" . ((name . "libtiff License") (osi-approved? . #f) (deprecated? . #f)))
     ("LiLiQ-P-1.1" . ((name . "Licence Libre du Québec – Permissive version 1.1") (osi-approved? . #t) (deprecated? . #f)))
     ("LiLiQ-R-1.1" . ((name . "Licence Libre du Québec – Réciprocité version 1.1") (osi-approved? . #t) (deprecated? . #f)))
     ("LiLiQ-Rplus-1.1" . ((name . "Licence Libre du Québec – Réciprocité forte version 1.1") (osi-approved? . #t) (deprecated? . #f)))
     ("LPL-1.0" . ((name . "Lucent Public License Version 1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("LPL-1.02" . ((name . "Lucent Public License v1.02") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("LPPL-1.0" . ((name . "LaTeX Project Public License v1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("LPPL-1.1" . ((name . "LaTeX Project Public License v1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("LPPL-1.2" . ((name . "LaTeX Project Public License v1.2") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("LPPL-1.3a" . ((name . "LaTeX Project Public License v1.3a") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("LPPL-1.3c" . ((name . "LaTeX Project Public License v1.3c") (osi-approved? . #t) (deprecated? . #f)))
     ("MakeIndex" . ((name . "MakeIndex License") (osi-approved? . #f) (deprecated? . #f)))
     ("MirOS" . ((name . "MirOS License") (osi-approved? . #t) (deprecated? . #f)))
     ("MIT" . ((name . "MIT License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("MIT-advertising" . ((name . "Enlightenment License (e16)") (osi-approved? . #f) (deprecated? . #f)))
     ("MIT-CMU" . ((name . "CMU License") (osi-approved? . #f) (deprecated? . #f)))
     ("MIT-enna" . ((name . "enna License") (osi-approved? . #f) (deprecated? . #f)))
     ("MIT-feh" . ((name . "feh License") (osi-approved? . #f) (deprecated? . #f)))
     ("MITNFA" . ((name . "MIT +no-false-attribs license") (osi-approved? . #f) (deprecated? . #f)))
     ("Motosoto" . ((name . "Motosoto License") (osi-approved? . #t) (deprecated? . #f)))
     ("mpich2" . ((name . "mpich2 License") (osi-approved? . #f) (deprecated? . #f)))
     ("MPL-1.0" . ((name . "Mozilla Public License 1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("MPL-1.1" . ((name . "Mozilla Public License 1.1") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("MPL-2.0" . ((name . "Mozilla Public License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("MPL-2.0-no-copyleft-exception" . ((name . "Mozilla Public License 2.0 (no copyleft exception)") (osi-approved? . #t) (deprecated? . #f)))
     ("MS-PL" . ((name . "Microsoft Public License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("MS-RL" . ((name . "Microsoft Reciprocal License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("MTLL" . ((name . "Matrix Template Library License") (osi-approved? . #f) (deprecated? . #f)))
     ("Multics" . ((name . "Multics License") (osi-approved? . #t) (deprecated? . #f)))
     ("Mup" . ((name . "Mup License") (osi-approved? . #f) (deprecated? . #f)))
     ("NASA-1.3" . ((name . "NASA Open Source Agreement 1.3") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #f)))
     ("Naumen" . ((name . "Naumen Public License") (osi-approved? . #t) (deprecated? . #f)))
     ("NBPL-1.0" . ((name . "Net Boolean Public License v1") (osi-approved? . #f) (deprecated? . #f)))
     ("NCSA" . ((name . "University of Illinois/NCSA Open Source License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Net-SNMP" . ((name . "Net-SNMP License") (osi-approved? . #f) (deprecated? . #f)))
     ("NetCDF" . ((name . "NetCDF license") (osi-approved? . #f) (deprecated? . #f)))
     ("Newsletr" . ((name . "Newsletr License") (osi-approved? . #f) (deprecated? . #f)))
     ("NGPL" . ((name . "Nethack General Public License") (osi-approved? . #t) (deprecated? . #f)))
     ("NLOD-1.0" . ((name . "Norwegian Licence for Open Government Data") (osi-approved? . #f) (deprecated? . #f)))
     ("NLPL" . ((name . "No Limit Public License") (osi-approved? . #f) (deprecated? . #f)))
     ("Nokia" . ((name . "Nokia Open Source License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("NOSL" . ((name . "Netizen Open Source License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Noweb" . ((name . "Noweb License") (osi-approved? . #f) (deprecated? . #f)))
     ("NPL-1.0" . ((name . "Netscape Public License v1.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("NPL-1.1" . ((name . "Netscape Public License v1.1") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("NPOSL-3.0" . ((name . "Non-Profit Open Software License 3.0") (osi-approved? . #t) (deprecated? . #f)))
     ("NRL" . ((name . "NRL License") (osi-approved? . #f) (deprecated? . #f)))
     ("NTP" . ((name . "NTP License") (osi-approved? . #t) (deprecated? . #f)))
     ("Nunit" . ((name . "Nunit License") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("OCCT-PL" . ((name . "Open CASCADE Technology Public License") (osi-approved? . #f) (deprecated? . #f)))
     ("OCLC-2.0" . ((name . "OCLC Research Public License 2.0") (osi-approved? . #t) (deprecated? . #f)))
     ("ODbL-1.0" . ((name . "ODC Open Database License v1.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("OFL-1.0" . ((name . "SIL Open Font License 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("OFL-1.1" . ((name . "SIL Open Font License 1.1") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("OGTSL" . ((name . "Open Group Test Suite License") (osi-approved? . #t) (deprecated? . #f)))
     ("OLDAP-1.1" . ((name . "Open LDAP Public License v1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-1.2" . ((name . "Open LDAP Public License v1.2") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-1.3" . ((name . "Open LDAP Public License v1.3") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-1.4" . ((name . "Open LDAP Public License v1.4") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.0" . ((name . "Open LDAP Public License v2.0 (or possibly 2.0A and 2.0B)") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.0.1" . ((name . "Open LDAP Public License v2.0.1") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.1" . ((name . "Open LDAP Public License v2.1") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.2" . ((name . "Open LDAP Public License v2.2") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.2.1" . ((name . "Open LDAP Public License v2.2.1") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.2.2" . ((name . "Open LDAP Public License 2.2.2") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.3" . ((name . "Open LDAP Public License v2.3") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.4" . ((name . "Open LDAP Public License v2.4") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.5" . ((name . "Open LDAP Public License v2.5") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.6" . ((name . "Open LDAP Public License v2.6") (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.7" . ((name . "Open LDAP Public License v2.7") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("OLDAP-2.8" . ((name . "Open LDAP Public License v2.8") (osi-approved? . #f) (deprecated? . #f)))
     ("OML" . ((name . "Open Market License") (osi-approved? . #f) (deprecated? . #f)))
     ("OpenSSL" . ((name . "OpenSSL License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("OPL-1.0" . ((name . "Open Public License v1.0") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("OSET-PL-2.1" . ((name . "OSET Public License version 2.1") (osi-approved? . #t) (deprecated? . #f)))
     ("OSL-1.0" . ((name . "Open Software License 1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("OSL-1.1" . ((name . "Open Software License 1.1") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("OSL-2.0" . ((name . "Open Software License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("OSL-2.1" . ((name . "Open Software License 2.1") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("OSL-3.0" . ((name . "Open Software License 3.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("PDDL-1.0" . ((name . "ODC Public Domain Dedication & License 1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("PHP-3.0" . ((name . "PHP License v3.0") (osi-approved? . #t) (deprecated? . #f)))
     ("PHP-3.01" . ((name . "PHP License v3.01") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Plexus" . ((name . "Plexus Classworlds License") (osi-approved? . #f) (deprecated? . #f)))
     ("PostgreSQL" . ((name . "PostgreSQL License") (osi-approved? . #t) (deprecated? . #f)))
     ("psfrag" . ((name . "psfrag License") (osi-approved? . #f) (deprecated? . #f)))
     ("psutils" . ((name . "psutils License") (osi-approved? . #f) (deprecated? . #f)))
     ("Python-2.0" . ((name . "Python License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Qhull" . ((name . "Qhull License") (osi-approved? . #f) (deprecated? . #f)))
     ("QPL-1.0" . ((name . "Q Public License 1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Rdisc" . ((name . "Rdisc License") (osi-approved? . #f) (deprecated? . #f)))
     ("RHeCos-1.1" . ((name . "Red Hat eCos Public License v1.1") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #f)))
     ("RPL-1.1" . ((name . "Reciprocal Public License 1.1") (osi-approved? . #t) (deprecated? . #f)))
     ("RPL-1.5" . ((name . "Reciprocal Public License 1.5") (osi-approved? . #t) (deprecated? . #f)))
     ("RPSL-1.0" . ((name . "RealNetworks Public Source License v1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("RSA-MD" . ((name . "RSA Message-Digest License ") (osi-approved? . #f) (deprecated? . #f)))
     ("RSCPL" . ((name . "Ricoh Source Code Public License") (osi-approved? . #t) (deprecated? . #f)))
     ("Ruby" . ((name . "Ruby License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("SAX-PD" . ((name . "Sax Public Domain Notice") (osi-approved? . #f) (deprecated? . #f)))
     ("Saxpath" . ((name . "Saxpath License") (osi-approved? . #f) (deprecated? . #f)))
     ("SCEA" . ((name . "SCEA Shared Source License") (osi-approved? . #f) (deprecated? . #f)))
     ("Sendmail" . ((name . "Sendmail License") (osi-approved? . #f) (deprecated? . #f)))
     ("SGI-B-1.0" . ((name . "SGI Free Software License B v1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("SGI-B-1.1" . ((name . "SGI Free Software License B v1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("SGI-B-2.0" . ((name . "SGI Free Software License B v2.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("SimPL-2.0" . ((name . "Simple Public License 2.0") (osi-approved? . #t) (deprecated? . #f)))
     ("SISSL" . ((name . "Sun Industry Standards Source License v1.1") (osi-approved? . #t) (deprecated? . #f)))
     ("SISSL-1.2" . ((name . "Sun Industry Standards Source License v1.2") (osi-approved? . #f) (deprecated? . #f)))
     ("Sleepycat" . ((name . "Sleepycat License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("SMLNJ" . ((name . "Standard ML of New Jersey License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("SMPPL" . ((name . "Secure Messaging Protocol Public License") (osi-approved? . #f) (deprecated? . #f)))
     ("SNIA" . ((name . "SNIA Public License 1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("Spencer-86" . ((name . "Spencer License 86") (osi-approved? . #f) (deprecated? . #f)))
     ("Spencer-94" . ((name . "Spencer License 94") (osi-approved? . #f) (deprecated? . #f)))
     ("Spencer-99" . ((name . "Spencer License 99") (osi-approved? . #f) (deprecated? . #f)))
     ("SPL-1.0" . ((name . "Sun Public License v1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("StandardML-NJ" . ((name . "Standard ML of New Jersey License") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("SugarCRM-1.1.3" . ((name . "SugarCRM Public License v1.1.3") (osi-approved? . #f) (deprecated? . #f)))
     ("SWL" . ((name . "Scheme Widget Library (SWL) Software License Agreement") (osi-approved? . #f) (deprecated? . #f)))
     ("TCL" . ((name . "TCL/TK License") (osi-approved? . #f) (deprecated? . #f)))
     ("TCP-wrappers" . ((name . "TCP Wrappers License") (osi-approved? . #f) (deprecated? . #f)))
     ("TMate" . ((name . "TMate Open Source License") (osi-approved? . #f) (deprecated? . #f)))
     ("TORQUE-1.1" . ((name . "TORQUE v2.5+ Software License v1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("TOSL" . ((name . "Trusster Open Source License") (osi-approved? . #f) (deprecated? . #f)))
     ("Unicode-DFS-2015" . ((name . "Unicode License Agreement - Data Files and Software (2015)") (osi-approved? . #f) (deprecated? . #f)))
     ("Unicode-DFS-2016" . ((name . "Unicode License Agreement - Data Files and Software (2016)") (osi-approved? . #f) (deprecated? . #f)))
     ("Unicode-TOU" . ((name . "Unicode Terms of Use") (osi-approved? . #f) (deprecated? . #f)))
     ("Unlicense" . ((name . "The Unlicense") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("UPL-1.0" . ((name . "Universal Permissive License v1.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("Vim" . ((name . "Vim License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("VOSTROM" . ((name . "VOSTROM Public License for Open Source") (osi-approved? . #f) (deprecated? . #f)))
     ("VSL-1.0" . ((name . "Vovida Software License v1.0") (osi-approved? . #t) (deprecated? . #f)))
     ("W3C" . ((name . "W3C Software Notice and License (2002-12-31)") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("W3C-19980720" . ((name . "W3C Software Notice and License (1998-07-20)") (osi-approved? . #f) (deprecated? . #f)))
     ("W3C-20150513" . ((name . "W3C Software Notice and Document License (2015-05-13)") (osi-approved? . #f) (deprecated? . #f)))
     ("Watcom-1.0" . ((name . "Sybase Open Watcom Public License 1.0") (fsf-libre? . #f) (osi-approved? . #t) (deprecated? . #f)))
     ("Wsuipa" . ((name . "Wsuipa License") (osi-approved? . #f) (deprecated? . #f)))
     ("WTFPL" . ((name . "Do What The F*ck You Want To Public License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("wxWindows" . ((name . "wxWindows Library License") (fsf-libre? . #f) (osi-approved? . #f) (deprecated? . #t)))
     ("X11" . ((name . "X11 License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Xerox" . ((name . "Xerox License") (osi-approved? . #f) (deprecated? . #f)))
     ("XFree86-1.1" . ((name . "XFree86 License 1.1") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("xinetd" . ((name . "xinetd License") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Xnet" . ((name . "X.Net License") (osi-approved? . #t) (deprecated? . #f)))
     ("xpp" . ((name . "XPP License") (osi-approved? . #f) (deprecated? . #f)))
     ("XSkat" . ((name . "XSkat License") (osi-approved? . #f) (deprecated? . #f)))
     ("YPL-1.0" . ((name . "Yahoo! Public License v1.0") (osi-approved? . #f) (deprecated? . #f)))
     ("YPL-1.1" . ((name . "Yahoo! Public License v1.1") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Zed" . ((name . "Zed License") (osi-approved? . #f) (deprecated? . #f)))
     ("Zend-2.0" . ((name . "Zend License v2.0") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Zimbra-1.3" . ((name . "Zimbra Public License v1.3") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ("Zimbra-1.4" . ((name . "Zimbra Public License v1.4") (osi-approved? . #f) (deprecated? . #f)))
     ("Zlib" . ((name . "zlib License") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("zlib-acknowledgement" . ((name . "zlib/libpng License with Acknowledgement") (osi-approved? . #f) (deprecated? . #f)))
     ("ZPL-1.1" . ((name . "Zope Public License 1.1") (osi-approved? . #f) (deprecated? . #f)))
     ("ZPL-2.0" . ((name . "Zope Public License 2.0") (fsf-libre? . #t) (osi-approved? . #t) (deprecated? . #f)))
     ("ZPL-2.1" . ((name . "Zope Public License 2.1") (fsf-libre? . #t) (osi-approved? . #f) (deprecated? . #f)))
     ))
)
