---
myst:
  html_meta:
    keywords: LaTeX, programmation, commandes, macros, étoile, commande étoilée, macro avec étoile, \@ifstar, LaTeX3
---
# Comment définir une commande ou un environnement étoilé ?

Vous l'avez sans doute remarqué, beaucoup d'environnements ou commandes standard de LaTeX existent sous une variante étoilée : par exemple, `\newcommand` et `\newcommand*`. Vous pouvez avoir envie de reprendre cette idée quand vous définissez vos commandes et arguments. Voyons ici comment la mettre en oeuvre.

## Cas des commandes

Ce cas pose des difficultés car l'étoile ne peut pas faire partie du nom de la commande. Il faut donc créer une commande qui devra être capable de regarder si elle est suivie ou pas d'une étoile et d'adapter son comportement en conséquence. Pour des raisons techniques, cette commande ne pourra pas accepter d'argument, mais pourra faire appel à des commandes qui en acceptent (voir le début de "[](/2_programmation/macros/macros_a_plusieurs_arguments_optionnels.md)").

### Avec l'extension <ctanpkg:ifthen>

La façon la plus simple pour un utilisateur d'écrire une telle commande consiste à utiliser l'extension <ctanpkg:ifthen> :

```{noedit}
\newcommand{\maCommande}[1]{\ifthenelse{\equal{#1}{*}}%
  {\maCommandeAvecEtoile}%
  {\maCommandeSansEtoile{#1}}%
}
\newcommand{\maCommandeAvecEtoile}{version étoilée}
\newcommand{\maCommandeSansEtoile}[1]{version normale}
```

Cela fait l'affaire mais uniquement pour des commandes suffisamment simples (la gestion des erreurs pouvant être compliquée et la méthode implique que la commande `\maCommandeSansEtoile` doit avoir au moins un argument).

### Avec la commande `\@ifstar`

La commande interne de LaTeX `\@ifstar` permet d'appeler une commande ou sa version étoilée en fonction du contexte. Elle figure au début de la définition de la commande et est suivie immédiatement par deux noms de commande : celle implémentant la version étoilée, puis celle implémentant la version non étoilée. 

Voici un exemple illustrant cette utilisation :

```
\documentclass{article}
  \usepackage{lmodern}   % Caractères plus lisibles
  \pagestyle{empty}      % N'affiche pas de numéro de page

  \newcommand{\maCommandeAvecEtoile}[1]{J'ai une étoile et le paramètre #1.}
  \newcommand{\maCommandeSansEtoile}[1]{Je n'ai pas d'étoile et le paramètre #1.}
  \makeatletter
  \newcommand{\maCommande}{\@ifstar\maCommandeAvecEtoile\maCommandeSansEtoile}
  \makeatother

\begin{document}
\maCommande{42}
\maCommande*{7*9}
\end{document}
```

Notez que : 
- les arguments de `\maCommandeAvecEtoile` et `\maCommandeSansEtoile` sont indépendants : chaque commande peut avoir ses propres arguments, sans être contraint par la technique que nous utilisons, contrairement à l'astuce décrite ci-dessus ;
- la commande `\@ifstar` est efficace mais elle nécessite que la définition soit encadrée par les commandes `\makeatletter` et `\makeatletter`, comme le détaille la question "[](makeatletter_et_makeatother.md)", à moins de les placer dans un fichier `sty` définissant une extension.


### Avec l'extension <ctanpkg:suffix> 

Une alternative intéressante est l'extension <ctanpkg:suffix> (qui nécessite [eTeX](/1_generalites/glossaire/etex.md), présent par défaut de nos jours). Elle vous permet de définir des variantes de vos commandes de façon assez élégante :

```{noedit}
\newcommand\maCommande{version normale}
\WithSuffix\newcommand\maCommande*{version étoilée}
```

Les arguments des commandes peuvent être spécifiés de manière normale, dans les deux définitions de commandes (après le `*` dans la version `\WithSuffix`). Vous pouvez également utiliser les commandes primitives de TeX, en créant une définition comme :

```{noedit}
\WithSuffix\gdef\maCommande*{version étoilée}
```

### Avec l'extension <ctanpkg:xparse>

Une autre option consiste à utiliser l'extension <ctanpkg:xparse> de l'ensemble <ctanpkg:l3packages>. Cette extension définit un ensemble de commandes (telles que `\NewDocumentCommand`) assez similaires `\newcommand` et autres. La grande différence réside dans la spécification des arguments de la commande. Ainsi, pour chaque argument, vous devez choisir parmi différents spécificateurs. Ainsi, pour créer une commande étoilée (dans le style LaTeX2e), on peut écrire :

```{noedit}
\NewDocumentCommand \maCommande { s m } {%
  % #1 est le spécificateur de l'étoile
  % #2 est l'argument obligatoire
  ...
}
```

Le spécificateur de l'étoile,`s` (pour *star*), apparaît comme `#1` et prend les valeurs `\BooleanTrue` (s'il y a une étoile) ou `\BooleanFalse` (sinon). L'autre spécificateur,`m` (pour *mandatory*), signale un argument obligatoire normal de style TeX et apparaît comme `#2`.

Bien que <ctanpkg:xparse> fournisse des spécifications agréables pour les arguments de commande, il *fait* partie du [LaTeX3](/1_generalites/histoire/c_est_quoi_latex3.md). Le charger pour obtenir `\DeclareDocumentCommand` conduit à charger tout le noyau LaTeX3 (un grand nombre d'extensions) via l'extension <ctanpkg:expl3>.


## Cas des environnements

Si vous avez lu la question "[](bien_nommer_ses_macros_et_environnements.md)", vous saurez déjà que c'est immédiat pour les environnements car l'étoile est un caractère autorisé dans les noms d'environnements. Il vous suffit donc de faire `\newenvironment{myenv}` et `\newenvironment{myenv*}` avec les définitions souhaitées.

:::{sources}
<https://www.elzevir.fr/imj/latex/tips.html#stared>
[Commands defined with \* options](faquk:FAQ-cmdstar)
:::
