/**
 * @file libadacl.c
 *
 * @brief The adacl (apply default acl) shared library.
 *
 */

/* Enables get_current_dir_name() in unistd.h, the O_PATH flag, and
 * the asprintf() function.
*/
#define _GNU_SOURCE

#include <dirent.h>     /* readdir(), etc. */
#include <errno.h>      /* EINVAL, ELOOP, ENOTDIR, etc. */
#include <fcntl.h>      /* openat() */
#include <libgen.h>     /* basename(), dirname() */
#include <limits.h>     /* PATH_MAX */
#include <stdbool.h>    /* the "bool" type */
#include <stdio.h>      /* perror(), asprintf() */
#include <stdlib.h>     /* free() */
#include <string.h>     /* strdup() */
#include <sys/stat.h>   /* fstat() */
#include <sys/xattr.h>  /* fgetxattr(), fsetxattr() */
#include <unistd.h>     /* get_current_dir_name() */

/* ACLs */
#include <acl/libacl.h> /* acl_get_perm, not portable */
#include <sys/acl.h>    /* all other acl_foo functions */

/* XATTR_NAME_POSIX_ACL_ACCESS and XATTR_NAME_POSIX_ACL_DEFAULT */
#include <linux/xattr.h>

#include "libadacl.h"


/* Even though most other library functions reliably return -1 for
 * error, it feels a little wrong to re-use the ACL_ERROR constant.
 */
#define CLOSE_ERROR -1
#define OPEN_ERROR -1
#define ASPRINTF_ERROR -1
#define STAT_ERROR -1
#define XATTR_ERROR -1


/* Prototypes */
int safe_open_ex(int at_fd, char* pathname, int flags);
int safe_open(const char* pathname, int flags);
int acl_update_entry(acl_t aclp, acl_entry_t updated_entry);
int acl_entry_count(acl_t acl);
int acl_is_minimal(acl_t acl);
int acl_execute_masked(acl_t acl);
int any_can_execute(int fd, const struct stat* sp);
int acl_copy_xattr(int src_fd,
                   acl_type_t src_type,
                   int dst_fd,
                   acl_type_t dst_type);
int has_default_acl_fd(int fd);
int apply_default_acl_fds(int parent_fd, int fd, bool recursive);
int apply_default_acl(const char* path, bool recursive);



/**
 * @brief The recursive portion of the @c safe_open function, used to
 *   open a file descriptor in a symlink-safe way when combined with
 *   the @c O_NOFOLLOW flag.
 *
 * @param at_fd
 *   A file descriptor relative to which @c pathname will be opened.
 *
 * @param pathname
 *   The path to the file/directory/whatever whose descriptor you want.
 *
 * @param flags
 *   File status flags to be passed to @c openat.
 *
 * @return a file descriptor for @c pathname if everything goes well,
 *   and @c OPEN_ERROR if not.
 */
int safe_open_ex(int at_fd, char* pathname, int flags) {
  if (pathname == NULL) {
    errno = EINVAL;
    perror("safe_open_ex (args)");
    return OPEN_ERROR;
  }

  char* firstslash = strchr(pathname, '/');
  if (firstslash == NULL) {
    /* No more slashes, this is the base case. */
    return openat(at_fd, pathname, flags);
  }
  if (firstslash[1] == '\0') {
    /* The first slash is the last character; ensure that we open
       a directory. */
    firstslash[0] = '\0';
    return openat(at_fd, pathname, flags | O_DIRECTORY);
  }

  /* The first slash exists and isn't the last character in the path,
     so we can split the path wherever that first slash lies and
     recurse. */
   *firstslash = '\0';
   int fd = openat(at_fd, pathname, flags | O_DIRECTORY | O_PATH);
   if (fd == OPEN_ERROR) {
     if (errno != ENOTDIR) {
       /* Don't output anything if we ignore a symlink */
       perror("safe_open_ex (safe_open_ex)");
     }
     return OPEN_ERROR;
   }

   /* The +1 is safe because there needs to be at least one character
      after the first slash (we checked this above). */
   int result = safe_open_ex(fd, firstslash+1, flags);
   if (close(fd) == CLOSE_ERROR) {
      perror("safe_open_ex (close)");
      return OPEN_ERROR;
    }
   return result;
}


/**
 * @brief A version of @c open that is completely symlink-safe when
 *   used with the @c O_NOFOLLOW flag.
 *
 * The @c openat function exists to ensure that you can anchor one
 * path to a particular directory while opening it; however, if you
 * open "b/c/d" relative to "/a", then even the @c openat function will
 * still follow symlinks in the "b" component. This can be exploited
 * by an attacker to make you open the wrong path.
 *
 * To avoid that problem, this function uses a recursive
 * implementation that opens every path from the root, one level at a
 * time. So "a" is opened relative to "/", and then "b" is opened
 * relative to "/a", and then "c" is opened relative to "/a/b",
 * etc. When the @c O_NOFOLLOW flag is used, this approach ensures
 * that no symlinks in any component are followed.
 *
 * @param pathname
 *   The path to the file/directory/whatever whose descriptor you want.
 *
 * @param flags
 *   File status flags to be passed to @c openat.
 *
 * @return a file descriptor for @c pathname if everything goes well,
 *   and @c OPEN_ERROR if not.
 */
int safe_open(const char* pathname, int flags) {
  if (pathname == NULL) {
    errno = EINVAL;
    perror("safe_open (args)");
    return OPEN_ERROR;
  }

  char* abspath = NULL;
  int asprintf_result = 0;
  if (strchr(pathname, '/') == pathname) {
    /* pathname is already absolute; just copy it. */
    asprintf_result = asprintf(&abspath, "%s", pathname);
  }
  else {
    /* Concatenate the current working directory and pathname into an
     * absolute path. We use realpath() ONLY on the cwd part, and not
     * on the pathname part, because realpath() resolves symlinks. And
     * the whole point of all this crap is to avoid following symlinks
     * in the pathname.
     *
     * Using realpath() on the cwd lets us operate on relative paths
     * while we're sitting in a directory that happens to have a
     * symlink in it; for example: cd /var/run && apply-default-acl foo.
     */
    char* cwd = get_current_dir_name();
    if (cwd == NULL) {
      perror("safe_open (get_current_dir_name)");
      return OPEN_ERROR;
    }

    char abs_cwd[PATH_MAX];
    if (realpath(cwd, abs_cwd) == NULL) {
      perror("safe_open (realpath)");
      free(cwd);
      return OPEN_ERROR;
    }
    asprintf_result = asprintf(&abspath, "%s/%s", abs_cwd, pathname);
    free(cwd);
  }
  if (asprintf_result == ASPRINTF_ERROR) {
    perror("safe_open (asprintf)");
    return OPEN_ERROR;
  }

  /* Beyond here, asprintf() worked, and we need to free abspath. */
  int result = OPEN_ERROR;

  bool abspath_is_root = (strcmp(abspath, "/") == 0);
  int rootflags = flags | O_DIRECTORY;
  if (!abspath_is_root) {
    /* Use O_PATH for some added safety if "/" is not our target */
    rootflags |= O_PATH;
  }
  int rootfd = open("/", rootflags);
  if (rootfd == OPEN_ERROR) {
    perror("safe_open (open)");
    result = OPEN_ERROR;
    goto cleanup;
  }

  if (abspath_is_root) {
    result = rootfd;
    goto cleanup;
  }

  result = safe_open_ex(rootfd, abspath+1, flags);
  if (close(rootfd) == CLOSE_ERROR) {
    perror("safe_open (close)");
    result = OPEN_ERROR;
    goto cleanup;
  }

 cleanup:
  free(abspath);
  return result;
}




/**
 * @brief Update an entry in an @b minimal ACL.
 *
 * @param aclp
 *   A pointer to the acl_t structure whose entry we want to update.
 *
 * @param updated_entry
 *   An updated copy of an existing entry in @c aclp.
 *
 * @return
 *   - @c ACL_SUCCESS - If we update an existing entry.
 *   - @c ACL_FAILURE - If we don't find an entry to update.
 *   - @c ACL_ERROR - Unexpected library error.
 */
int acl_update_entry(acl_t aclp, acl_entry_t updated_entry) {
  if (aclp == NULL || updated_entry == NULL) {
    errno = EINVAL;
    perror("acl_update_entry (args)");
    return ACL_ERROR;
  }

  acl_tag_t updated_tag;
  if (acl_get_tag_type(updated_entry, &updated_tag) == ACL_ERROR) {
    perror("acl_update_entry (acl_get_tag_type)");
    return ACL_ERROR;
  }

  acl_permset_t updated_permset;
  if (acl_get_permset(updated_entry, &updated_permset) == ACL_ERROR) {
    perror("acl_update_entry (acl_get_permset)");
    return ACL_ERROR;
  }

  /* This can allocate memory, so from here on out we have to jump to
     the "cleanup" label to exit. */
  void* updated_qualifier = acl_get_qualifier(updated_entry);
  if (updated_qualifier == NULL &&
      (updated_tag == ACL_USER || updated_tag == ACL_GROUP)) {
    /* acl_get_qualifier() can return NULL, but it shouldn't for
       ACL_USER or ACL_GROUP entries. */
    perror("acl_update_entry (acl_get_qualifier)");
    return ACL_ERROR;
  }

  /* Our return value. Default to failure, and change to success if we
     actually update something. */
  int result = ACL_FAILURE;

  acl_entry_t existing_entry;
  /* Loop through the given ACL looking for matching entries. */
  int get_entry_result = acl_get_entry(aclp, ACL_FIRST_ENTRY, &existing_entry);

  while (get_entry_result == ACL_SUCCESS) {
    acl_tag_t existing_tag = ACL_UNDEFINED_TAG;

    if (acl_get_tag_type(existing_entry, &existing_tag) == ACL_ERROR) {
       perror("set_acl_tag_permset (acl_get_tag_type)");
       result = ACL_ERROR;
       goto cleanup;
    }

    if (existing_tag == updated_tag) {
      /* Our tag types match, but if we have a named user or group
         entry, then we need to check that the user/group (that is,
         the qualifier) matches too. */
      bool qualifiers_match = false;

      /* There are three ways the qualifiers can match... */
      void* existing_qualifier = acl_get_qualifier(existing_entry);
      if (existing_qualifier == NULL) {
        if (existing_tag == ACL_USER || existing_tag == ACL_GROUP) {
          perror("acl_update_entry (acl_get_qualifier)");
          result = ACL_ERROR;
          goto cleanup;
        }
        else {
          /* First, we could be dealing with an entry that isn't a
             named user or group, in which case they "match
             vacuously." */
          qualifiers_match = true;
        }
      }

      /* Second, they could have matching UIDs. We don't really need to
         check both tags here, since we know that they're equal. However,
         clang-tidy can't figure that out, and the redundant equality
         check prevents it from complaining about a potential null pointer
         dereference. */
      if (updated_tag == ACL_USER && existing_tag == ACL_USER) {
        qualifiers_match = ( *((uid_t*)existing_qualifier)
                             ==
                             *((uid_t*)updated_qualifier) );
      }

      /* Third, they could have matching GIDs. See above for why
         we check the redundant condition existing_tag == ACL_GROUP. */
      if (updated_tag == ACL_GROUP && existing_tag == ACL_GROUP) {
        qualifiers_match = ( *((gid_t*)existing_qualifier)
                             ==
                             *((gid_t*)updated_qualifier) );
      }

      /* Be sure to free this inside the loop, where memory is allocated. */
      acl_free(existing_qualifier);

      if (qualifiers_match) {
        /* If we update something, we're done and return ACL_SUCCESS */
        if (acl_set_permset(existing_entry, updated_permset) == ACL_ERROR) {
          perror("acl_update_entry (acl_set_permset)");
          result = ACL_ERROR;
          goto cleanup;
        }

        result = ACL_SUCCESS;
        goto cleanup;
      }
    }

    get_entry_result = acl_get_entry(aclp, ACL_NEXT_ENTRY, &existing_entry);
  }

  /* This catches both the initial acl_get_entry and the ones at the
     end of the loop. */
  if (get_entry_result == ACL_ERROR) {
    perror("acl_update_entry (acl_get_entry)");
    result = ACL_ERROR;
  }

 cleanup:
  acl_free(updated_qualifier);
  return result;
}



/**
 * @brief Determine the number of entries in the given ACL.
 *
 * @param acl
 *   The ACL to inspect.
 *
 * @return Either the non-negative number of entries in @c acl, or
 *   @c ACL_ERROR on error.
 */
int acl_entry_count(acl_t acl) {

  acl_entry_t entry;
  int entry_count = 0;
  int result = acl_get_entry(acl, ACL_FIRST_ENTRY, &entry);

  while (result == ACL_SUCCESS) {
    entry_count++;
    result = acl_get_entry(acl, ACL_NEXT_ENTRY, &entry);
  }

  if (result == ACL_ERROR) {
    perror("acl_entry_count (acl_get_entry)");
    return ACL_ERROR;
  }

  return entry_count;
}



/**
 * @brief Determine whether or not the given ACL is minimal.
 *
 * An ACL is minimal if it has fewer than four entries.
 *
 * @param acl
 *   The ACL whose minimality is in question.
 *
 * @return
 *   - @c ACL_SUCCESS - @c acl is minimal
 *   - @c ACL_FAILURE - @c acl is not minimal
 *   - @c ACL_ERROR - Unexpected library error
 */
int acl_is_minimal(acl_t acl) {
  if (acl == NULL) {
    errno = EINVAL;
    perror("acl_is_minimal (args)");
    return ACL_ERROR;
  }

  int ec = acl_entry_count(acl);

  if (ec == ACL_ERROR) {
    perror("acl_is_minimal (acl_entry_count)");
    return ACL_ERROR;
  }

  if (ec < 4) {
    return ACL_SUCCESS;
  }
  else {
    return ACL_FAILURE;
  }
}



/**
 * @brief Determine whether the given ACL's mask denies execute.
 *
 * @param acl
 *   The ACL whose mask we want to check.
 *
 * @return
 *   - @c ACL_SUCCESS - The @c acl has a mask which denies execute.
 *   - @c ACL_FAILURE - The @c acl has a mask which does not deny execute.
 *   - @c ACL_ERROR - Unexpected library error.
 */
int acl_execute_masked(acl_t acl) {
  if (acl == NULL) {
    errno = EINVAL;
    perror("acl_execute_masked (args)");
    return ACL_ERROR;
  }

  acl_entry_t entry;
  int ge_result = acl_get_entry(acl, ACL_FIRST_ENTRY, &entry);

  while (ge_result == ACL_SUCCESS) {
    acl_tag_t tag = ACL_UNDEFINED_TAG;

    if (acl_get_tag_type(entry, &tag) == ACL_ERROR) {
       perror("acl_execute_masked (acl_get_tag_type)");
       return ACL_ERROR;
    }

    if (tag == ACL_MASK) {
      /* This is the mask entry, get its permissions, and see if
         execute is specified. */
      acl_permset_t permset;

      if (acl_get_permset(entry, &permset) == ACL_ERROR) {
        perror("acl_execute_masked (acl_get_permset)");
        return ACL_ERROR;
      }

      int gp_result = acl_get_perm(permset, ACL_EXECUTE);
      if (gp_result == ACL_ERROR) {
        perror("acl_execute_masked (acl_get_perm)");
        return ACL_ERROR;
      }

      if (gp_result == ACL_FAILURE) {
        /* No execute bit set in the mask; execute not allowed. */
        return ACL_SUCCESS;
      }
    }

    ge_result = acl_get_entry(acl, ACL_NEXT_ENTRY, &entry);
  }

  return ACL_FAILURE;
}



/**
 * @brief Determine whether @c fd is executable by anyone.
 *
 *
 * This is used as part of the heuristic to determine whether or not
 * we should mask the execute bit when inheriting an ACL. If @c fd
 * describes a file, we check the @a effective permissions, contrary
 * to what setfacl does.
 *
 * @param fd
 *   The file descriptor to check.
 *
 * @param sp
 *   A pointer to a stat structure for @c fd.
 *
 * @return
 *   - @c ACL_SUCCESS - Someone has effective execute permissions on @c fd.
 *   - @c ACL_FAILURE - Nobody can execute @c fd.
 *   - @c ACL_ERROR - Unexpected library error.
 */
int any_can_execute(int fd, const struct stat* sp) {
  if (sp == NULL) {
    errno = EINVAL;
    perror("any_can_execute (args)");
    return ACL_ERROR;
  }

  acl_t acl = acl_get_fd(fd);

  if (acl == (acl_t)NULL) {
    perror("any_can_execute (acl_get_fd)");
    return ACL_ERROR;
  }

  /* Our return value. */
  int result = ACL_FAILURE;

  if (acl_is_minimal(acl)) {
    if (sp->st_mode & (S_IXUSR | S_IXOTH | S_IXGRP)) {
      result = ACL_SUCCESS;
      goto cleanup;
    }
    else {
      result = ACL_FAILURE;
      goto cleanup;
    }
  }

  acl_entry_t entry;
  int ge_result = acl_get_entry(acl, ACL_FIRST_ENTRY, &entry);

  while (ge_result == ACL_SUCCESS) {
    /* The first thing we do is check to see if this is a mask
       entry. If it is, we skip it entirely. */
    acl_tag_t tag = ACL_UNDEFINED_TAG;

    if (acl_get_tag_type(entry, &tag) == ACL_ERROR) {
      perror("any_can_execute_or (acl_get_tag_type)");
      result = ACL_ERROR;
      goto cleanup;
    }

    if (tag == ACL_MASK) {
      ge_result = acl_get_entry(acl, ACL_NEXT_ENTRY, &entry);
      continue;
    }

    /* Ok, so it's not a mask entry. Check the execute perms. */
    acl_permset_t permset;

    if (acl_get_permset(entry, &permset) == ACL_ERROR) {
      perror("any_can_execute_or (acl_get_permset)");
      result = ACL_ERROR;
      goto cleanup;
    }

    int gp_result = acl_get_perm(permset, ACL_EXECUTE);
    if (gp_result == ACL_ERROR) {
      perror("any_can_execute (acl_get_perm)");
      result = ACL_ERROR;
      goto cleanup;
    }

    if (gp_result == ACL_SUCCESS) {
      /* Only return ACL_SUCCESS if this execute bit is not masked. */
      if (acl_execute_masked(acl) != ACL_SUCCESS) {
        result = ACL_SUCCESS;
        goto cleanup;
      }
    }

    ge_result = acl_get_entry(acl, ACL_NEXT_ENTRY, &entry);
  }

  if (ge_result == ACL_ERROR) {
    perror("any_can_execute (acl_get_entry)");
    result = ACL_ERROR;
    goto cleanup;
  }

 cleanup:
  acl_free(acl);
  return result;
}



/**
 * @brief Copy ACLs between file descriptors as xattrs, verbatim.
 *
 * There is a small deficiency in libacl, namely that there is no way
 * to get or set default ACLs through file descriptors. The @c
 * acl_get_file and @c acl_set_file functions can do it, but they use
 * paths, and are vulnerable to symlink attacks.
 *
 * Fortunately, when inheriting an ACL, we don't really need to look
 * at what it contains. That means that we can copy the on-disk xattrs
 * from the source directory to the destination file/directory without
 * passing through libacl, and this can be done with file descriptors
 * through @c fgetxattr and @c fsetxattr. That's what this function
 * does.
 *
 * @param src_fd
 *   The file descriptor from which the ACL will be copied.
 *
 * @param src_type
 *   The type of ACL (either @c ACL_TYPE_ACCESS or @c ACL_TYPE_DEFAULT)
 *   to copy from @c src_fd.
 *
 * @param dst_fd
 *   The file descriptor whose ACL will be overwritten with the one
 *   from @c src_fd.
 *
 * @param dst_type
 *   The type of ACL (either @c ACL_TYPE_ACCESS or @c ACL_TYPE_DEFAULT)
 *   to replace on @c dst_fd.
 *
 * @return
 *   - @c ACL_SUCCESS - The ACL was copied successfully.
 *   - @c ACL_FAILURE - There was no ACL on @c src_fd.
 *   - @c ACL_ERROR - Unexpected library error.
 */
int acl_copy_xattr(int src_fd,
                   acl_type_t src_type,
                   int dst_fd,
                   acl_type_t dst_type) {

  const char* src_name;
  if (src_type == ACL_TYPE_ACCESS) {
    src_name = XATTR_NAME_POSIX_ACL_ACCESS;
  }
  else if (src_type == ACL_TYPE_DEFAULT) {
    src_name = XATTR_NAME_POSIX_ACL_DEFAULT;
  }
  else {
    errno = EINVAL;
    perror("acl_copy_xattr (src type)");
    return ACL_ERROR;
  }

  const char* dst_name;
  if (dst_type == ACL_TYPE_ACCESS) {
    dst_name = XATTR_NAME_POSIX_ACL_ACCESS;
  }
  else if (dst_type == ACL_TYPE_DEFAULT) {
    dst_name = XATTR_NAME_POSIX_ACL_DEFAULT;
  }
  else {
    errno = EINVAL;
    perror("acl_copy_xattr (dst type)");
    return ACL_ERROR;
  }

  ssize_t src_size_guess = fgetxattr(src_fd, src_name, NULL, 0);
  if (src_size_guess == XATTR_ERROR) {
    if (errno == ENODATA) {
      /* A missing ACL isn't really an error. ENOATTR and ENODATA are
         synonyms, but using ENODATA here lets us avoid another
         "include" directive. */
      return ACL_FAILURE;
    }
    perror("acl_copy_xattr (fgetxattr size guess)");
    return ACL_ERROR;
  }
  char* src_acl_p = alloca(src_size_guess);
  /* The actual size may be smaller than our guess? I don't know. The
     return value from fgetxattr() will either be nonnegative, or
     XATTR_ERROR (which we've already ruled out), so it's safe to cast
     it to an unsigned size_t here to avoid a compiler warning. */
  ssize_t src_size = fgetxattr(src_fd,
                               src_name,
                               src_acl_p,
                               (size_t)src_size_guess);
  if (src_size == XATTR_ERROR) {
    if (errno == ENODATA) {
      /* A missing ACL isn't an error. */
      return ACL_FAILURE;
    }
    perror("acl_copy_xattr (fgetxattr)");
    return ACL_ERROR;
  }

  /* See above: src_size must be nonnegative at this point,so we cast
     it to size_t to avoid a compiler warning. */
  if (fsetxattr(dst_fd,
                dst_name,
                src_acl_p,
                (size_t)src_size,
                0)
      == XATTR_ERROR) {
    perror("acl_copy_xattr (fsetxattr)");
    return ACL_ERROR;
  }

  return ACL_SUCCESS;
}


/**
 * @brief Determine if a file descriptor has a default ACL.
 *
 * @param fd
 *   The file descriptor whose default ACL is in question.
 *
 * @return
 *   - @c ACL_SUCCESS - If @c fd has a default ACL.
 *   - @c ACL_FAILURE - If @c fd does not have a default ACL.
 *   - @c ACL_ERROR - Unexpected library error.
 */
int has_default_acl_fd(int fd) {
  if (fgetxattr(fd, XATTR_NAME_POSIX_ACL_DEFAULT, NULL, 0) == XATTR_ERROR) {
    if (errno == ENODATA) {
      return ACL_FAILURE;
    }
    perror("has_default_acl_fd (fgetxattr)");
    return ACL_ERROR;
  }

  return ACL_SUCCESS;
}



/**
 * @brief The recursive portion of @c apply_default_acl.
 *
 * The @c apply_default_acl function takes a path, but then opens file
 * descriptors for the path and its parent. Afterwards, everything is
 * done using file descriptors, including the recursive application on
 * the path's children. This function encapsulates the portion of @c
 * apply_default_acl that uses only file descriptors; for the
 * recursion, this function ultimately calls itself.
 *
 * This overwrites any existing ACLs on @c fd and, if @c recursive is
 * @c true, its children. When @c recursive is @c true, the "worst"
 * result encountered is returned as the overall result.
 *
 * @param parent_fd
 *   A file descriptor for the parent directory of @c fd.
 *
 * @param fd
 *   The file descriptor that should inherit its parent's default ACL.
 *
 * @param recursive
 *   Should we recurse into subdirectories?
 *
 * @return
 *   - @c ACL_SUCCESS - The parent default ACLs were inherited successfully.
 *   - @c ACL_FAILURE - If symlinks or hard links are encountered.
 *   - @c ACL_ERROR - Unexpected library error.
 */
int apply_default_acl_fds(int parent_fd, int fd, bool recursive) {
  int result = ACL_SUCCESS;

  /* The new ACL for this path */
  acl_t new_acl = (acl_t)NULL;

  /* A copy of new_acl, to be made before we begin mangling new_acl in
     order to mask the execute bit. */
  acl_t new_acl_unmasked = (acl_t)NULL;

  /* Refuse to operate on hard links, which can be abused by an
   * attacker to trick us into changing the ACL on a file we didn't
   * intend to; namely the "target" of the hard link. There is TOCTOU
   * race condition here, but the window is as small as possible
   * between when we open the file descriptor (look above) and when we
   * fstat it.
  */
  struct stat s;
  if (fstat(fd, &s) == STAT_ERROR) {
    perror("apply_default_acl_fds (fstat)");
    /* We can't recurse without the stat struct for fd */
    goto cleanup;
  }


  /* Check to make sure the parent descriptor actually has a default
     ACL. If it doesn't, then we can "succeed" immediately, saving a
     little work, particularly in any_can_execute(). Note that we
     can't skip the fstat() above, because we need it in case we
     recurse. */
  if (has_default_acl_fd(parent_fd) == ACL_FAILURE) {
    result = ACL_SUCCESS;
    /* Just because this target can't inherit anything doesn't mean
       that one of it's children can't. For example, if there's a
       default on "c" in "a/b/c/d", then we don't want to skip all
       children of "a"! */
    goto recurse;
  }


  if (!S_ISDIR(s.st_mode)) {
    /* If it's not a directory, make sure it's a regular,
       non-hard-linked file. */
    if (!S_ISREG(s.st_mode) || s.st_nlink != 1) {
      result = ACL_FAILURE;
      goto cleanup; /* It's not a directory, so we can skip the recursion. */
    }
  }


  /* Next We try to guess whether or not to strip the execute bits.
   * This behavior is modeled after the capital 'X' perms of setfacl.
  */
  int ace_result = any_can_execute(fd, &s);

  if (ace_result == ACL_ERROR) {
    perror("apply_default_acl_fds (any_can_execute)");
    result = ACL_ERROR;
    goto cleanup;
  }

  /* Never mask the execute bit on directories. */
  bool allow_exec = (bool)ace_result || S_ISDIR(s.st_mode);


  /* If it's a directory, inherit the parent's default.  */
  if (S_ISDIR(s.st_mode)) {
    if (acl_copy_xattr(parent_fd,
                       ACL_TYPE_DEFAULT,
                       fd,
                       ACL_TYPE_DEFAULT) == ACL_ERROR) {
      perror("apply_default_acl_fds (acl_copy_xattr default)");
      result = ACL_ERROR;
      goto cleanup;
    }
  }

  /* If it's anything, _apply_ the parent's default. */
  if (acl_copy_xattr(parent_fd,
                     ACL_TYPE_DEFAULT,
                     fd,
                     ACL_TYPE_ACCESS) == ACL_ERROR) {
    perror("apply_default_acl_fds (acl_copy_xattr access)");
    result = ACL_ERROR;
    goto cleanup;
  }

  /* There's a good reason why we saved the ACL above, even though
   * we're about to read it back into memory and mess with it on the
   * next line. The acl_copy_xattr() function is already a hack to let
   * us copy default ACLs without resorting to path names; we simply
   * have no way to read the parent's default ACL into memory using
   * parent_fd. We can, however, copy the parent's ACL to a file (with
   * acl_copy_xattr), and then read the ACL from a file using
   * "fd". It's quite the circus, but it works and should be safe from
   * sym/hardlink attacks.
  */

  /* Now we potentially need to mask the execute permissions in the
     ACL on fd; or maybe not. */
  if (allow_exec) {
    /* Skip the mask code for this target, but don't skip its children! */
    goto recurse;
  }

  /* OK, we need to mask some execute permissions. First obtain the
     current ACL... */
  new_acl = acl_get_fd(fd);
  if (new_acl == (acl_t)NULL) {
    perror("apply_default_acl_fds (acl_get_fd)");
    result = ACL_ERROR;
    goto cleanup;
  }

  /* ...and now make a copy of it, because otherwise when we loop
     below, some shit gets stuck (modifying the structure while
     looping over it no worky). */
  new_acl_unmasked = acl_dup(new_acl);
  if (new_acl_unmasked == (acl_t)NULL) {
    perror("apply_default_acl_fds (acl_dup)");
    result = ACL_ERROR;
    goto cleanup;
  }

  acl_entry_t entry;
  int ge_result = acl_get_entry(new_acl_unmasked, ACL_FIRST_ENTRY, &entry);

  while (ge_result == ACL_SUCCESS) {
    acl_tag_t tag = ACL_UNDEFINED_TAG;

    if (acl_get_tag_type(entry, &tag) == ACL_ERROR) {
       perror("apply_default_acl_fds (acl_get_tag_type)");
       result = ACL_ERROR;
       goto cleanup;
    }


    /* We've got an entry/tag from the default ACL. Get its permset. */
    acl_permset_t permset;
    if (acl_get_permset(entry, &permset) == ACL_ERROR) {
      perror("apply_default_acl_fds (acl_get_permset)");
      result = ACL_ERROR;
      goto cleanup;
    }

    /* To mimic what the kernel does, I think we could drop
       ACL_GROUP_OBJ from the list below? */
    if (tag == ACL_MASK ||
        tag == ACL_USER_OBJ ||
        tag == ACL_GROUP_OBJ ||
        tag == ACL_OTHER) {

      /* The mask doesn't affect acl_user_obj, acl_group_obj (in
         minimal ACLs) or acl_other entries, so if execute should be
         masked, we have to do it manually. */
      if (acl_delete_perm(permset, ACL_EXECUTE) == ACL_ERROR) {
        perror("apply_default_acl_fds (acl_delete_perm)");
        result = ACL_ERROR;
        goto cleanup;
      }

      if (acl_set_permset(entry, permset) == ACL_ERROR) {
        perror("apply_default_acl_fds (acl_set_permset)");
        result = ACL_ERROR;
        goto cleanup;
      }
    }

    if (acl_update_entry(new_acl, entry) == ACL_ERROR) {
      perror("apply_default_acl_fds (acl_update_entry)");
      result = ACL_ERROR;
      goto cleanup;
    }

    ge_result = acl_get_entry(new_acl_unmasked, ACL_NEXT_ENTRY, &entry);
  }

  /* Catches the first acl_get_entry as well as the ones at the end of
     the loop. */
  if (ge_result == ACL_ERROR) {
    perror("apply_default_acl_fds (acl_get_entry)");
    result = ACL_ERROR;
    goto cleanup;
  }

  if (acl_set_fd(fd, new_acl) == ACL_ERROR) {
    perror("apply_default_acl_fds (acl_set_fd)");
    result = ACL_ERROR;
    goto cleanup;
  }

 recurse:
  if (recursive && S_ISDIR(s.st_mode)) {
    /* Recurse into subdirectories. Don't call closedir() on d! It
       closes the open file descriptor as well, and subsequent calls
       to close() then throw errors. */
    DIR* d = fdopendir(fd);
    if (d == NULL) {
      perror("apply_default_acl_fds (fdopendir)");
      result = ACL_ERROR;
      goto cleanup;
    }

    struct dirent* de;
    int new_fd = 0;
    while ((de = readdir(d)) != NULL) {
      if (de->d_type != DT_DIR && de->d_type != DT_REG) {
        /* Hit a symlink or whatever. */
        result = ACL_FAILURE;
        continue;
      }
      if (strcmp(de->d_name, ".") == 0) { continue; }
      if (strcmp(de->d_name, "..") == 0) { continue; }

      /* Be careful not to "return" out of this loop and leave the
         new_fd open! */
      new_fd = openat(fd, de->d_name, O_NOFOLLOW);
      if (new_fd == OPEN_ERROR) {
        if (errno == ELOOP || errno == ENOTDIR) {
          /* We hit a symlink, either in the last path component (ELOOP)
             or higher up (ENOTDIR). */
          if (result == ACL_SUCCESS) {
            /* Don't overwrite an error result with success/failure. */
            result = ACL_FAILURE;
          }
          continue;
        }
        else {
          perror("apply_default_acl_fds (openat)");
          result = ACL_ERROR;
          continue;
        }
      }
      switch (apply_default_acl_fds(fd, new_fd, recursive)) {
        /* Don't overwrite an error result with success/failure. */
        case ACL_FAILURE:
          if (result == ACL_SUCCESS) {
            result = ACL_FAILURE;
          }
          break;
        case ACL_ERROR:
          result = ACL_ERROR;
        default:
          if (close(new_fd) == CLOSE_ERROR) {
            perror("apply_default_acl_fds (close)");
            result = ACL_ERROR;
          }
      }
    }
  }

 cleanup:
  acl_free(new_acl);
  acl_free(new_acl_unmasked);
  return result;
}


/**
 * @brief Apply parent default ACL to a path and optionally its children.
 *
 * This overwrites any existing ACLs on the target, and, if @c
 * recursive is @c true, its children. When @c recursive is @c true,
 * the "worst" result encountered is returned as the overall result.
 *
 * @param path
 *   The path whose ACL we would like to reset to its default.
 *
 * @param recursive
 *   Should we recurse into subdirectories?
 *
 * @return
 *   - @c ACL_SUCCESS - The parent default ACLs were inherited successfully.
 *   - @c ACL_FAILURE - If symlinks or hard links are encountered.
 *   - @c ACL_ERROR - Unexpected library error.
 */
int apply_default_acl(const char* path, bool recursive) {

  if (path == NULL) {
    errno = EINVAL;
    perror("apply_default_acl (args)");
    return ACL_ERROR;
  }

  /* Define these next three variables here because we may have to
   * jump to the cleanup routine which expects them to exist.
   */

  /* Our return value. */
  int result = ACL_SUCCESS;

  /* The file descriptor corresponding to "path" */
  int fd = 0;

  /* The file descriptor for the directory containing "path" */
  int parent_fd = 0;

  /* dirname() and basename() mangle their arguments, so we need
     to make copies of "path" before using them. */
  char* dirname_path_copy = NULL;
  char* basename_path_copy = NULL;

  /* Get the parent directory of "path" with dirname(), which happens
   * to murder its argument and necessitates a path_copy. */
  dirname_path_copy = strdup(path);
  if (dirname_path_copy == NULL) {
    perror("apply_default_acl (strdup)");
    return ACL_ERROR;
  }
  char* parent = dirname(dirname_path_copy);

  basename_path_copy = strdup(path);
  if (basename_path_copy == NULL) {
    perror("apply_default_acl (strdup)");
    result = ACL_ERROR;
    goto cleanup;
  }
  char* child = basename(basename_path_copy);

  /* Just kidding, if the path is "." or "..", then dirname will do
   * the wrong thing and give us "." as its parent, too. So, we handle
   * those as special cases. We use "child" instead of "path" here to
   * catch things like "./" and "../"
   */
  bool path_is_dots = strcmp(child, ".") == 0 || strcmp(child, "..") == 0;
  char dots_parent[6] = "../";
  if (path_is_dots) {
    /* We know that "child" contains no more than two characters here, and
       using strncat to enforce that belief keeps clang-tidy happy. */
    parent = strncat(dots_parent, child, 2);
  }

  parent_fd = safe_open(parent, O_DIRECTORY | O_NOFOLLOW);

  if (parent_fd == OPEN_ERROR) {
    if (errno == ELOOP || errno == ENOTDIR) {
      /* We hit a symlink, either in the last path component (ELOOP)
         or higher up (ENOTDIR). */
      result = ACL_FAILURE;
      goto cleanup;
    }
    else {
      perror("apply_default_acl (open parent fd)");
      result = ACL_ERROR;
      goto cleanup;
    }
  }

  /* We already obtained the parent fd safely, so if we use the
   * basename of path here instead of the full thing, then we can get
   * away with using openat() and spare ourselves the slowness of
   * another safe_open().
   *
   * Note that if the basename is "." or "..", then we don't want to
   * open it relative to the parent_fd, so we need another special
   * case for those paths here.
   */
  if (path_is_dots) {
    fd = open(child, O_NOFOLLOW);
  }
  else {
    fd = openat(parent_fd, child, O_NOFOLLOW);
  }
  if (fd == OPEN_ERROR) {
    if (errno == ELOOP || errno == ENOTDIR) {
      /* We hit a symlink, either in the last path component (ELOOP)
         or higher up (ENOTDIR). */
      result = ACL_FAILURE;
      goto cleanup;
    }
    else {
      perror("apply_default_acl (open fd)");
      result = ACL_ERROR;
      goto cleanup;
    }
  }

  result = apply_default_acl_fds(parent_fd, fd, recursive);

 cleanup:
  free(dirname_path_copy);
  free(basename_path_copy);

  if (parent_fd > 0 && close(parent_fd) == CLOSE_ERROR) {
    perror("apply_default_acl (close parent_fd)");
    result = ACL_ERROR;
  }
  if (fd > 0 && close(fd) == CLOSE_ERROR) {
    perror("apply_default_acl (close fd)");
    result = ACL_ERROR;
  }
  return result;
}
