/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package errs

import (
	"errors"
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestNewCLIError(t *testing.T) {
	t.Parallel()

	type args struct {
		msg      string
		exitCode int
	}

	tests := []struct {
		name string
		args args
		want error
	}{
		{
			"+valid",
			args{"cli err", 1},
			&CLIError{Message: "cli err", ExitCode: 1},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			err := NewCLIError(tt.args.msg, tt.args.exitCode)

			if diff := cmp.Diff(tt.want, err); diff != "" {
				t.Fatalf("NewCLIError() missmatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestCLIError_Error(t *testing.T) {
	t.Parallel()

	type fields struct {
		Message  string
		ExitCode int
	}

	tests := []struct {
		name   string
		fields fields
		want   string
	}{
		{
			"+valid",
			fields{
				Message:  "cli err",
				ExitCode: 1,
			},
			"cli err",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			e := &CLIError{
				Message:  tt.fields.Message,
				ExitCode: tt.fields.ExitCode,
			}

			got := e.Error()
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("CLIError.Error() missmatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestCLIError_errorsAs(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name     string
		caseFunc func(t *testing.T)
	}{
		{
			"+valid",
			func(t *testing.T) { //nolint:thelper
				dest := &CLIError{}

				if !errors.As(NewCLIError("cli err", 1), &dest) {
					t.Fatalf(
						"errors.As(NewCLIError()) didn't match the target error",
					)
				}

				if diff := cmp.Diff(
					&CLIError{Message: "cli err", ExitCode: 1}, dest,
				); diff != "" {
					t.Fatalf(
						"errors.As(NewCLIError()) missmatch (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"+wrapped",
			func(t *testing.T) { //nolint:thelper
				dest := &CLIError{}

				if !errors.As(
					Wrap(NewCLIError("cli err", 1), "wrapping"),
					&dest,
				) {
					t.Fatalf(
						"errors.As(NewCLIError()) didn't match the target error",
					)
				}

				if diff := cmp.Diff(
					&CLIError{Message: "cli err", ExitCode: 1}, dest,
				); diff != "" {
					t.Fatalf(
						"errors.As(NewCLIError()) missmatch (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"+joined",
			func(t *testing.T) { //nolint:thelper
				dest := &CLIError{}

				if !errors.As(
					errors.Join(
						New("fail"),
						Wrap(NewCLIError("cli err", 1), "wrapping"),
					),
					&dest,
				) {
					t.Fatalf(
						"errors.As(NewCLIError()) didn't match the target error",
					)
				}

				if diff := cmp.Diff(
					&CLIError{Message: "cli err", ExitCode: 1}, dest,
				); diff != "" {
					t.Fatalf(
						"errors.As(NewCLIError()) missmatch (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"+firstBranchOfJoin",
			func(t *testing.T) { //nolint:thelper
				dest := &CLIError{}

				if !errors.As(
					errors.Join(
						Wrap(
							Wrap(NewCLIError("cli err 2", 2), "wrapping"),
							"wrapping2",
						),
						Wrap(NewCLIError("cli err", 1), "wrapping"),
					),
					&dest,
				) {
					t.Fatalf(
						"errors.As(NewCLIError()) didn't match the target error",
					)
				}

				if diff := cmp.Diff(
					&CLIError{Message: "cli err 2", ExitCode: 2}, dest,
				); diff != "" {
					t.Fatalf(
						"errors.As(NewCLIError()) missmatch (-want +got):\n%s",
						diff,
					)
				}
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			tt.caseFunc(t)
		})
	}
}
