/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

struct Control {
    public unowned string label;
    public unowned string label_for_min;
    public unowned string label_for_max;
    public int min; 
    public int max;
}

enum ControlIndex {
    TIME = 0,
    DEPTH = 1,
    DISCERNMENT = 2,
}

struct NamedBrainSpec {
    unowned string name;
    BrainSpec spec;
}

class BrainSetter : Gtk.Grid {
    /**
     * Settings for visual control of BrainSpec
     */
    private Control[] controls = {
        Control() { label = N_("Thinking Time in Seconds:"),
                    // fast: the computer must finish thinking within 1 second
                    label_for_min = N_("fast"), 
                    // slow: the computer can take up to Max.TIME seconds to think
                    label_for_max = N_("slow"),
                    min = 1, 
                    max = 10
        },
        Control() { label = N_("Number of Moves to Think Ahead:"),
                    // shallow: the computer thinks by looking one move ahead
                    label_for_min = N_("shallow"),
                    // deep: the computer thinks by looking 10 moves ahead
                    label_for_max = N_("deep"),
                    min = 1, 
                    max = 10
        },
        Control() { label = N_("Ability to Discern Good Positions:"),
                    // fuzzy: the computer does not always choose the best moves
                    label_for_min = N_("fuzzy"), 
                    // sharp: the computer prefers the best moves
                    label_for_max = N_("sharp"),
                    min = 1, 
                    max = 10
        }
    };

    private NamedBrainSpec[] brain_settings = {
        NamedBrainSpec() { 
            // TRANSLATORS: Virtual personality.
            // Cakoi: fried garlic bread. Please don't translate. Leave it, or transliterate.
            name = N_("Cakoi"),
            spec = BrainSpec() {thinking_time = 1, thinking_depth = 1, discernment = 1}
        },
        NamedBrainSpec() {
            // TRANSLATORS: Virtual personality.
            // Ambuyat: gummy sago paste. Please don't translate. Leave it, or transliterate.
            name = N_("Ambuyat"),   
            spec = BrainSpec() {thinking_time = 1, thinking_depth = 2, discernment = 2}
        },
        NamedBrainSpec() {
            // TRANSLATORS: Virtual personality.
            // Belacan: fermented shrimp paste. Please don't translate. Leave it, or transliterate.
            name = N_("Belacan"), 
            spec = BrainSpec() {thinking_time = 2, thinking_depth = 4, discernment = 4}
        },
        NamedBrainSpec() {
            // TRANSLATORS: Virtual personality.
            // Cincaluk: fermented shrimp dip. Please don't translate. Leave it, or transliterate.
            name = N_("Cincaluk"),
            spec = BrainSpec() {thinking_time = 4, thinking_depth = 5, discernment = 5}
        },
        NamedBrainSpec() {
            // TRANSLATORS: Virtual personality.
            // Binjai: a kind of fruit. Please don't translate. Leave it, or transliterate.
            name = N_("Binjai"),
            spec = BrainSpec() {thinking_time = 1, thinking_depth = 2, discernment = 10}
        },
        NamedBrainSpec() {
            // TRANSLATORS: Virtual personality.
            // Piasau: coconut. Please don't translate. Leave it, or transliterate.
            name = N_("Piasau"),
            spec = BrainSpec() {thinking_time = 2, thinking_depth = 10, discernment = 10}
        },
        NamedBrainSpec() {
            // TRANSLATORS: Virtual personality.
            // Kambayau: a kind of fruit. Please don't translate. Leave it, or transliterate.
            name = N_("Kambayau"),
            spec = BrainSpec() {thinking_time = 6, thinking_depth = 10, discernment = 10}
        }
    };

    private Gtk.Adjustment[] sliders = null;
    private Gtk.ComboBoxText name_list = null;

    /**
     * Set to true to prevent a chain reaction of change signals.
     */
    private bool auto_adjusting = false;

    public BrainSpec brain_spec = BrainSpec ();

    public BrainSetter () {
        create_components ();
    }

    private void create_components () {
        orientation = Gtk.Orientation.VERTICAL;
        set_row_spacing (10);

        // List out preset brains
        name_list = new Gtk.ComboBoxText ();
        name_list.changed.connect (on_name_changed);
        foreach (var vp in brain_settings) {
            name_list.append_text ( _(vp.name));
        }
        add (name_list);

        // Construct Adjustments and their views
        sliders = new Gtk.Adjustment[controls.length];
        for (int i=0; i < controls.length; i++) {
            sliders[i] = new Gtk.Adjustment (controls[i].min, controls[i].min, controls[i].max + 1, 1, 1, 1);
            sliders[i].value_changed.connect (on_value_changed);

            var scale = new Gtk.Scale (Gtk.Orientation.HORIZONTAL, sliders[i]);
            scale.set_digits (0);
            scale.set_size_request (300, -1);
            scale.set_draw_value (true);
            scale.set_value_pos (Gtk.PositionType.BOTTOM);
            var index = i;
            scale.format_value.connect ((source, val) => {return tick_label (val, index);});

            var label = new Gtk.Label ( _(controls[i].label));
            var box = new Gtk.Grid ();
            box.orientation = Gtk.Orientation.VERTICAL;
            box.add (label);
            box.add (scale);
            add (box);
        }//endfor i

        // Initial brain spec: choose Ambuyat
        name_list.active = 1;
        get_style_context () .add_class ("box-top-level");
        show_all ();
    }

    /**
     * Format for slider ticks
     */
    private string tick_label (double _val, int id) {
        int val = (int) _val;
        if (val == controls[id].min) return _(controls[id].label_for_min);
        if (val == controls[id].max) return _(controls[id].label_for_max);
        return "" + val.to_string ();
    }

    /**
     * Callback when a slider value changes.
     */
    private void on_value_changed () {
        int  thinking_time  = (int) sliders[ControlIndex.TIME].@value;
        int  thinking_depth = (int) sliders[ControlIndex.DEPTH].@value;
        int  discernment   = (int) sliders[ControlIndex.DISCERNMENT].@value;
        name_list.active = -1;
        for (int i=0; i < brain_settings.length; i++) {
            if (thinking_time == brain_settings[i].spec.thinking_time
                && thinking_depth == brain_settings[i].spec.thinking_depth
                && discernment == brain_settings[i].spec.discernment) {
                auto_adjusting = true;
                name_list.active = i;
                auto_adjusting = false;
                break;
            }
        }
        update_brain_spec ();
    }

    /**
     * Callback when a different named virtual personality is chosen.
     */
    private void on_name_changed () {
        if (auto_adjusting) return;
        int row = name_list.active;
        if (row != -1) {
            auto_adjusting = true;
            sliders[ControlIndex.TIME].@value = brain_settings[row].spec.thinking_time;
            sliders[ControlIndex.DEPTH].@value = brain_settings[row].spec.thinking_depth;
            sliders[ControlIndex.DISCERNMENT].@value = brain_settings[row].spec.discernment;
            auto_adjusting = false;
        }
        update_brain_spec ();
    }

    /**
     * Update the public BrainSpec
     */
    private void update_brain_spec () {
        brain_spec.thinking_time  = (int) sliders[ControlIndex.TIME].@value;
        brain_spec.thinking_depth = (int) sliders[ControlIndex.DEPTH].@value;
        brain_spec.discernment   = (int) sliders[ControlIndex.DISCERNMENT].@value;
    }
}//class
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
