/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2015  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class Pattern {
    /**
     * Untranslated name of the pattern.
     */
    public string name;

    /**
     * String representation of the pattern. Only half is given. The other half
     * can be obtained by symmetry.
     */
    public string dots;

    /**
     * Pixbuf rendition of patterns.
     */
    protected uint8[] pixbuf_data = null;
    protected Gdk.Pixbuf pixbuf = null;

    /**
     * The size of each white dot inside the pixbuf
     */
    public int dot_width = 4;

    /**
     * Button for user selection.
     * Button image will be updated when the pixbuf is changed.
     */
    public Gtk.RadioButton button = new Gtk.RadioButton (null);

    public Pattern (string name, string dots) {
        this.name = name;
        this.dots = dots;
        create_pixbuf ();
    }

    public Pattern.empty (string name) {
        this.name = name;
    }

    /**
     * Change dot_width
     */
    public void set_dot_width (int width) {
        if (dot_width == width) return;
        dot_width = width;
        create_pixbuf ();
    }

    /**
     * Background of the pattern (each call paint one pixel only).
     * n : position of the pixel relative to pixbuf_data.
     * x, y : position of the pixel relative to top-left corner of the dot.
     */
    protected virtual void background (int n, int x, int y) {
        pixbuf_data[n + 0] = 
        pixbuf_data[n + 1] = 
        pixbuf_data[n + 2] = 100 - 100 / (3 - (x + y) % 3);
        pixbuf_data[n + 3] = 255;
    }

    /**
     * Render a pattern as a pixbuf. dots must be legal, i.e. compilable into
     * a board with 60 whites and 60 blacks.
     */
    protected virtual void create_pixbuf () {
        Piece[] board = new Piece[BOARD_SIZE];
        Game.compile (dots, ref board);
        create_pixbuf_from_board (ref board);
    }

    /**
     * Render a pattern as a pixbuf. board doesn't have to be legal.
     * e.g. there may be less than 60 whites.
     * This is to cater for unfinished custom patterns.
     */
    protected void create_pixbuf_from_board (ref Piece[] board) {
        // size in pixels
        int dot_cut = dot_width;
        int image_width = BOARD_WIDTH * dot_width;
        // size in bytes (in-memory representation)
        int sizeof_pixel = 4;
        int sizeof_dot = dot_width * dot_width * sizeof_pixel;
        int sizeof_image = image_width * image_width * sizeof_dot;
        pixbuf_data = new uint8[sizeof_image];

        for (int i=0; i < BOARD_SIZE; i++) {
            int dot_ndx = i / BOARD_WIDTH * BOARD_WIDTH * sizeof_dot
                          + i % BOARD_WIDTH * dot_width * sizeof_pixel;
            
            for (int px=0; px < dot_width; px++) {
                for (int py=0; py < dot_width; py++) {
                    int n = dot_ndx  +  py * image_width * sizeof_pixel  +  px * sizeof_pixel;
                    if (board[i] == Piece.WHITE && px % dot_cut != dot_cut-1) {
                        // The pattern weaved in gold
                        pixbuf_data[n + 0] = 
                        pixbuf_data[n + 1] = 155 + 100 / (py % dot_cut  + 1);
                        pixbuf_data[n + 2] = 55 + 100 / (py % dot_cut + 1);;
                        pixbuf_data[n + 3] = 255;
                    }
                    else {
                        background (n, px, py);
                    }
                }//endfor py
            }//endfor px
        }//endfor i

        pixbuf = new Gdk.Pixbuf.with_unowned_data (
            pixbuf_data,
            Gdk.Colorspace.RGB, 
            true,                     // has alpha
            8,                        // bits per sample
            image_width,
            image_width,
            image_width * sizeof_pixel);

        button.image = new Gtk.Image.from_pixbuf (pixbuf);
    }
}//class Pattern

}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
