/* $Id$ */
/*
 * utilities.
 */
#include <ctype.h>
#if defined HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#if defined HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif /* HAVE_SYS_IOCTL_H */
#include "common.h"
#include "utils.h"

struct progress_meter {
  int count;
  int count_max;
  int count2;
  int col;
  char *tmp;
};

static void replace_case(char *dst, const char *src, const char *from, const char *to);
static char *strstr_case(const char *haystack, const char *needle);
static int terminal_columns(void);

ProgressMeter *progress_meter_new(int count_max)
{
  ProgressMeter *p;
  int col;

  col = terminal_columns();
  p = XMALLOC(ProgressMeter, 1);
  p->count = 0;
  p->count_max = count_max;
  p->count2 = 0;
  p->col = col;
  p->tmp = XMALLOC(char, col);
  return p;
}

void progress_meter_delete(ProgressMeter *p)
{
  XFREE(p->tmp);
  XFREE(p);
}

void progress_meter_print(ProgressMeter *p)
{
  const char vmeter[] = "|\\-/";
  char *tmp;
  int n, m, l;

  if ((p->count++)%20 == 0) {
    return;
  }
  n = p->count*100/p->count_max; /* percent of progress */
  m = n*(p->col-21)/100;
  l = p->col-20-m;
  tmp = p->tmp;
  rstr(tmp, '=', m);
  rstr(tmp+m, ' ', l);
  fprintf(stderr, "\rprogress|%s%d%%%c", tmp, n, vmeter[p->count2%4]);
  p->count2++;
}

void progress_meter_clear(ProgressMeter *p)
{
  char *tmp;
  int col;

  tmp = p->tmp;
  col = p->col;
  fprintf(stderr, "\r%s\r", rstr(tmp, ' ', col-3));
}


FILE *file_open(char *filename)
{
  FILE *fp;

  if (filename != NULL) {
    fp = fopen(filename, "r");
  } else {
    fp = stdin;
  }
  if (fp == NULL) {
    exit(EXIT_FAILURE);
  }
  return fp;
}

/*
 * replace font property.
 */
int replace_property(FILE *fp, size_t size, Property *prop)
{
  char *buf, *outbuf, *keyword;
  int max_chars = -1;
  char font[2][16], slant[2][8];
  const char *weight_name[2];
  int i;

  if (prop->weight[0] != NULL) {
    for (i = 0; i < 2; i++) {
      weight_name[i] = prop->weight[i];
    }
  } else {
    for (i = 0; i < 2; i++) {
      weight_name[i] = "";
    }
  }
  if (prop->slant[0] != NULL) {
    for (i = 0; i < 2; i++) {
      sprintf(slant[i], "\"%.2s\"", prop->slant[i]);
      sprintf(font[i], "%.11s-%.2s-", weight_name[i], prop->slant[i]);
    }
  } else {
    for (i = 0; i < 2; i++) {
      sprintf(font[i], "%.11s-", weight_name[i]);
    }
  }
  buf = XMALLOC(char, size);
  outbuf = XMALLOC(char, size);
  keyword = XMALLOC(char, size);
  while (fgets(buf, size, fp) != NULL) {
    if (isalpha(buf[0])) {
      sscanf(buf, "%s", keyword);
      if (strcmp(keyword, "FONT") == 0) {
        /* replace the font property */
        replace_case(outbuf, buf, font[0], font[1]);
        fputs(outbuf, stdout);
        continue;
      }
      if ((strcmp(keyword, "WEIGHT_NAME") == 0) && (prop->weight[0] != NULL)) {
        /* replace the font property */
        replace_case(outbuf, buf, weight_name[0], weight_name[1]);
        fputs(outbuf, stdout);
        continue;
      }
      if ((strcmp(keyword, "SLANT") == 0) && (prop->slant[0] != NULL)) {
        /* replace the font property */
        replace_case(outbuf, buf, slant[0], slant[1]);
        fputs(outbuf, stdout);
        continue;
      }
      if (strcmp(keyword, "FONTBOUNDINGBOX") == 0) {
        /* recast the metric information from the bounding box */
        if (!(*(prop->recast))(outbuf, prop->width, prop->height, prop->dw, prop->dy, buf)) {
          return -1;
        }
        fputs(outbuf, stdout);
        continue;
      }
      if (strcmp(keyword, "CHARS") == 0) {
        max_chars = atoi(buf+6);
        fputs(buf, stdout);
        break;
      }
    }
    fputs(buf, stdout);
  }
  XFREE(buf);
  XFREE(outbuf);
  XFREE(keyword);
  return max_chars;
}

/*
 * replace substring ignoring case.
 */
static void replace_case(char *dst, const char *src, const char *from, const char *to)
{
  const char *index;
  size_t len, len_from, len_to;

  len_from = strlen(from);
  len_to = strlen(to);
  while ((index = strstr_case(src, from)) != NULL) {
    len = index-src;
    strncpy(dst, src, len);
    dst += len;
    src += len+len_from;
    strncpy(dst, to, len_to);
    dst += len_to;
  }
  strcpy(dst, src);
}

/*
 * repeat character string.
 */
char *rstr(char *buf, char c, size_t n)
{
  char *p;
  size_t i;

  p = buf;
  for (i = 0; i < n; i++) {
    *(p++) = c;
  }
  *p = '\0';
  return buf;
}

/*
 * strstr() ignoring case.
 */
static char *strstr_case(const char *haystack, const char *needle)
{
  const char *p1, *p2;

  while (*haystack != '\0') {
    p1 = haystack;
    p2 = needle;
    while (tolower(*p2) == tolower(*p1)) {
      p1++;
      p2++;
    }
    if (*p2 == '\0') {
      return (char *)haystack;
    }
    haystack++;
  }
  return NULL;
}

static int terminal_columns(void)
{
  int columns;
  const char *s;

#if defined HAVE_UNISTD_H
#if defined HAVE_SYS_IOCTL_H
#if defined TIOCGWINSZ
  struct winsize win;
  int err;

  err = ioctl(STDERR_FILENO, TIOCGWINSZ, &win);
  if ((err == 0) && (win.ws_col > 0)) {
    return win.ws_col;
  }
#endif /* TIOCGWINSZ */
#endif /* HAVE_SYS_IOCTL_H */
#endif /* HAVE_UNISTD_H */

  columns = 80;
  if ((s = getenv("COLUMNS")) != NULL) {
    columns = atoi(s);
  }
  return columns;
}
