package utils

import (
	"bytes"
	"encoding/json"
	"fmt"
	"os"
	"path/filepath"
	"regexp"
	"runtime"
	"strconv"
	"strings"

	"github.com/jfrog/jfrog-client-go/utils/io/fileutils"

	"github.com/jfrog/jfrog-client-go/utils/errorutils"
	"github.com/jfrog/jfrog-client-go/utils/log"
)

const (
	Development = "development"
	Agent       = "jfrog-client-go"
	Version     = "0.28.1"
)

// In order to limit the number of items loaded from a reader into the memory, we use a buffers with this size limit.
var MaxBufferSize = 50000

var userAgent = getDefaultUserAgent()

func getVersion() string {
	return Version
}

func GetUserAgent() string {
	return userAgent
}

func SetUserAgent(newUserAgent string) {
	userAgent = newUserAgent
}

func getDefaultUserAgent() string {
	return fmt.Sprintf("%s/%s", Agent, getVersion())
}

// Get the local root path, from which to start collecting artifacts to be used for:
// 1. Uploaded to Artifactory,
// 2. Adding to the local build-info, to be later published to Artifactory.
func GetRootPath(path string, patternType PatternType, parentheses ParenthesesSlice) string {
	// The first step is to split the local path pattern into sections, by the file separator.
	separator := "/"
	sections := strings.Split(path, separator)
	if len(sections) == 1 {
		separator = "\\"
		sections = strings.Split(path, separator)
	}

	// Now we start building the root path, making sure to leave out the sub-directory that includes the pattern.
	rootPath := ""
	for _, section := range sections {
		if section == "" {
			continue
		}
		if patternType == RegExp {
			if strings.Index(section, "(") != -1 {
				break
			}
		} else {
			if strings.Index(section, "*") != -1 {
				break
			}
			if strings.Index(section, "(") != -1 {
				temp := rootPath + section
				if isWildcardParentheses(temp, parentheses) {
					break
				}
			}
			if patternType == AntPattern {
				if strings.Index(section, "?") != -1 {
					break
				}
			}
		}
		if rootPath != "" {
			rootPath += separator
		}
		if section == "~" {
			rootPath += GetUserHomeDir()
		} else {
			rootPath += section
		}
	}
	if len(sections) > 0 && sections[0] == "" {
		rootPath = separator + rootPath
	}
	if rootPath == "" {
		return "."
	}
	return rootPath
}

// Return true if the ‘str’ argument contains open parentasis, that is related to a placeholder.
// The ‘parentheses’ argument contains all the indexes of placeholder parentheses.
func isWildcardParentheses(str string, parentheses ParenthesesSlice) bool {
	toFind := "("
	currStart := 0
	for {
		idx := strings.Index(str, toFind)
		if idx == -1 {
			break
		}
		if parentheses.IsPresent(idx) {
			return true
		}
		currStart += idx + len(toFind)
		str = str[idx+len(toFind):]
	}
	return false
}

func StringToBool(boolVal string, defaultValue bool) (bool, error) {
	if len(boolVal) > 0 {
		result, err := strconv.ParseBool(boolVal)
		errorutils.CheckError(err)
		return result, err
	}
	return defaultValue, nil
}

func AddTrailingSlashIfNeeded(url string) string {
	if url != "" && !strings.HasSuffix(url, "/") {
		url += "/"
	}
	return url
}

func IndentJson(jsonStr []byte) string {
	return doIndentJson(jsonStr, "", "  ")
}

func IndentJsonArray(jsonStr []byte) string {
	return doIndentJson(jsonStr, "  ", "  ")
}

func doIndentJson(jsonStr []byte, prefix, indent string) string {
	var content bytes.Buffer
	err := json.Indent(&content, jsonStr, prefix, indent)
	if err == nil {
		return content.String()
	}
	return string(jsonStr)
}

func MergeMaps(src map[string]string, dst map[string]string) {
	for k, v := range src {
		dst[k] = v
	}
}

func CopyMap(src map[string]string) (dst map[string]string) {
	dst = make(map[string]string)
	for k, v := range src {
		dst[k] = v
	}
	return
}

func PrepareLocalPathForUpload(localPath string, patternType PatternType) string {
	if localPath == "./" || localPath == ".\\" {
		return "^.*$"
	}
	if strings.HasPrefix(localPath, "./") {
		localPath = localPath[2:]
	} else if strings.HasPrefix(localPath, ".\\") {
		localPath = localPath[3:]
	}
	if patternType == AntPattern {
		localPath = antPatternToRegExp(cleanPath(localPath))
	} else if patternType == WildCardPattern {
		localPath = WildcardPathToRegExp(cleanPath(localPath))
	}

	return localPath
}

// Clean /../ | /./ using filepath.Clean.
func cleanPath(path string) string {
	temp := path[len(path)-1:]
	path = filepath.Clean(path)
	if temp == `\` || temp == "/" {
		path += temp
	}
	// Since filepath.Clean replaces \\ with \, we revert this action.
	path = strings.Replace(path, `\`, `\\`, -1)
	return path
}

func WildcardPathToRegExp(localPath string) string {
	localPath = replaceSpecialChars(localPath)
	var wildcard = ".*"
	localPath = strings.Replace(localPath, "*", wildcard, -1)
	if strings.HasSuffix(localPath, "/") || strings.HasSuffix(localPath, "\\") {
		localPath += wildcard
	}
	return "^" + localPath + "$"
}

func antPatternToRegExp(localPath string) string {
	localPath = replaceSpecialChars(localPath)
	separator := getFileSeparator()
	var wildcard = ".*"
	// ant `*` ~ regexp `([^/]*)` : `*` matches zero or more characters except from `/`.
	var regAsterisk = "([^" + separator + "]*)"
	// ant `**` ~ regexp `(.*)?` : `**` matches zero or more 'directories' in a path.
	var doubleRegAsterisk = "(" + wildcard + ")?"
	var doubleRegAsteriskWithSeperatorPrefix = "(" + wildcard + separator + ")?"
	var doubleRegAsteriskWithSeperatorSuffix = "(" + separator + wildcard + ")?"

	// `?` => `.{1}` : `?` matches one character.
	localPath = strings.Replace(localPath, `?`, ".{1}", -1)
	// `*` => `([^/]*)`
	localPath = strings.Replace(localPath, `*`, regAsterisk, -1)
	// `**` => `(.*)?`
	localPath = strings.Replace(localPath, regAsterisk+regAsterisk, doubleRegAsterisk, -1)
	// `(.*)?/` => `(.*/)?`
	localPath = strings.Replace(localPath, doubleRegAsterisk+separator, doubleRegAsteriskWithSeperatorPrefix, -1)
	// Convert the last '/**' in the expression if exist : `/(.*)?` => `(/.*)?`
	if strings.HasSuffix(localPath, separator+doubleRegAsterisk) {
		localPath = strings.TrimSuffix(localPath, separator+doubleRegAsterisk) + doubleRegAsteriskWithSeperatorSuffix
	}

	if strings.HasSuffix(localPath, "/") || strings.HasSuffix(localPath, "\\") {
		localPath += wildcard
	}
	return "^" + localPath + "$"
}

func getFileSeparator() string {
	if IsWindows() {
		return "\\\\"
	}
	return "/"
}

func replaceSpecialChars(path string) string {
	var specialChars = []string{".", "^", "$", "+"}
	for _, char := range specialChars {
		path = strings.Replace(path, char, "\\"+char, -1)
	}
	return path
}

// Replaces matched regular expression from path to corresponding placeholder {i} at target.
// Example 1:
//
//	pattern = "repoA/1(.*)234" ; path = "repoA/1hello234" ; target = "{1}" ; ignoreRepo = false
//	returns "hello"
//
// Example 2:
//
//	pattern = "repoA/1(.*)234" ; path = "repoB/1hello234" ; target = "{1}" ; ignoreRepo = true
//	returns "hello"
func BuildTargetPath(pattern, path, target string, ignoreRepo bool) (string, error) {
	asteriskIndex := strings.Index(pattern, "*")
	slashIndex := strings.Index(pattern, "/")
	if shouldRemoveRepo(ignoreRepo, asteriskIndex, slashIndex) {
		// Removing the repository part of the path is required when working with virtual repositories, as the pattern
		// may contain the virtual-repository name, but the path contains the local-repository name.
		pattern = removeRepoFromPath(pattern)
		path = removeRepoFromPath(path)
	}
	pattern = addEscapingParentheses(pattern, target)
	pattern = WildcardPathToRegExp(pattern)
	if slashIndex < 0 {
		// If '/' doesn't exist, add an optional trailing-slash to support cases in which the provided pattern
		// is only the repository name.
		dollarIndex := strings.LastIndex(pattern, "$")
		pattern = pattern[:dollarIndex]
		pattern += "(/.*)?$"
	}

	r, err := regexp.Compile(pattern)
	err = errorutils.CheckError(err)
	if err != nil {
		return "", err
	}

	groups := r.FindStringSubmatch(path)
	size := len(groups)
	if size > 0 {
		for i := 1; i < size; i++ {
			group := strings.Replace(groups[i], "\\", "/", -1)
			target = strings.Replace(target, "{"+strconv.Itoa(i)+"}", group, -1)
		}
	}
	return target, nil
}

func GetLogMsgPrefix(threadId int, dryRun bool) string {
	var strDryRun string
	if dryRun {
		strDryRun = "[Dry run] "
	}
	return "[Thread " + strconv.Itoa(threadId) + "] " + strDryRun
}

func TrimPath(path string) string {
	path = strings.Replace(path, "\\", "/", -1)
	path = strings.Replace(path, "//", "/", -1)
	path = strings.Replace(path, "../", "", -1)
	path = strings.Replace(path, "./", "", -1)
	return path
}

func Bool2Int(b bool) int {
	if b {
		return 1
	}
	return 0
}

func ReplaceTildeWithUserHome(path string) string {
	if len(path) > 1 && path[0:1] == "~" {
		return GetUserHomeDir() + path[1:]
	}
	return path
}

func GetUserHomeDir() string {
	if IsWindows() {
		home := os.Getenv("HOMEDRIVE") + os.Getenv("HOMEPATH")
		if home == "" {
			home = os.Getenv("USERPROFILE")
		}
		return strings.Replace(home, "\\", "\\\\", -1)
	}
	return os.Getenv("HOME")
}

func GetBoolEnvValue(flagName string, defValue bool) (bool, error) {
	envVarValue := os.Getenv(flagName)
	if envVarValue == "" {
		return defValue, nil
	}
	val, err := strconv.ParseBool(envVarValue)
	err = CheckErrorWithMessage(err, "can't parse environment variable "+flagName)
	return val, err
}

func CheckErrorWithMessage(err error, message string) error {
	if err != nil {
		log.Error(message)
		err = errorutils.CheckError(err)
	}
	return err
}

func ConvertSliceToMap(slice []string) map[string]bool {
	mapFromSlice := make(map[string]bool)
	for _, value := range slice {
		mapFromSlice[value] = true
	}
	return mapFromSlice
}

func removeRepoFromPath(path string) string {
	if idx := strings.Index(path, "/"); idx != -1 {
		return path[idx:]
	}
	return path
}

func shouldRemoveRepo(ignoreRepo bool, asteriskIndex, slashIndex int) bool {
	if !ignoreRepo || slashIndex < 0 {
		return false
	}
	if asteriskIndex < 0 {
		return true
	}
	return IsSlashPrecedeAsterisk(asteriskIndex, slashIndex)
}

func IsSlashPrecedeAsterisk(asteriskIndex, slashIndex int) bool {
	return slashIndex < asteriskIndex && slashIndex >= 0
}

// Split str by the provided separator, escaping the separator if it is prefixed by a back-slash.
func SplitWithEscape(str string, separator rune) []string {
	var parts []string
	var current bytes.Buffer
	escaped := false
	for _, char := range str {
		if char == '\\' {
			if escaped {
				current.WriteRune(char)
			}
			escaped = true
		} else if char == separator && !escaped {
			parts = append(parts, current.String())
			current.Reset()
		} else {
			escaped = false
			current.WriteRune(char)
		}
	}
	parts = append(parts, current.String())
	return parts
}

func AddProps(oldProps, additionalProps string) string {
	if len(oldProps) > 0 && !strings.HasSuffix(oldProps, ";") && len(additionalProps) > 0 {
		oldProps += ";"
	}
	return oldProps + additionalProps
}

func IsWindows() bool {
	return runtime.GOOS == "windows"
}

type Artifact struct {
	LocalPath         string
	TargetPath        string
	SymlinkTargetPath string
}

const (
	WildCardPattern PatternType = "wildcard"
	RegExp          PatternType = "regexp"
	AntPattern      PatternType = "ant"
)

type PatternType string

type PatternTypes struct {
	RegExp bool
	Ant    bool
}

func GetPatternType(patternTypes PatternTypes) PatternType {
	if patternTypes.RegExp {
		return RegExp
	}
	if patternTypes.Ant {
		return AntPattern
	}
	return WildCardPattern
}

type Sha256Summary struct {
	sha256    string
	succeeded bool
}

func NewSha256Summary() *Sha256Summary {
	return &Sha256Summary{}
}

func (bps *Sha256Summary) IsSucceeded() bool {
	return bps.succeeded
}

func (bps *Sha256Summary) SetSucceeded(succeeded bool) *Sha256Summary {
	bps.succeeded = succeeded
	return bps
}

func (bps *Sha256Summary) GetSha256() string {
	return bps.sha256
}

func (bps *Sha256Summary) SetSha256(sha256 string) *Sha256Summary {
	bps.sha256 = sha256
	return bps
}

// Represents a file transfer from SourcePath to TargetPath.
// Each of the paths can be on the local machine (full or relative) or in Artifactory (full URL).
// The file's Sha256 is calculated by Artifactory during the upload. we read the sha256 from the HTTP's response body.
type FileTransferDetails struct {
	SourcePath string `json:"sourcePath,omitempty"`
	TargetPath string `json:"targetPath,omitempty"`
	Sha256     string `json:"sha256,omitempty"`
}

// Represent deployed artifact's details returned from build-info project for maven and gradle.
type DeployableArtifactDetails struct {
	SourcePath       string `json:"sourcePath,omitempty"`
	ArtifactDest     string `json:"artifactDest,omitempty"`
	Sha256           string `json:"sha256,omitempty"`
	DeploySucceeded  bool   `json:"deploySucceeded,omitempty"`
	TargetRepository string `json:"targetRepository,omitempty"`
}

func (detailes *DeployableArtifactDetails) CreateFileTransferDetails(rtUrl, targetRepository string) FileTransferDetails {
	targetPath := rtUrl + targetRepository + "/" + detailes.ArtifactDest
	return FileTransferDetails{SourcePath: detailes.SourcePath, TargetPath: targetPath, Sha256: detailes.Sha256}
}

type UploadResponseBody struct {
	Checksums ChecksumDetails `json:"checksums,omitempty"`
}

type ChecksumDetails struct {
	Md5    string
	Sha1   string
	Sha256 string
}

func SaveFileTransferDetailsInTempFile(filesDetails *[]FileTransferDetails) (string, error) {
	tempFile, err := fileutils.CreateTempFile()
	if err != nil {
		return "", err
	}
	filePath := tempFile.Name()
	return filePath, SaveFileTransferDetailsInFile(filePath, filesDetails)
}

func SaveFileTransferDetailsInFile(filePath string, details *[]FileTransferDetails) error {
	// Marshal and save files details to a file.
	// The details will be saved in a json format in an array with key "files" for printing later
	finalResult := struct {
		Files *[]FileTransferDetails `json:"files"`
	}{}
	finalResult.Files = details
	files, err := json.Marshal(finalResult)
	if err != nil {
		return errorutils.CheckError(err)
	}
	return errorutils.CheckError(os.WriteFile(filePath, files, 0700))
}

// Extract sha256 of the uploaded file (calculated by artifactory) from the response's body.
// In case of uploading archive with "--explode" the response body will be empty and sha256 won't be shown at
// the detailed summary.
func ExtractSha256FromResponseBody(body []byte) (string, error) {
	if len(body) > 0 {
		responseBody := new(UploadResponseBody)
		err := json.Unmarshal(body, &responseBody)
		if errorutils.CheckError(err) != nil {
			return "", err
		}
		return responseBody.Checksums.Sha256, nil
	}
	return "", nil
}
