package smartdetectoralertrules

import (
	"fmt"
	"strings"

	"github.com/hashicorp/go-azure-helpers/resourcemanager/recaser"
	"github.com/hashicorp/go-azure-helpers/resourcemanager/resourceids"
)

// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License. See NOTICE.txt in the project root for license information.

func init() {
	recaser.RegisterResourceId(&SmartDetectorAlertRuleId{})
}

var _ resourceids.ResourceId = &SmartDetectorAlertRuleId{}

// SmartDetectorAlertRuleId is a struct representing the Resource ID for a Smart Detector Alert Rule
type SmartDetectorAlertRuleId struct {
	SubscriptionId             string
	ResourceGroupName          string
	SmartDetectorAlertRuleName string
}

// NewSmartDetectorAlertRuleID returns a new SmartDetectorAlertRuleId struct
func NewSmartDetectorAlertRuleID(subscriptionId string, resourceGroupName string, smartDetectorAlertRuleName string) SmartDetectorAlertRuleId {
	return SmartDetectorAlertRuleId{
		SubscriptionId:             subscriptionId,
		ResourceGroupName:          resourceGroupName,
		SmartDetectorAlertRuleName: smartDetectorAlertRuleName,
	}
}

// ParseSmartDetectorAlertRuleID parses 'input' into a SmartDetectorAlertRuleId
func ParseSmartDetectorAlertRuleID(input string) (*SmartDetectorAlertRuleId, error) {
	parser := resourceids.NewParserFromResourceIdType(&SmartDetectorAlertRuleId{})
	parsed, err := parser.Parse(input, false)
	if err != nil {
		return nil, fmt.Errorf("parsing %q: %+v", input, err)
	}

	id := SmartDetectorAlertRuleId{}
	if err = id.FromParseResult(*parsed); err != nil {
		return nil, err
	}

	return &id, nil
}

// ParseSmartDetectorAlertRuleIDInsensitively parses 'input' case-insensitively into a SmartDetectorAlertRuleId
// note: this method should only be used for API response data and not user input
func ParseSmartDetectorAlertRuleIDInsensitively(input string) (*SmartDetectorAlertRuleId, error) {
	parser := resourceids.NewParserFromResourceIdType(&SmartDetectorAlertRuleId{})
	parsed, err := parser.Parse(input, true)
	if err != nil {
		return nil, fmt.Errorf("parsing %q: %+v", input, err)
	}

	id := SmartDetectorAlertRuleId{}
	if err = id.FromParseResult(*parsed); err != nil {
		return nil, err
	}

	return &id, nil
}

func (id *SmartDetectorAlertRuleId) FromParseResult(input resourceids.ParseResult) error {
	var ok bool

	if id.SubscriptionId, ok = input.Parsed["subscriptionId"]; !ok {
		return resourceids.NewSegmentNotSpecifiedError(id, "subscriptionId", input)
	}

	if id.ResourceGroupName, ok = input.Parsed["resourceGroupName"]; !ok {
		return resourceids.NewSegmentNotSpecifiedError(id, "resourceGroupName", input)
	}

	if id.SmartDetectorAlertRuleName, ok = input.Parsed["smartDetectorAlertRuleName"]; !ok {
		return resourceids.NewSegmentNotSpecifiedError(id, "smartDetectorAlertRuleName", input)
	}

	return nil
}

// ValidateSmartDetectorAlertRuleID checks that 'input' can be parsed as a Smart Detector Alert Rule ID
func ValidateSmartDetectorAlertRuleID(input interface{}, key string) (warnings []string, errors []error) {
	v, ok := input.(string)
	if !ok {
		errors = append(errors, fmt.Errorf("expected %q to be a string", key))
		return
	}

	if _, err := ParseSmartDetectorAlertRuleID(v); err != nil {
		errors = append(errors, err)
	}

	return
}

// ID returns the formatted Smart Detector Alert Rule ID
func (id SmartDetectorAlertRuleId) ID() string {
	fmtString := "/subscriptions/%s/resourceGroups/%s/providers/Microsoft.AlertsManagement/smartDetectorAlertRules/%s"
	return fmt.Sprintf(fmtString, id.SubscriptionId, id.ResourceGroupName, id.SmartDetectorAlertRuleName)
}

// Segments returns a slice of Resource ID Segments which comprise this Smart Detector Alert Rule ID
func (id SmartDetectorAlertRuleId) Segments() []resourceids.Segment {
	return []resourceids.Segment{
		resourceids.StaticSegment("staticSubscriptions", "subscriptions", "subscriptions"),
		resourceids.SubscriptionIdSegment("subscriptionId", "12345678-1234-9876-4563-123456789012"),
		resourceids.StaticSegment("staticResourceGroups", "resourceGroups", "resourceGroups"),
		resourceids.ResourceGroupSegment("resourceGroupName", "example-resource-group"),
		resourceids.StaticSegment("staticProviders", "providers", "providers"),
		resourceids.ResourceProviderSegment("staticMicrosoftAlertsManagement", "Microsoft.AlertsManagement", "Microsoft.AlertsManagement"),
		resourceids.StaticSegment("staticSmartDetectorAlertRules", "smartDetectorAlertRules", "smartDetectorAlertRules"),
		resourceids.UserSpecifiedSegment("smartDetectorAlertRuleName", "smartDetectorAlertRuleName"),
	}
}

// String returns a human-readable description of this Smart Detector Alert Rule ID
func (id SmartDetectorAlertRuleId) String() string {
	components := []string{
		fmt.Sprintf("Subscription: %q", id.SubscriptionId),
		fmt.Sprintf("Resource Group Name: %q", id.ResourceGroupName),
		fmt.Sprintf("Smart Detector Alert Rule Name: %q", id.SmartDetectorAlertRuleName),
	}
	return fmt.Sprintf("Smart Detector Alert Rule (%s)", strings.Join(components, "\n"))
}
