// Copyright (C) 2024 T. Zachary Laine
//
// Distributed under the Boost Software License, Version 1.0. (See
// accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
#ifndef BOOST_PARSER_UNICODE_CHAR_SETS_HPP
#define BOOST_PARSER_UNICODE_CHAR_SETS_HPP

#include <boost/parser/parser_fwd.hpp>


namespace boost::parser::detail {

    // Produced from https://util.unicode.org/UnicodeJsps/list-unicodeset.jsp,
    // using "[:Pc:][:Pd:][:Pe:][:Pf:][:Pi:][:Ps:][:Po:]" for the Input field,
    // using the categories found at
    // https://www.fileformat.info/info/unicode/category/index.htm
    template<>
    struct char_set<punct_chars>
    {
        static constexpr uint32_t chars[] = {
            0x21,    0x22,    0x23,    0x25,    0x26,    0x27,    0x28,
            0x29,    0x2a,    0x2c,    0x2d,    0x2e,    0x2f,    0x3a,
            0x3b,    0x3f,    0x40,    0x5b,    0x5c,    0x5d,    0x5f,
            0x7b,    0x7d,    0xa1,    0xa7,    0xab,    0xb6,    0xb7,
            0xbb,    0xbf,    0x37e,   0x387,   0x55a,   0x55b,   0x55c,
            0x55d,   0x55e,   0x55f,   0x589,   0x58a,   0x5be,   0x5c0,
            0x5c3,   0x5c6,   0x5f3,   0x5f4,   0x609,   0x60a,   0x60c,
            0x60d,   0x61b,   0x61d,   0x61e,   0x61f,   0x66a,   0x66b,
            0x66c,   0x66d,   0x6d4,   0x700,   0x701,   0x702,   0x703,
            0x704,   0x705,   0x706,   0x707,   0x708,   0x709,   0x70a,
            0x70b,   0x70c,   0x70d,   0x7f7,   0x7f8,   0x7f9,   0x830,
            0x831,   0x832,   0x833,   0x834,   0x835,   0x836,   0x837,
            0x838,   0x839,   0x83a,   0x83b,   0x83c,   0x83d,   0x83e,
            0x85e,   0x964,   0x965,   0x970,   0x9fd,   0xa76,   0xaf0,
            0xc77,   0xc84,   0xdf4,   0xe4f,   0xe5a,   0xe5b,   0xf04,
            0xf05,   0xf06,   0xf07,   0xf08,   0xf09,   0xf0a,   0xf0b,
            0xf0c,   0xf0d,   0xf0e,   0xf0f,   0xf10,   0xf11,   0xf12,
            0xf14,   0xf3a,   0xf3b,   0xf3c,   0xf3d,   0xf85,   0xfd0,
            0xfd1,   0xfd2,   0xfd3,   0xfd4,   0xfd9,   0xfda,   0x104a,
            0x104b,  0x104c,  0x104d,  0x104e,  0x104f,  0x10fb,  0x1360,
            0x1361,  0x1362,  0x1363,  0x1364,  0x1365,  0x1366,  0x1367,
            0x1368,  0x1400,  0x166e,  0x169b,  0x169c,  0x16eb,  0x16ec,
            0x16ed,  0x1735,  0x1736,  0x17d4,  0x17d5,  0x17d6,  0x17d8,
            0x17d9,  0x17da,  0x1800,  0x1801,  0x1802,  0x1803,  0x1804,
            0x1805,  0x1806,  0x1807,  0x1808,  0x1809,  0x180a,  0x1944,
            0x1945,  0x1a1e,  0x1a1f,  0x1aa0,  0x1aa1,  0x1aa2,  0x1aa3,
            0x1aa4,  0x1aa5,  0x1aa6,  0x1aa8,  0x1aa9,  0x1aaa,  0x1aab,
            0x1aac,  0x1aad,  0x1b5a,  0x1b5b,  0x1b5c,  0x1b5d,  0x1b5e,
            0x1b5f,  0x1b60,  0x1b7d,  0x1b7e,  0x1bfc,  0x1bfd,  0x1bfe,
            0x1bff,  0x1c3b,  0x1c3c,  0x1c3d,  0x1c3e,  0x1c3f,  0x1c7e,
            0x1c7f,  0x1cc0,  0x1cc1,  0x1cc2,  0x1cc3,  0x1cc4,  0x1cc5,
            0x1cc6,  0x1cc7,  0x1cd3,  0x2010,  0x2011,  0x2012,  0x2013,
            0x2014,  0x2015,  0x2016,  0x2017,  0x2018,  0x2019,  0x201a,
            0x201b,  0x201c,  0x201d,  0x201e,  0x201f,  0x2020,  0x2021,
            0x2022,  0x2023,  0x2024,  0x2025,  0x2026,  0x2027,  0x2030,
            0x2031,  0x2032,  0x2033,  0x2034,  0x2035,  0x2036,  0x2037,
            0x2038,  0x2039,  0x203a,  0x203b,  0x203c,  0x203d,  0x203e,
            0x203f,  0x2040,  0x2041,  0x2042,  0x2043,  0x2045,  0x2046,
            0x2047,  0x2048,  0x2049,  0x204a,  0x204b,  0x204c,  0x204d,
            0x204e,  0x204f,  0x2050,  0x2051,  0x2053,  0x2054,  0x2055,
            0x2056,  0x2057,  0x2058,  0x2059,  0x205a,  0x205b,  0x205c,
            0x205d,  0x205e,  0x207d,  0x207e,  0x208d,  0x208e,  0x2308,
            0x2309,  0x230a,  0x230b,  0x2329,  0x232a,  0x2768,  0x2769,
            0x276a,  0x276b,  0x276c,  0x276d,  0x276e,  0x276f,  0x2770,
            0x2771,  0x2772,  0x2773,  0x2774,  0x2775,  0x27c5,  0x27c6,
            0x27e6,  0x27e7,  0x27e8,  0x27e9,  0x27ea,  0x27eb,  0x27ec,
            0x27ed,  0x27ee,  0x27ef,  0x2983,  0x2984,  0x2985,  0x2986,
            0x2987,  0x2988,  0x2989,  0x298a,  0x298b,  0x298c,  0x298d,
            0x298e,  0x298f,  0x2990,  0x2991,  0x2992,  0x2993,  0x2994,
            0x2995,  0x2996,  0x2997,  0x2998,  0x29d8,  0x29d9,  0x29da,
            0x29db,  0x29fc,  0x29fd,  0x2cf9,  0x2cfa,  0x2cfb,  0x2cfc,
            0x2cfe,  0x2cff,  0x2d70,  0x2e00,  0x2e01,  0x2e02,  0x2e03,
            0x2e04,  0x2e05,  0x2e06,  0x2e07,  0x2e08,  0x2e09,  0x2e0a,
            0x2e0b,  0x2e0c,  0x2e0d,  0x2e0e,  0x2e0f,  0x2e10,  0x2e11,
            0x2e12,  0x2e13,  0x2e14,  0x2e15,  0x2e16,  0x2e17,  0x2e18,
            0x2e19,  0x2e1a,  0x2e1b,  0x2e1c,  0x2e1d,  0x2e1e,  0x2e1f,
            0x2e20,  0x2e21,  0x2e22,  0x2e23,  0x2e24,  0x2e25,  0x2e26,
            0x2e27,  0x2e28,  0x2e29,  0x2e2a,  0x2e2b,  0x2e2c,  0x2e2d,
            0x2e2e,  0x2e30,  0x2e31,  0x2e32,  0x2e33,  0x2e34,  0x2e35,
            0x2e36,  0x2e37,  0x2e38,  0x2e39,  0x2e3a,  0x2e3b,  0x2e3c,
            0x2e3d,  0x2e3e,  0x2e3f,  0x2e40,  0x2e41,  0x2e42,  0x2e43,
            0x2e44,  0x2e45,  0x2e46,  0x2e47,  0x2e48,  0x2e49,  0x2e4a,
            0x2e4b,  0x2e4c,  0x2e4d,  0x2e4e,  0x2e4f,  0x2e52,  0x2e53,
            0x2e54,  0x2e55,  0x2e56,  0x2e57,  0x2e58,  0x2e59,  0x2e5a,
            0x2e5b,  0x2e5c,  0x2e5d,  0x3001,  0x3002,  0x3003,  0x3008,
            0x3009,  0x300a,  0x300b,  0x300c,  0x300d,  0x300e,  0x300f,
            0x3010,  0x3011,  0x3014,  0x3015,  0x3016,  0x3017,  0x3018,
            0x3019,  0x301a,  0x301b,  0x301c,  0x301d,  0x301e,  0x301f,
            0x3030,  0x303d,  0x30a0,  0x30fb,  0xa4fe,  0xa4ff,  0xa60d,
            0xa60e,  0xa60f,  0xa673,  0xa67e,  0xa6f2,  0xa6f3,  0xa6f4,
            0xa6f5,  0xa6f6,  0xa6f7,  0xa874,  0xa875,  0xa876,  0xa877,
            0xa8ce,  0xa8cf,  0xa8f8,  0xa8f9,  0xa8fa,  0xa8fc,  0xa92e,
            0xa92f,  0xa95f,  0xa9c1,  0xa9c2,  0xa9c3,  0xa9c4,  0xa9c5,
            0xa9c6,  0xa9c7,  0xa9c8,  0xa9c9,  0xa9ca,  0xa9cb,  0xa9cc,
            0xa9cd,  0xa9de,  0xa9df,  0xaa5c,  0xaa5d,  0xaa5e,  0xaa5f,
            0xaade,  0xaadf,  0xaaf0,  0xaaf1,  0xabeb,  0xfd3e,  0xfd3f,
            0xfe10,  0xfe11,  0xfe12,  0xfe13,  0xfe14,  0xfe15,  0xfe16,
            0xfe17,  0xfe18,  0xfe19,  0xfe30,  0xfe31,  0xfe32,  0xfe33,
            0xfe34,  0xfe35,  0xfe36,  0xfe37,  0xfe38,  0xfe39,  0xfe3a,
            0xfe3b,  0xfe3c,  0xfe3d,  0xfe3e,  0xfe3f,  0xfe40,  0xfe41,
            0xfe42,  0xfe43,  0xfe44,  0xfe45,  0xfe46,  0xfe47,  0xfe48,
            0xfe49,  0xfe4a,  0xfe4b,  0xfe4c,  0xfe4d,  0xfe4e,  0xfe4f,
            0xfe50,  0xfe51,  0xfe52,  0xfe54,  0xfe55,  0xfe56,  0xfe57,
            0xfe58,  0xfe59,  0xfe5a,  0xfe5b,  0xfe5c,  0xfe5d,  0xfe5e,
            0xfe5f,  0xfe60,  0xfe61,  0xfe63,  0xfe68,  0xfe6a,  0xfe6b,
            0xff01,  0xff02,  0xff03,  0xff05,  0xff06,  0xff07,  0xff08,
            0xff09,  0xff0a,  0xff0c,  0xff0d,  0xff0e,  0xff0f,  0xff1a,
            0xff1b,  0xff1f,  0xff20,  0xff3b,  0xff3c,  0xff3d,  0xff3f,
            0xff5b,  0xff5d,  0xff5f,  0xff60,  0xff61,  0xff62,  0xff63,
            0xff64,  0xff65,  0x10100, 0x10101, 0x10102, 0x1039f, 0x103d0,
            0x1056f, 0x10857, 0x1091f, 0x1093f, 0x10a50, 0x10a51, 0x10a52,
            0x10a53, 0x10a54, 0x10a55, 0x10a56, 0x10a57, 0x10a58, 0x10a7f,
            0x10af0, 0x10af1, 0x10af2, 0x10af3, 0x10af4, 0x10af5, 0x10af6,
            0x10b39, 0x10b3a, 0x10b3b, 0x10b3c, 0x10b3d, 0x10b3e, 0x10b3f,
            0x10b99, 0x10b9a, 0x10b9b, 0x10b9c, 0x10ead, 0x10f55, 0x10f56,
            0x10f57, 0x10f58, 0x10f59, 0x10f86, 0x10f87, 0x10f88, 0x10f89,
            0x11047, 0x11048, 0x11049, 0x1104a, 0x1104b, 0x1104c, 0x1104d,
            0x110bb, 0x110bc, 0x110be, 0x110bf, 0x110c0, 0x110c1, 0x11140,
            0x11141, 0x11142, 0x11143, 0x11174, 0x11175, 0x111c5, 0x111c6,
            0x111c7, 0x111c8, 0x111cd, 0x111db, 0x111dd, 0x111de, 0x111df,
            0x11238, 0x11239, 0x1123a, 0x1123b, 0x1123c, 0x1123d, 0x112a9,
            0x1144b, 0x1144c, 0x1144d, 0x1144e, 0x1144f, 0x1145a, 0x1145b,
            0x1145d, 0x114c6, 0x115c1, 0x115c2, 0x115c3, 0x115c4, 0x115c5,
            0x115c6, 0x115c7, 0x115c8, 0x115c9, 0x115ca, 0x115cb, 0x115cc,
            0x115cd, 0x115ce, 0x115cf, 0x115d0, 0x115d1, 0x115d2, 0x115d3,
            0x115d4, 0x115d5, 0x115d6, 0x115d7, 0x11641, 0x11642, 0x11643,
            0x11660, 0x11661, 0x11662, 0x11663, 0x11664, 0x11665, 0x11666,
            0x11667, 0x11668, 0x11669, 0x1166a, 0x1166b, 0x1166c, 0x116b9,
            0x1173c, 0x1173d, 0x1173e, 0x1183b, 0x11944, 0x11945, 0x11946,
            0x119e2, 0x11a3f, 0x11a40, 0x11a41, 0x11a42, 0x11a43, 0x11a44,
            0x11a45, 0x11a46, 0x11a9a, 0x11a9b, 0x11a9c, 0x11a9e, 0x11a9f,
            0x11aa0, 0x11aa1, 0x11aa2, 0x11b00, 0x11b01, 0x11b02, 0x11b03,
            0x11b04, 0x11b05, 0x11b06, 0x11b07, 0x11b08, 0x11b09, 0x11c41,
            0x11c42, 0x11c43, 0x11c44, 0x11c45, 0x11c70, 0x11c71, 0x11ef7,
            0x11ef8, 0x11f43, 0x11f44, 0x11f45, 0x11f46, 0x11f47, 0x11f48,
            0x11f49, 0x11f4a, 0x11f4b, 0x11f4c, 0x11f4d, 0x11f4e, 0x11f4f,
            0x11fff, 0x12470, 0x12471, 0x12472, 0x12473, 0x12474, 0x12ff1,
            0x12ff2, 0x16a6e, 0x16a6f, 0x16af5, 0x16b37, 0x16b38, 0x16b39,
            0x16b3a, 0x16b3b, 0x16b44, 0x16e97, 0x16e98, 0x16e99, 0x16e9a,
            0x16fe2, 0x1bc9f, 0x1da87, 0x1da88, 0x1da89, 0x1da8a, 0x1da8b,
            0x1e95e, 0x1e95f};
    };

    // Produced from https://util.unicode.org/UnicodeJsps/list-unicodeset.jsp,
    // using "[:Sc:][:Sk:][:Sm:][:So:]" for the Input field,
    // using the categories found at
    // https://www.fileformat.info/info/unicode/category/index.htm
    template<>
    struct char_set<symb_chars>
    {
        static constexpr uint32_t chars[] = {
	    0x24, 0x2B, 0x3C, 0x3D, 0x3E, 0x5E, 0x60, 0x7C, 0x7E, 0xA2, 0xA3,
	    0xA4, 0xA5, 0xA6, 0xA8, 0xA9, 0xAC, 0xAE, 0xAF, 0xB0, 0xB1, 0xB4,
	    0xB8, 0xD7, 0xF7, 0x2C2, 0x2C3, 0x2C4, 0x2C5, 0x2D2, 0x2D3, 0x2D4,
	    0x2D5, 0x2D6, 0x2D7, 0x2D8, 0x2D9, 0x2DA, 0x2DB, 0x2DC, 0x2DD,
	    0x2DE, 0x2DF, 0x2E5, 0x2E6, 0x2E7, 0x2E8, 0x2E9, 0x2EA, 0x2EB,
	    0x2ED, 0x2EF, 0x2F0, 0x2F1, 0x2F2, 0x2F3, 0x2F4, 0x2F5, 0x2F6,
	    0x2F7, 0x2F8, 0x2F9, 0x2FA, 0x2FB, 0x2FC, 0x2FD, 0x2FE, 0x2FF,
	    0x375, 0x384, 0x385, 0x3F6, 0x482, 0x58D, 0x58E, 0x58F, 0x606,
	    0x607, 0x608, 0x60B, 0x60E, 0x60F, 0x6DE, 0x6E9, 0x6FD, 0x6FE,
	    0x7F6, 0x7FE, 0x7FF, 0x888, 0x9F2, 0x9F3, 0x9FA, 0x9FB, 0xAF1,
	    0xB70, 0xBF3, 0xBF4, 0xBF5, 0xBF6, 0xBF7, 0xBF8, 0xBF9, 0xBFA,
	    0xC7F, 0xD4F, 0xD79, 0xE3F, 0xF01, 0xF02, 0xF03, 0xF13, 0xF15,
	    0xF16, 0xF17, 0xF1A, 0xF1B, 0xF1C, 0xF1D, 0xF1E, 0xF1F, 0xF34,
	    0xF36, 0xF38, 0xFBE, 0xFBF, 0xFC0, 0xFC1, 0xFC2, 0xFC3, 0xFC4,
	    0xFC5, 0xFC7, 0xFC8, 0xFC9, 0xFCA, 0xFCB, 0xFCC, 0xFCE, 0xFCF,
	    0xFD5, 0xFD6, 0xFD7, 0xFD8, 0x109E, 0x109F, 0x1390, 0x1391, 0x1392,
	    0x1393, 0x1394, 0x1395, 0x1396, 0x1397, 0x1398, 0x1399, 0x166D,
	    0x17DB, 0x1940, 0x19DE, 0x19DF, 0x19E0, 0x19E1, 0x19E2, 0x19E3,
	    0x19E4, 0x19E5, 0x19E6, 0x19E7, 0x19E8, 0x19E9, 0x19EA, 0x19EB,
	    0x19EC, 0x19ED, 0x19EE, 0x19EF, 0x19F0, 0x19F1, 0x19F2, 0x19F3,
	    0x19F4, 0x19F5, 0x19F6, 0x19F7, 0x19F8, 0x19F9, 0x19FA, 0x19FB,
	    0x19FC, 0x19FD, 0x19FE, 0x19FF, 0x1B61, 0x1B62, 0x1B63, 0x1B64,
	    0x1B65, 0x1B66, 0x1B67, 0x1B68, 0x1B69, 0x1B6A, 0x1B74, 0x1B75,
	    0x1B76, 0x1B77, 0x1B78, 0x1B79, 0x1B7A, 0x1B7B, 0x1B7C, 0x1FBD,
	    0x1FBF, 0x1FC0, 0x1FC1, 0x1FCD, 0x1FCE, 0x1FCF, 0x1FDD, 0x1FDE,
	    0x1FDF, 0x1FED, 0x1FEE, 0x1FEF, 0x1FFD, 0x1FFE, 0x2044, 0x2052,
	    0x207A, 0x207B, 0x207C, 0x208A, 0x208B, 0x208C, 0x20A0, 0x20A1,
	    0x20A2, 0x20A3, 0x20A4, 0x20A5, 0x20A6, 0x20A7, 0x20A8, 0x20A9,
	    0x20AA, 0x20AB, 0x20AC, 0x20AD, 0x20AE, 0x20AF, 0x20B0, 0x20B1,
	    0x20B2, 0x20B3, 0x20B4, 0x20B5, 0x20B6, 0x20B7, 0x20B8, 0x20B9,
	    0x20BA, 0x20BB, 0x20BC, 0x20BD, 0x20BE, 0x20BF, 0x20C0, 0x2100,
	    0x2101, 0x2103, 0x2104, 0x2105, 0x2106, 0x2108, 0x2109, 0x2114,
	    0x2116, 0x2117, 0x2118, 0x211E, 0x211F, 0x2120, 0x2121, 0x2122,
	    0x2123, 0x2125, 0x2127, 0x2129, 0x212E, 0x213A, 0x213B, 0x2140,
	    0x2141, 0x2142, 0x2143, 0x2144, 0x214A, 0x214B, 0x214C, 0x214D,
	    0x214F, 0x218A, 0x218B, 0x2190, 0x2191, 0x2192, 0x2193, 0x2194,
	    0x2195, 0x2196, 0x2197, 0x2198, 0x2199, 0x219A, 0x219B, 0x219C,
	    0x219D, 0x219E, 0x219F, 0x21A0, 0x21A1, 0x21A2, 0x21A3, 0x21A4,
	    0x21A5, 0x21A6, 0x21A7, 0x21A8, 0x21A9, 0x21AA, 0x21AB, 0x21AC,
	    0x21AD, 0x21AE, 0x21AF, 0x21B0, 0x21B1, 0x21B2, 0x21B3, 0x21B4,
	    0x21B5, 0x21B6, 0x21B7, 0x21B8, 0x21B9, 0x21BA, 0x21BB, 0x21BC,
	    0x21BD, 0x21BE, 0x21BF, 0x21C0, 0x21C1, 0x21C2, 0x21C3, 0x21C4,
	    0x21C5, 0x21C6, 0x21C7, 0x21C8, 0x21C9, 0x21CA, 0x21CB, 0x21CC,
	    0x21CD, 0x21CE, 0x21CF, 0x21D0, 0x21D1, 0x21D2, 0x21D3, 0x21D4,
	    0x21D5, 0x21D6, 0x21D7, 0x21D8, 0x21D9, 0x21DA, 0x21DB, 0x21DC,
	    0x21DD, 0x21DE, 0x21DF, 0x21E0, 0x21E1, 0x21E2, 0x21E3, 0x21E4,
	    0x21E5, 0x21E6, 0x21E7, 0x21E8, 0x21E9, 0x21EA, 0x21EB, 0x21EC,
	    0x21ED, 0x21EE, 0x21EF, 0x21F0, 0x21F1, 0x21F2, 0x21F3, 0x21F4,
	    0x21F5, 0x21F6, 0x21F7, 0x21F8, 0x21F9, 0x21FA, 0x21FB, 0x21FC,
	    0x21FD, 0x21FE, 0x21FF, 0x2200, 0x2201, 0x2202, 0x2203, 0x2204,
	    0x2205, 0x2206, 0x2207, 0x2208, 0x2209, 0x220A, 0x220B, 0x220C,
	    0x220D, 0x220E, 0x220F, 0x2210, 0x2211, 0x2212, 0x2213, 0x2214,
	    0x2215, 0x2216, 0x2217, 0x2218, 0x2219, 0x221A, 0x221B, 0x221C,
	    0x221D, 0x221E, 0x221F, 0x2220, 0x2221, 0x2222, 0x2223, 0x2224,
	    0x2225, 0x2226, 0x2227, 0x2228, 0x2229, 0x222A, 0x222B, 0x222C,
	    0x222D, 0x222E, 0x222F, 0x2230, 0x2231, 0x2232, 0x2233, 0x2234,
	    0x2235, 0x2236, 0x2237, 0x2238, 0x2239, 0x223A, 0x223B, 0x223C,
	    0x223D, 0x223E, 0x223F, 0x2240, 0x2241, 0x2242, 0x2243, 0x2244,
	    0x2245, 0x2246, 0x2247, 0x2248, 0x2249, 0x224A, 0x224B, 0x224C,
	    0x224D, 0x224E, 0x224F, 0x2250, 0x2251, 0x2252, 0x2253, 0x2254,
	    0x2255, 0x2256, 0x2257, 0x2258, 0x2259, 0x225A, 0x225B, 0x225C,
	    0x225D, 0x225E, 0x225F, 0x2260, 0x2261, 0x2262, 0x2263, 0x2264,
	    0x2265, 0x2266, 0x2267, 0x2268, 0x2269, 0x226A, 0x226B, 0x226C,
	    0x226D, 0x226E, 0x226F, 0x2270, 0x2271, 0x2272, 0x2273, 0x2274,
	    0x2275, 0x2276, 0x2277, 0x2278, 0x2279, 0x227A, 0x227B, 0x227C,
	    0x227D, 0x227E, 0x227F, 0x2280, 0x2281, 0x2282, 0x2283, 0x2284,
	    0x2285, 0x2286, 0x2287, 0x2288, 0x2289, 0x228A, 0x228B, 0x228C,
	    0x228D, 0x228E, 0x228F, 0x2290, 0x2291, 0x2292, 0x2293, 0x2294,
	    0x2295, 0x2296, 0x2297, 0x2298, 0x2299, 0x229A, 0x229B, 0x229C,
	    0x229D, 0x229E, 0x229F, 0x22A0, 0x22A1, 0x22A2, 0x22A3, 0x22A4,
	    0x22A5, 0x22A6, 0x22A7, 0x22A8, 0x22A9, 0x22AA, 0x22AB, 0x22AC,
	    0x22AD, 0x22AE, 0x22AF, 0x22B0, 0x22B1, 0x22B2, 0x22B3, 0x22B4,
	    0x22B5, 0x22B6, 0x22B7, 0x22B8, 0x22B9, 0x22BA, 0x22BB, 0x22BC,
	    0x22BD, 0x22BE, 0x22BF, 0x22C0, 0x22C1, 0x22C2, 0x22C3, 0x22C4,
	    0x22C5, 0x22C6, 0x22C7, 0x22C8, 0x22C9, 0x22CA, 0x22CB, 0x22CC,
	    0x22CD, 0x22CE, 0x22CF, 0x22D0, 0x22D1, 0x22D2, 0x22D3, 0x22D4,
	    0x22D5, 0x22D6, 0x22D7, 0x22D8, 0x22D9, 0x22DA, 0x22DB, 0x22DC,
	    0x22DD, 0x22DE, 0x22DF, 0x22E0, 0x22E1, 0x22E2, 0x22E3, 0x22E4,
	    0x22E5, 0x22E6, 0x22E7, 0x22E8, 0x22E9, 0x22EA, 0x22EB, 0x22EC,
	    0x22ED, 0x22EE, 0x22EF, 0x22F0, 0x22F1, 0x22F2, 0x22F3, 0x22F4,
	    0x22F5, 0x22F6, 0x22F7, 0x22F8, 0x22F9, 0x22FA, 0x22FB, 0x22FC,
	    0x22FD, 0x22FE, 0x22FF, 0x2300, 0x2301, 0x2302, 0x2303, 0x2304,
	    0x2305, 0x2306, 0x2307, 0x230C, 0x230D, 0x230E, 0x230F, 0x2310,
	    0x2311, 0x2312, 0x2313, 0x2314, 0x2315, 0x2316, 0x2317, 0x2318,
	    0x2319, 0x231A, 0x231B, 0x231C, 0x231D, 0x231E, 0x231F, 0x2320,
	    0x2321, 0x2322, 0x2323, 0x2324, 0x2325, 0x2326, 0x2327, 0x2328,
	    0x232B, 0x232C, 0x232D, 0x232E, 0x232F, 0x2330, 0x2331, 0x2332,
	    0x2333, 0x2334, 0x2335, 0x2336, 0x2337, 0x2338, 0x2339, 0x233A,
	    0x233B, 0x233C, 0x233D, 0x233E, 0x233F, 0x2340, 0x2341, 0x2342,
	    0x2343, 0x2344, 0x2345, 0x2346, 0x2347, 0x2348, 0x2349, 0x234A,
	    0x234B, 0x234C, 0x234D, 0x234E, 0x234F, 0x2350, 0x2351, 0x2352,
	    0x2353, 0x2354, 0x2355, 0x2356, 0x2357, 0x2358, 0x2359, 0x235A,
	    0x235B, 0x235C, 0x235D, 0x235E, 0x235F, 0x2360, 0x2361, 0x2362,
	    0x2363, 0x2364, 0x2365, 0x2366, 0x2367, 0x2368, 0x2369, 0x236A,
	    0x236B, 0x236C, 0x236D, 0x236E, 0x236F, 0x2370, 0x2371, 0x2372,
	    0x2373, 0x2374, 0x2375, 0x2376, 0x2377, 0x2378, 0x2379, 0x237A,
	    0x237B, 0x237C, 0x237D, 0x237E, 0x237F, 0x2380, 0x2381, 0x2382,
	    0x2383, 0x2384, 0x2385, 0x2386, 0x2387, 0x2388, 0x2389, 0x238A,
	    0x238B, 0x238C, 0x238D, 0x238E, 0x238F, 0x2390, 0x2391, 0x2392,
	    0x2393, 0x2394, 0x2395, 0x2396, 0x2397, 0x2398, 0x2399, 0x239A,
	    0x239B, 0x239C, 0x239D, 0x239E, 0x239F, 0x23A0, 0x23A1, 0x23A2,
	    0x23A3, 0x23A4, 0x23A5, 0x23A6, 0x23A7, 0x23A8, 0x23A9, 0x23AA,
	    0x23AB, 0x23AC, 0x23AD, 0x23AE, 0x23AF, 0x23B0, 0x23B1, 0x23B2,
	    0x23B3, 0x23B4, 0x23B5, 0x23B6, 0x23B7, 0x23B8, 0x23B9, 0x23BA,
	    0x23BB, 0x23BC, 0x23BD, 0x23BE, 0x23BF, 0x23C0, 0x23C1, 0x23C2,
	    0x23C3, 0x23C4, 0x23C5, 0x23C6, 0x23C7, 0x23C8, 0x23C9, 0x23CA,
	    0x23CB, 0x23CC, 0x23CD, 0x23CE, 0x23CF, 0x23D0, 0x23D1, 0x23D2,
	    0x23D3, 0x23D4, 0x23D5, 0x23D6, 0x23D7, 0x23D8, 0x23D9, 0x23DA,
	    0x23DB, 0x23DC, 0x23DD, 0x23DE, 0x23DF, 0x23E0, 0x23E1, 0x23E2,
	    0x23E3, 0x23E4, 0x23E5, 0x23E6, 0x23E7, 0x23E8, 0x23E9, 0x23EA,
	    0x23EB, 0x23EC, 0x23ED, 0x23EE, 0x23EF, 0x23F0, 0x23F1, 0x23F2,
	    0x23F3, 0x23F4, 0x23F5, 0x23F6, 0x23F7, 0x23F8, 0x23F9, 0x23FA,
	    0x23FB, 0x23FC, 0x23FD, 0x23FE, 0x23FF, 0x2400, 0x2401, 0x2402,
	    0x2403, 0x2404, 0x2405, 0x2406, 0x2407, 0x2408, 0x2409, 0x240A,
	    0x240B, 0x240C, 0x240D, 0x240E, 0x240F, 0x2410, 0x2411, 0x2412,
	    0x2413, 0x2414, 0x2415, 0x2416, 0x2417, 0x2418, 0x2419, 0x241A,
	    0x241B, 0x241C, 0x241D, 0x241E, 0x241F, 0x2420, 0x2421, 0x2422,
	    0x2423, 0x2424, 0x2425, 0x2426, 0x2427, 0x2428, 0x2429, 0x2440,
	    0x2441, 0x2442, 0x2443, 0x2444, 0x2445, 0x2446, 0x2447, 0x2448,
	    0x2449, 0x244A, 0x249C, 0x249D, 0x249E, 0x249F, 0x24A0, 0x24A1,
	    0x24A2, 0x24A3, 0x24A4, 0x24A5, 0x24A6, 0x24A7, 0x24A8, 0x24A9,
	    0x24AA, 0x24AB, 0x24AC, 0x24AD, 0x24AE, 0x24AF, 0x24B0, 0x24B1,
	    0x24B2, 0x24B3, 0x24B4, 0x24B5, 0x24B6, 0x24B7, 0x24B8, 0x24B9,
	    0x24BA, 0x24BB, 0x24BC, 0x24BD, 0x24BE, 0x24BF, 0x24C0, 0x24C1,
	    0x24C2, 0x24C3, 0x24C4, 0x24C5, 0x24C6, 0x24C7, 0x24C8, 0x24C9,
	    0x24CA, 0x24CB, 0x24CC, 0x24CD, 0x24CE, 0x24CF, 0x24D0, 0x24D1,
	    0x24D2, 0x24D3, 0x24D4, 0x24D5, 0x24D6, 0x24D7, 0x24D8, 0x24D9,
	    0x24DA, 0x24DB, 0x24DC, 0x24DD, 0x24DE, 0x24DF, 0x24E0, 0x24E1,
	    0x24E2, 0x24E3, 0x24E4, 0x24E5, 0x24E6, 0x24E7, 0x24E8, 0x24E9,
	    0x2500, 0x2501, 0x2502, 0x2503, 0x2504, 0x2505, 0x2506, 0x2507,
	    0x2508, 0x2509, 0x250A, 0x250B, 0x250C, 0x250D, 0x250E, 0x250F,
	    0x2510, 0x2511, 0x2512, 0x2513, 0x2514, 0x2515, 0x2516, 0x2517,
	    0x2518, 0x2519, 0x251A, 0x251B, 0x251C, 0x251D, 0x251E, 0x251F,
	    0x2520, 0x2521, 0x2522, 0x2523, 0x2524, 0x2525, 0x2526, 0x2527,
	    0x2528, 0x2529, 0x252A, 0x252B, 0x252C, 0x252D, 0x252E, 0x252F,
	    0x2530, 0x2531, 0x2532, 0x2533, 0x2534, 0x2535, 0x2536, 0x2537,
	    0x2538, 0x2539, 0x253A, 0x253B, 0x253C, 0x253D, 0x253E, 0x253F,
	    0x2540, 0x2541, 0x2542, 0x2543, 0x2544, 0x2545, 0x2546, 0x2547,
	    0x2548, 0x2549, 0x254A, 0x254B, 0x254C, 0x254D, 0x254E, 0x254F,
	    0x2550, 0x2551, 0x2552, 0x2553, 0x2554, 0x2555, 0x2556, 0x2557,
	    0x2558, 0x2559, 0x255A, 0x255B, 0x255C, 0x255D, 0x255E, 0x255F,
	    0x2560, 0x2561, 0x2562, 0x2563, 0x2564, 0x2565, 0x2566, 0x2567,
	    0x2568, 0x2569, 0x256A, 0x256B, 0x256C, 0x256D, 0x256E, 0x256F,
	    0x2570, 0x2571, 0x2572, 0x2573, 0x2574, 0x2575, 0x2576, 0x2577,
	    0x2578, 0x2579, 0x257A, 0x257B, 0x257C, 0x257D, 0x257E, 0x257F,
	    0x2580, 0x2581, 0x2582, 0x2583, 0x2584, 0x2585, 0x2586, 0x2587,
	    0x2588, 0x2589, 0x258A, 0x258B, 0x258C, 0x258D, 0x258E, 0x258F,
	    0x2590, 0x2591, 0x2592, 0x2593, 0x2594, 0x2595, 0x2596, 0x2597,
	    0x2598, 0x2599, 0x259A, 0x259B, 0x259C, 0x259D, 0x259E, 0x259F,
	    0x25A0, 0x25A1, 0x25A2, 0x25A3, 0x25A4, 0x25A5, 0x25A6, 0x25A7,
	    0x25A8, 0x25A9, 0x25AA, 0x25AB, 0x25AC, 0x25AD, 0x25AE, 0x25AF,
	    0x25B0, 0x25B1, 0x25B2, 0x25B3, 0x25B4, 0x25B5, 0x25B6, 0x25B7,
	    0x25B8, 0x25B9, 0x25BA, 0x25BB, 0x25BC, 0x25BD, 0x25BE, 0x25BF,
	    0x25C0, 0x25C1, 0x25C2, 0x25C3, 0x25C4, 0x25C5, 0x25C6, 0x25C7,
	    0x25C8, 0x25C9, 0x25CA, 0x25CB, 0x25CC, 0x25CD, 0x25CE, 0x25CF,
	    0x25D0, 0x25D1, 0x25D2, 0x25D3, 0x25D4, 0x25D5, 0x25D6, 0x25D7,
	    0x25D8, 0x25D9, 0x25DA, 0x25DB, 0x25DC, 0x25DD, 0x25DE, 0x25DF,
	    0x25E0, 0x25E1, 0x25E2, 0x25E3, 0x25E4, 0x25E5, 0x25E6, 0x25E7,
	    0x25E8, 0x25E9, 0x25EA, 0x25EB, 0x25EC, 0x25ED, 0x25EE, 0x25EF,
	    0x25F0, 0x25F1, 0x25F2, 0x25F3, 0x25F4, 0x25F5, 0x25F6, 0x25F7,
	    0x25F8, 0x25F9, 0x25FA, 0x25FB, 0x25FC, 0x25FD, 0x25FE, 0x25FF,
	    0x2600, 0x2601, 0x2602, 0x2603, 0x2604, 0x2605, 0x2606, 0x2607,
	    0x2608, 0x2609, 0x260A, 0x260B, 0x260C, 0x260D, 0x260E, 0x260F,
	    0x2610, 0x2611, 0x2612, 0x2613, 0x2614, 0x2615, 0x2616, 0x2617,
	    0x2618, 0x2619, 0x261A, 0x261B, 0x261C, 0x261D, 0x261E, 0x261F,
	    0x2620, 0x2621, 0x2622, 0x2623, 0x2624, 0x2625, 0x2626, 0x2627,
	    0x2628, 0x2629, 0x262A, 0x262B, 0x262C, 0x262D, 0x262E, 0x262F,
	    0x2630, 0x2631, 0x2632, 0x2633, 0x2634, 0x2635, 0x2636, 0x2637,
	    0x2638, 0x2639, 0x263A, 0x263B, 0x263C, 0x263D, 0x263E, 0x263F,
	    0x2640, 0x2641, 0x2642, 0x2643, 0x2644, 0x2645, 0x2646, 0x2647,
	    0x2648, 0x2649, 0x264A, 0x264B, 0x264C, 0x264D, 0x264E, 0x264F,
	    0x2650, 0x2651, 0x2652, 0x2653, 0x2654, 0x2655, 0x2656, 0x2657,
	    0x2658, 0x2659, 0x265A, 0x265B, 0x265C, 0x265D, 0x265E, 0x265F,
	    0x2660, 0x2661, 0x2662, 0x2663, 0x2664, 0x2665, 0x2666, 0x2667,
	    0x2668, 0x2669, 0x266A, 0x266B, 0x266C, 0x266D, 0x266E, 0x266F,
	    0x2670, 0x2671, 0x2672, 0x2673, 0x2674, 0x2675, 0x2676, 0x2677,
	    0x2678, 0x2679, 0x267A, 0x267B, 0x267C, 0x267D, 0x267E, 0x267F,
	    0x2680, 0x2681, 0x2682, 0x2683, 0x2684, 0x2685, 0x2686, 0x2687,
	    0x2688, 0x2689, 0x268A, 0x268B, 0x268C, 0x268D, 0x268E, 0x268F,
	    0x2690, 0x2691, 0x2692, 0x2693, 0x2694, 0x2695, 0x2696, 0x2697,
	    0x2698, 0x2699, 0x269A, 0x269B, 0x269C, 0x269D, 0x269E, 0x269F,
	    0x26A0, 0x26A1, 0x26A2, 0x26A3, 0x26A4, 0x26A5, 0x26A6, 0x26A7,
	    0x26A8, 0x26A9, 0x26AA, 0x26AB, 0x26AC, 0x26AD, 0x26AE, 0x26AF,
	    0x26B0, 0x26B1, 0x26B2, 0x26B3, 0x26B4, 0x26B5, 0x26B6, 0x26B7,
	    0x26B8, 0x26B9, 0x26BA, 0x26BB, 0x26BC, 0x26BD, 0x26BE, 0x26BF,
	    0x26C0, 0x26C1, 0x26C2, 0x26C3, 0x26C4, 0x26C5, 0x26C6, 0x26C7,
	    0x26C8, 0x26C9, 0x26CA, 0x26CB, 0x26CC, 0x26CD, 0x26CE, 0x26CF,
	    0x26D0, 0x26D1, 0x26D2, 0x26D3, 0x26D4, 0x26D5, 0x26D6, 0x26D7,
	    0x26D8, 0x26D9, 0x26DA, 0x26DB, 0x26DC, 0x26DD, 0x26DE, 0x26DF,
	    0x26E0, 0x26E1, 0x26E2, 0x26E3, 0x26E4, 0x26E5, 0x26E6, 0x26E7,
	    0x26E8, 0x26E9, 0x26EA, 0x26EB, 0x26EC, 0x26ED, 0x26EE, 0x26EF,
	    0x26F0, 0x26F1, 0x26F2, 0x26F3, 0x26F4, 0x26F5, 0x26F6, 0x26F7,
	    0x26F8, 0x26F9, 0x26FA, 0x26FB, 0x26FC, 0x26FD, 0x26FE, 0x26FF,
	    0x2700, 0x2701, 0x2702, 0x2703, 0x2704, 0x2705, 0x2706, 0x2707,
	    0x2708, 0x2709, 0x270A, 0x270B, 0x270C, 0x270D, 0x270E, 0x270F,
	    0x2710, 0x2711, 0x2712, 0x2713, 0x2714, 0x2715, 0x2716, 0x2717,
	    0x2718, 0x2719, 0x271A, 0x271B, 0x271C, 0x271D, 0x271E, 0x271F,
	    0x2720, 0x2721, 0x2722, 0x2723, 0x2724, 0x2725, 0x2726, 0x2727,
	    0x2728, 0x2729, 0x272A, 0x272B, 0x272C, 0x272D, 0x272E, 0x272F,
	    0x2730, 0x2731, 0x2732, 0x2733, 0x2734, 0x2735, 0x2736, 0x2737,
	    0x2738, 0x2739, 0x273A, 0x273B, 0x273C, 0x273D, 0x273E, 0x273F,
	    0x2740, 0x2741, 0x2742, 0x2743, 0x2744, 0x2745, 0x2746, 0x2747,
	    0x2748, 0x2749, 0x274A, 0x274B, 0x274C, 0x274D, 0x274E, 0x274F,
	    0x2750, 0x2751, 0x2752, 0x2753, 0x2754, 0x2755, 0x2756, 0x2757,
	    0x2758, 0x2759, 0x275A, 0x275B, 0x275C, 0x275D, 0x275E, 0x275F,
	    0x2760, 0x2761, 0x2762, 0x2763, 0x2764, 0x2765, 0x2766, 0x2767,
	    0x2794, 0x2795, 0x2796, 0x2797, 0x2798, 0x2799, 0x279A, 0x279B,
	    0x279C, 0x279D, 0x279E, 0x279F, 0x27A0, 0x27A1, 0x27A2, 0x27A3,
	    0x27A4, 0x27A5, 0x27A6, 0x27A7, 0x27A8, 0x27A9, 0x27AA, 0x27AB,
	    0x27AC, 0x27AD, 0x27AE, 0x27AF, 0x27B0, 0x27B1, 0x27B2, 0x27B3,
	    0x27B4, 0x27B5, 0x27B6, 0x27B7, 0x27B8, 0x27B9, 0x27BA, 0x27BB,
	    0x27BC, 0x27BD, 0x27BE, 0x27BF, 0x27C0, 0x27C1, 0x27C2, 0x27C3,
	    0x27C4, 0x27C7, 0x27C8, 0x27C9, 0x27CA, 0x27CB, 0x27CC, 0x27CD,
	    0x27CE, 0x27CF, 0x27D0, 0x27D1, 0x27D2, 0x27D3, 0x27D4, 0x27D5,
	    0x27D6, 0x27D7, 0x27D8, 0x27D9, 0x27DA, 0x27DB, 0x27DC, 0x27DD,
	    0x27DE, 0x27DF, 0x27E0, 0x27E1, 0x27E2, 0x27E3, 0x27E4, 0x27E5,
	    0x27F0, 0x27F1, 0x27F2, 0x27F3, 0x27F4, 0x27F5, 0x27F6, 0x27F7,
	    0x27F8, 0x27F9, 0x27FA, 0x27FB, 0x27FC, 0x27FD, 0x27FE, 0x27FF,
	    0x2800, 0x2801, 0x2802, 0x2803, 0x2804, 0x2805, 0x2806, 0x2807,
	    0x2808, 0x2809, 0x280A, 0x280B, 0x280C, 0x280D, 0x280E, 0x280F,
	    0x2810, 0x2811, 0x2812, 0x2813, 0x2814, 0x2815, 0x2816, 0x2817,
	    0x2818, 0x2819, 0x281A, 0x281B, 0x281C, 0x281D, 0x281E, 0x281F,
	    0x2820, 0x2821, 0x2822, 0x2823, 0x2824, 0x2825, 0x2826, 0x2827,
	    0x2828, 0x2829, 0x282A, 0x282B, 0x282C, 0x282D, 0x282E, 0x282F,
	    0x2830, 0x2831, 0x2832, 0x2833, 0x2834, 0x2835, 0x2836, 0x2837,
	    0x2838, 0x2839, 0x283A, 0x283B, 0x283C, 0x283D, 0x283E, 0x283F,
	    0x2840, 0x2841, 0x2842, 0x2843, 0x2844, 0x2845, 0x2846, 0x2847,
	    0x2848, 0x2849, 0x284A, 0x284B, 0x284C, 0x284D, 0x284E, 0x284F,
	    0x2850, 0x2851, 0x2852, 0x2853, 0x2854, 0x2855, 0x2856, 0x2857,
	    0x2858, 0x2859, 0x285A, 0x285B, 0x285C, 0x285D, 0x285E, 0x285F,
	    0x2860, 0x2861, 0x2862, 0x2863, 0x2864, 0x2865, 0x2866, 0x2867,
	    0x2868, 0x2869, 0x286A, 0x286B, 0x286C, 0x286D, 0x286E, 0x286F,
	    0x2870, 0x2871, 0x2872, 0x2873, 0x2874, 0x2875, 0x2876, 0x2877,
	    0x2878, 0x2879, 0x287A, 0x287B, 0x287C, 0x287D, 0x287E, 0x287F,
	    0x2880, 0x2881, 0x2882, 0x2883, 0x2884, 0x2885, 0x2886, 0x2887,
	    0x2888, 0x2889, 0x288A, 0x288B, 0x288C, 0x288D, 0x288E, 0x288F,
	    0x2890, 0x2891, 0x2892, 0x2893, 0x2894, 0x2895, 0x2896, 0x2897,
	    0x2898, 0x2899, 0x289A, 0x289B, 0x289C, 0x289D, 0x289E, 0x289F,
	    0x28A0, 0x28A1, 0x28A2, 0x28A3, 0x28A4, 0x28A5, 0x28A6, 0x28A7,
	    0x28A8, 0x28A9, 0x28AA, 0x28AB, 0x28AC, 0x28AD, 0x28AE, 0x28AF,
	    0x28B0, 0x28B1, 0x28B2, 0x28B3, 0x28B4, 0x28B5, 0x28B6, 0x28B7,
	    0x28B8, 0x28B9, 0x28BA, 0x28BB, 0x28BC, 0x28BD, 0x28BE, 0x28BF,
	    0x28C0, 0x28C1, 0x28C2, 0x28C3, 0x28C4, 0x28C5, 0x28C6, 0x28C7,
	    0x28C8, 0x28C9, 0x28CA, 0x28CB, 0x28CC, 0x28CD, 0x28CE, 0x28CF,
	    0x28D0, 0x28D1, 0x28D2, 0x28D3, 0x28D4, 0x28D5, 0x28D6, 0x28D7,
	    0x28D8, 0x28D9, 0x28DA, 0x28DB, 0x28DC, 0x28DD, 0x28DE, 0x28DF,
	    0x28E0, 0x28E1, 0x28E2, 0x28E3, 0x28E4, 0x28E5, 0x28E6, 0x28E7,
	    0x28E8, 0x28E9, 0x28EA, 0x28EB, 0x28EC, 0x28ED, 0x28EE, 0x28EF,
	    0x28F0, 0x28F1, 0x28F2, 0x28F3, 0x28F4, 0x28F5, 0x28F6, 0x28F7,
	    0x28F8, 0x28F9, 0x28FA, 0x28FB, 0x28FC, 0x28FD, 0x28FE, 0x28FF,
	    0x2900, 0x2901, 0x2902, 0x2903, 0x2904, 0x2905, 0x2906, 0x2907,
	    0x2908, 0x2909, 0x290A, 0x290B, 0x290C, 0x290D, 0x290E, 0x290F,
	    0x2910, 0x2911, 0x2912, 0x2913, 0x2914, 0x2915, 0x2916, 0x2917,
	    0x2918, 0x2919, 0x291A, 0x291B, 0x291C, 0x291D, 0x291E, 0x291F,
	    0x2920, 0x2921, 0x2922, 0x2923, 0x2924, 0x2925, 0x2926, 0x2927,
	    0x2928, 0x2929, 0x292A, 0x292B, 0x292C, 0x292D, 0x292E, 0x292F,
	    0x2930, 0x2931, 0x2932, 0x2933, 0x2934, 0x2935, 0x2936, 0x2937,
	    0x2938, 0x2939, 0x293A, 0x293B, 0x293C, 0x293D, 0x293E, 0x293F,
	    0x2940, 0x2941, 0x2942, 0x2943, 0x2944, 0x2945, 0x2946, 0x2947,
	    0x2948, 0x2949, 0x294A, 0x294B, 0x294C, 0x294D, 0x294E, 0x294F,
	    0x2950, 0x2951, 0x2952, 0x2953, 0x2954, 0x2955, 0x2956, 0x2957,
	    0x2958, 0x2959, 0x295A, 0x295B, 0x295C, 0x295D, 0x295E, 0x295F,
	    0x2960, 0x2961, 0x2962, 0x2963, 0x2964, 0x2965, 0x2966, 0x2967,
	    0x2968, 0x2969, 0x296A, 0x296B, 0x296C, 0x296D, 0x296E, 0x296F,
	    0x2970, 0x2971, 0x2972, 0x2973, 0x2974, 0x2975, 0x2976, 0x2977,
	    0x2978, 0x2979, 0x297A, 0x297B, 0x297C, 0x297D, 0x297E, 0x297F,
	    0x2980, 0x2981, 0x2982, 0x2999, 0x299A, 0x299B, 0x299C, 0x299D,
	    0x299E, 0x299F, 0x29A0, 0x29A1, 0x29A2, 0x29A3, 0x29A4, 0x29A5,
	    0x29A6, 0x29A7, 0x29A8, 0x29A9, 0x29AA, 0x29AB, 0x29AC, 0x29AD,
	    0x29AE, 0x29AF, 0x29B0, 0x29B1, 0x29B2, 0x29B3, 0x29B4, 0x29B5,
	    0x29B6, 0x29B7, 0x29B8, 0x29B9, 0x29BA, 0x29BB, 0x29BC, 0x29BD,
	    0x29BE, 0x29BF, 0x29C0, 0x29C1, 0x29C2, 0x29C3, 0x29C4, 0x29C5,
	    0x29C6, 0x29C7, 0x29C8, 0x29C9, 0x29CA, 0x29CB, 0x29CC, 0x29CD,
	    0x29CE, 0x29CF, 0x29D0, 0x29D1, 0x29D2, 0x29D3, 0x29D4, 0x29D5,
	    0x29D6, 0x29D7, 0x29DC, 0x29DD, 0x29DE, 0x29DF, 0x29E0, 0x29E1,
	    0x29E2, 0x29E3, 0x29E4, 0x29E5, 0x29E6, 0x29E7, 0x29E8, 0x29E9,
	    0x29EA, 0x29EB, 0x29EC, 0x29ED, 0x29EE, 0x29EF, 0x29F0, 0x29F1,
	    0x29F2, 0x29F3, 0x29F4, 0x29F5, 0x29F6, 0x29F7, 0x29F8, 0x29F9,
	    0x29FA, 0x29FB, 0x29FE, 0x29FF, 0x2A00, 0x2A01, 0x2A02, 0x2A03,
	    0x2A04, 0x2A05, 0x2A06, 0x2A07, 0x2A08, 0x2A09, 0x2A0A, 0x2A0B,
	    0x2A0C, 0x2A0D, 0x2A0E, 0x2A0F, 0x2A10, 0x2A11, 0x2A12, 0x2A13,
	    0x2A14, 0x2A15, 0x2A16, 0x2A17, 0x2A18, 0x2A19, 0x2A1A, 0x2A1B,
	    0x2A1C, 0x2A1D, 0x2A1E, 0x2A1F, 0x2A20, 0x2A21, 0x2A22, 0x2A23,
	    0x2A24, 0x2A25, 0x2A26, 0x2A27, 0x2A28, 0x2A29, 0x2A2A, 0x2A2B,
	    0x2A2C, 0x2A2D, 0x2A2E, 0x2A2F, 0x2A30, 0x2A31, 0x2A32, 0x2A33,
	    0x2A34, 0x2A35, 0x2A36, 0x2A37, 0x2A38, 0x2A39, 0x2A3A, 0x2A3B,
	    0x2A3C, 0x2A3D, 0x2A3E, 0x2A3F, 0x2A40, 0x2A41, 0x2A42, 0x2A43,
	    0x2A44, 0x2A45, 0x2A46, 0x2A47, 0x2A48, 0x2A49, 0x2A4A, 0x2A4B,
	    0x2A4C, 0x2A4D, 0x2A4E, 0x2A4F, 0x2A50, 0x2A51, 0x2A52, 0x2A53,
	    0x2A54, 0x2A55, 0x2A56, 0x2A57, 0x2A58, 0x2A59, 0x2A5A, 0x2A5B,
	    0x2A5C, 0x2A5D, 0x2A5E, 0x2A5F, 0x2A60, 0x2A61, 0x2A62, 0x2A63,
	    0x2A64, 0x2A65, 0x2A66, 0x2A67, 0x2A68, 0x2A69, 0x2A6A, 0x2A6B,
	    0x2A6C, 0x2A6D, 0x2A6E, 0x2A6F, 0x2A70, 0x2A71, 0x2A72, 0x2A73,
	    0x2A74, 0x2A75, 0x2A76, 0x2A77, 0x2A78, 0x2A79, 0x2A7A, 0x2A7B,
	    0x2A7C, 0x2A7D, 0x2A7E, 0x2A7F, 0x2A80, 0x2A81, 0x2A82, 0x2A83,
	    0x2A84, 0x2A85, 0x2A86, 0x2A87, 0x2A88, 0x2A89, 0x2A8A, 0x2A8B,
	    0x2A8C, 0x2A8D, 0x2A8E, 0x2A8F, 0x2A90, 0x2A91, 0x2A92, 0x2A93,
	    0x2A94, 0x2A95, 0x2A96, 0x2A97, 0x2A98, 0x2A99, 0x2A9A, 0x2A9B,
	    0x2A9C, 0x2A9D, 0x2A9E, 0x2A9F, 0x2AA0, 0x2AA1, 0x2AA2, 0x2AA3,
	    0x2AA4, 0x2AA5, 0x2AA6, 0x2AA7, 0x2AA8, 0x2AA9, 0x2AAA, 0x2AAB,
	    0x2AAC, 0x2AAD, 0x2AAE, 0x2AAF, 0x2AB0, 0x2AB1, 0x2AB2, 0x2AB3,
	    0x2AB4, 0x2AB5, 0x2AB6, 0x2AB7, 0x2AB8, 0x2AB9, 0x2ABA, 0x2ABB,
	    0x2ABC, 0x2ABD, 0x2ABE, 0x2ABF, 0x2AC0, 0x2AC1, 0x2AC2, 0x2AC3,
	    0x2AC4, 0x2AC5, 0x2AC6, 0x2AC7, 0x2AC8, 0x2AC9, 0x2ACA, 0x2ACB,
	    0x2ACC, 0x2ACD, 0x2ACE, 0x2ACF, 0x2AD0, 0x2AD1, 0x2AD2, 0x2AD3,
	    0x2AD4, 0x2AD5, 0x2AD6, 0x2AD7, 0x2AD8, 0x2AD9, 0x2ADA, 0x2ADB,
	    0x2ADC, 0x2ADD, 0x2ADE, 0x2ADF, 0x2AE0, 0x2AE1, 0x2AE2, 0x2AE3,
	    0x2AE4, 0x2AE5, 0x2AE6, 0x2AE7, 0x2AE8, 0x2AE9, 0x2AEA, 0x2AEB,
	    0x2AEC, 0x2AED, 0x2AEE, 0x2AEF, 0x2AF0, 0x2AF1, 0x2AF2, 0x2AF3,
	    0x2AF4, 0x2AF5, 0x2AF6, 0x2AF7, 0x2AF8, 0x2AF9, 0x2AFA, 0x2AFB,
	    0x2AFC, 0x2AFD, 0x2AFE, 0x2AFF, 0x2B00, 0x2B01, 0x2B02, 0x2B03,
	    0x2B04, 0x2B05, 0x2B06, 0x2B07, 0x2B08, 0x2B09, 0x2B0A, 0x2B0B,
	    0x2B0C, 0x2B0D, 0x2B0E, 0x2B0F, 0x2B10, 0x2B11, 0x2B12, 0x2B13,
	    0x2B14, 0x2B15, 0x2B16, 0x2B17, 0x2B18, 0x2B19, 0x2B1A, 0x2B1B,
	    0x2B1C, 0x2B1D, 0x2B1E, 0x2B1F, 0x2B20, 0x2B21, 0x2B22, 0x2B23,
	    0x2B24, 0x2B25, 0x2B26, 0x2B27, 0x2B28, 0x2B29, 0x2B2A, 0x2B2B,
	    0x2B2C, 0x2B2D, 0x2B2E, 0x2B2F, 0x2B30, 0x2B31, 0x2B32, 0x2B33,
	    0x2B34, 0x2B35, 0x2B36, 0x2B37, 0x2B38, 0x2B39, 0x2B3A, 0x2B3B,
	    0x2B3C, 0x2B3D, 0x2B3E, 0x2B3F, 0x2B40, 0x2B41, 0x2B42, 0x2B43,
	    0x2B44, 0x2B45, 0x2B46, 0x2B47, 0x2B48, 0x2B49, 0x2B4A, 0x2B4B,
	    0x2B4C, 0x2B4D, 0x2B4E, 0x2B4F, 0x2B50, 0x2B51, 0x2B52, 0x2B53,
	    0x2B54, 0x2B55, 0x2B56, 0x2B57, 0x2B58, 0x2B59, 0x2B5A, 0x2B5B,
	    0x2B5C, 0x2B5D, 0x2B5E, 0x2B5F, 0x2B60, 0x2B61, 0x2B62, 0x2B63,
	    0x2B64, 0x2B65, 0x2B66, 0x2B67, 0x2B68, 0x2B69, 0x2B6A, 0x2B6B,
	    0x2B6C, 0x2B6D, 0x2B6E, 0x2B6F, 0x2B70, 0x2B71, 0x2B72, 0x2B73,
	    0x2B76, 0x2B77, 0x2B78, 0x2B79, 0x2B7A, 0x2B7B, 0x2B7C, 0x2B7D,
	    0x2B7E, 0x2B7F, 0x2B80, 0x2B81, 0x2B82, 0x2B83, 0x2B84, 0x2B85,
	    0x2B86, 0x2B87, 0x2B88, 0x2B89, 0x2B8A, 0x2B8B, 0x2B8C, 0x2B8D,
	    0x2B8E, 0x2B8F, 0x2B90, 0x2B91, 0x2B92, 0x2B93, 0x2B94, 0x2B95,
	    0x2B97, 0x2B98, 0x2B99, 0x2B9A, 0x2B9B, 0x2B9C, 0x2B9D, 0x2B9E,
	    0x2B9F, 0x2BA0, 0x2BA1, 0x2BA2, 0x2BA3, 0x2BA4, 0x2BA5, 0x2BA6,
	    0x2BA7, 0x2BA8, 0x2BA9, 0x2BAA, 0x2BAB, 0x2BAC, 0x2BAD, 0x2BAE,
	    0x2BAF, 0x2BB0, 0x2BB1, 0x2BB2, 0x2BB3, 0x2BB4, 0x2BB5, 0x2BB6,
	    0x2BB7, 0x2BB8, 0x2BB9, 0x2BBA, 0x2BBB, 0x2BBC, 0x2BBD, 0x2BBE,
	    0x2BBF, 0x2BC0, 0x2BC1, 0x2BC2, 0x2BC3, 0x2BC4, 0x2BC5, 0x2BC6,
	    0x2BC7, 0x2BC8, 0x2BC9, 0x2BCA, 0x2BCB, 0x2BCC, 0x2BCD, 0x2BCE,
	    0x2BCF, 0x2BD0, 0x2BD1, 0x2BD2, 0x2BD3, 0x2BD4, 0x2BD5, 0x2BD6,
	    0x2BD7, 0x2BD8, 0x2BD9, 0x2BDA, 0x2BDB, 0x2BDC, 0x2BDD, 0x2BDE,
	    0x2BDF, 0x2BE0, 0x2BE1, 0x2BE2, 0x2BE3, 0x2BE4, 0x2BE5, 0x2BE6,
	    0x2BE7, 0x2BE8, 0x2BE9, 0x2BEA, 0x2BEB, 0x2BEC, 0x2BED, 0x2BEE,
	    0x2BEF, 0x2BF0, 0x2BF1, 0x2BF2, 0x2BF3, 0x2BF4, 0x2BF5, 0x2BF6,
	    0x2BF7, 0x2BF8, 0x2BF9, 0x2BFA, 0x2BFB, 0x2BFC, 0x2BFD, 0x2BFE,
	    0x2BFF, 0x2CE5, 0x2CE6, 0x2CE7, 0x2CE8, 0x2CE9, 0x2CEA, 0x2E50,
	    0x2E51, 0x2E80, 0x2E81, 0x2E82, 0x2E83, 0x2E84, 0x2E85, 0x2E86,
	    0x2E87, 0x2E88, 0x2E89, 0x2E8A, 0x2E8B, 0x2E8C, 0x2E8D, 0x2E8E,
	    0x2E8F, 0x2E90, 0x2E91, 0x2E92, 0x2E93, 0x2E94, 0x2E95, 0x2E96,
	    0x2E97, 0x2E98, 0x2E99, 0x2E9B, 0x2E9C, 0x2E9D, 0x2E9E, 0x2E9F,
	    0x2EA0, 0x2EA1, 0x2EA2, 0x2EA3, 0x2EA4, 0x2EA5, 0x2EA6, 0x2EA7,
	    0x2EA8, 0x2EA9, 0x2EAA, 0x2EAB, 0x2EAC, 0x2EAD, 0x2EAE, 0x2EAF,
	    0x2EB0, 0x2EB1, 0x2EB2, 0x2EB3, 0x2EB4, 0x2EB5, 0x2EB6, 0x2EB7,
	    0x2EB8, 0x2EB9, 0x2EBA, 0x2EBB, 0x2EBC, 0x2EBD, 0x2EBE, 0x2EBF,
	    0x2EC0, 0x2EC1, 0x2EC2, 0x2EC3, 0x2EC4, 0x2EC5, 0x2EC6, 0x2EC7,
	    0x2EC8, 0x2EC9, 0x2ECA, 0x2ECB, 0x2ECC, 0x2ECD, 0x2ECE, 0x2ECF,
	    0x2ED0, 0x2ED1, 0x2ED2, 0x2ED3, 0x2ED4, 0x2ED5, 0x2ED6, 0x2ED7,
	    0x2ED8, 0x2ED9, 0x2EDA, 0x2EDB, 0x2EDC, 0x2EDD, 0x2EDE, 0x2EDF,
	    0x2EE0, 0x2EE1, 0x2EE2, 0x2EE3, 0x2EE4, 0x2EE5, 0x2EE6, 0x2EE7,
	    0x2EE8, 0x2EE9, 0x2EEA, 0x2EEB, 0x2EEC, 0x2EED, 0x2EEE, 0x2EEF,
	    0x2EF0, 0x2EF1, 0x2EF2, 0x2EF3, 0x2F00, 0x2F01, 0x2F02, 0x2F03,
	    0x2F04, 0x2F05, 0x2F06, 0x2F07, 0x2F08, 0x2F09, 0x2F0A, 0x2F0B,
	    0x2F0C, 0x2F0D, 0x2F0E, 0x2F0F, 0x2F10, 0x2F11, 0x2F12, 0x2F13,
	    0x2F14, 0x2F15, 0x2F16, 0x2F17, 0x2F18, 0x2F19, 0x2F1A, 0x2F1B,
	    0x2F1C, 0x2F1D, 0x2F1E, 0x2F1F, 0x2F20, 0x2F21, 0x2F22, 0x2F23,
	    0x2F24, 0x2F25, 0x2F26, 0x2F27, 0x2F28, 0x2F29, 0x2F2A, 0x2F2B,
	    0x2F2C, 0x2F2D, 0x2F2E, 0x2F2F, 0x2F30, 0x2F31, 0x2F32, 0x2F33,
	    0x2F34, 0x2F35, 0x2F36, 0x2F37, 0x2F38, 0x2F39, 0x2F3A, 0x2F3B,
	    0x2F3C, 0x2F3D, 0x2F3E, 0x2F3F, 0x2F40, 0x2F41, 0x2F42, 0x2F43,
	    0x2F44, 0x2F45, 0x2F46, 0x2F47, 0x2F48, 0x2F49, 0x2F4A, 0x2F4B,
	    0x2F4C, 0x2F4D, 0x2F4E, 0x2F4F, 0x2F50, 0x2F51, 0x2F52, 0x2F53,
	    0x2F54, 0x2F55, 0x2F56, 0x2F57, 0x2F58, 0x2F59, 0x2F5A, 0x2F5B,
	    0x2F5C, 0x2F5D, 0x2F5E, 0x2F5F, 0x2F60, 0x2F61, 0x2F62, 0x2F63,
	    0x2F64, 0x2F65, 0x2F66, 0x2F67, 0x2F68, 0x2F69, 0x2F6A, 0x2F6B,
	    0x2F6C, 0x2F6D, 0x2F6E, 0x2F6F, 0x2F70, 0x2F71, 0x2F72, 0x2F73,
	    0x2F74, 0x2F75, 0x2F76, 0x2F77, 0x2F78, 0x2F79, 0x2F7A, 0x2F7B,
	    0x2F7C, 0x2F7D, 0x2F7E, 0x2F7F, 0x2F80, 0x2F81, 0x2F82, 0x2F83,
	    0x2F84, 0x2F85, 0x2F86, 0x2F87, 0x2F88, 0x2F89, 0x2F8A, 0x2F8B,
	    0x2F8C, 0x2F8D, 0x2F8E, 0x2F8F, 0x2F90, 0x2F91, 0x2F92, 0x2F93,
	    0x2F94, 0x2F95, 0x2F96, 0x2F97, 0x2F98, 0x2F99, 0x2F9A, 0x2F9B,
	    0x2F9C, 0x2F9D, 0x2F9E, 0x2F9F, 0x2FA0, 0x2FA1, 0x2FA2, 0x2FA3,
	    0x2FA4, 0x2FA5, 0x2FA6, 0x2FA7, 0x2FA8, 0x2FA9, 0x2FAA, 0x2FAB,
	    0x2FAC, 0x2FAD, 0x2FAE, 0x2FAF, 0x2FB0, 0x2FB1, 0x2FB2, 0x2FB3,
	    0x2FB4, 0x2FB5, 0x2FB6, 0x2FB7, 0x2FB8, 0x2FB9, 0x2FBA, 0x2FBB,
	    0x2FBC, 0x2FBD, 0x2FBE, 0x2FBF, 0x2FC0, 0x2FC1, 0x2FC2, 0x2FC3,
	    0x2FC4, 0x2FC5, 0x2FC6, 0x2FC7, 0x2FC8, 0x2FC9, 0x2FCA, 0x2FCB,
	    0x2FCC, 0x2FCD, 0x2FCE, 0x2FCF, 0x2FD0, 0x2FD1, 0x2FD2, 0x2FD3,
	    0x2FD4, 0x2FD5, 0x2FF0, 0x2FF1, 0x2FF2, 0x2FF3, 0x2FF4, 0x2FF5,
	    0x2FF6, 0x2FF7, 0x2FF8, 0x2FF9, 0x2FFA, 0x2FFB, 0x2FFC, 0x2FFD,
	    0x2FFE, 0x2FFF, 0x3004, 0x3012, 0x3013, 0x3020, 0x3036, 0x3037,
	    0x303E, 0x303F, 0x309B, 0x309C, 0x3190, 0x3191, 0x3196, 0x3197,
	    0x3198, 0x3199, 0x319A, 0x319B, 0x319C, 0x319D, 0x319E, 0x319F,
	    0x31C0, 0x31C1, 0x31C2, 0x31C3, 0x31C4, 0x31C5, 0x31C6, 0x31C7,
	    0x31C8, 0x31C9, 0x31CA, 0x31CB, 0x31CC, 0x31CD, 0x31CE, 0x31CF,
	    0x31D0, 0x31D1, 0x31D2, 0x31D3, 0x31D4, 0x31D5, 0x31D6, 0x31D7,
	    0x31D8, 0x31D9, 0x31DA, 0x31DB, 0x31DC, 0x31DD, 0x31DE, 0x31DF,
	    0x31E0, 0x31E1, 0x31E2, 0x31E3, 0x31E4, 0x31E5, 0x31EF, 0x3200,
	    0x3201, 0x3202, 0x3203, 0x3204, 0x3205, 0x3206, 0x3207, 0x3208,
	    0x3209, 0x320A, 0x320B, 0x320C, 0x320D, 0x320E, 0x320F, 0x3210,
	    0x3211, 0x3212, 0x3213, 0x3214, 0x3215, 0x3216, 0x3217, 0x3218,
	    0x3219, 0x321A, 0x321B, 0x321C, 0x321D, 0x321E, 0x322A, 0x322B,
	    0x322C, 0x322D, 0x322E, 0x322F, 0x3230, 0x3231, 0x3232, 0x3233,
	    0x3234, 0x3235, 0x3236, 0x3237, 0x3238, 0x3239, 0x323A, 0x323B,
	    0x323C, 0x323D, 0x323E, 0x323F, 0x3240, 0x3241, 0x3242, 0x3243,
	    0x3244, 0x3245, 0x3246, 0x3247, 0x3250, 0x3260, 0x3261, 0x3262,
	    0x3263, 0x3264, 0x3265, 0x3266, 0x3267, 0x3268, 0x3269, 0x326A,
	    0x326B, 0x326C, 0x326D, 0x326E, 0x326F, 0x3270, 0x3271, 0x3272,
	    0x3273, 0x3274, 0x3275, 0x3276, 0x3277, 0x3278, 0x3279, 0x327A,
	    0x327B, 0x327C, 0x327D, 0x327E, 0x327F, 0x328A, 0x328B, 0x328C,
	    0x328D, 0x328E, 0x328F, 0x3290, 0x3291, 0x3292, 0x3293, 0x3294,
	    0x3295, 0x3296, 0x3297, 0x3298, 0x3299, 0x329A, 0x329B, 0x329C,
	    0x329D, 0x329E, 0x329F, 0x32A0, 0x32A1, 0x32A2, 0x32A3, 0x32A4,
	    0x32A5, 0x32A6, 0x32A7, 0x32A8, 0x32A9, 0x32AA, 0x32AB, 0x32AC,
	    0x32AD, 0x32AE, 0x32AF, 0x32B0, 0x32C0, 0x32C1, 0x32C2, 0x32C3,
	    0x32C4, 0x32C5, 0x32C6, 0x32C7, 0x32C8, 0x32C9, 0x32CA, 0x32CB,
	    0x32CC, 0x32CD, 0x32CE, 0x32CF, 0x32D0, 0x32D1, 0x32D2, 0x32D3,
	    0x32D4, 0x32D5, 0x32D6, 0x32D7, 0x32D8, 0x32D9, 0x32DA, 0x32DB,
	    0x32DC, 0x32DD, 0x32DE, 0x32DF, 0x32E0, 0x32E1, 0x32E2, 0x32E3,
	    0x32E4, 0x32E5, 0x32E6, 0x32E7, 0x32E8, 0x32E9, 0x32EA, 0x32EB,
	    0x32EC, 0x32ED, 0x32EE, 0x32EF, 0x32F0, 0x32F1, 0x32F2, 0x32F3,
	    0x32F4, 0x32F5, 0x32F6, 0x32F7, 0x32F8, 0x32F9, 0x32FA, 0x32FB,
	    0x32FC, 0x32FD, 0x32FE, 0x32FF, 0x3300, 0x3301, 0x3302, 0x3303,
	    0x3304, 0x3305, 0x3306, 0x3307, 0x3308, 0x3309, 0x330A, 0x330B,
	    0x330C, 0x330D, 0x330E, 0x330F, 0x3310, 0x3311, 0x3312, 0x3313,
	    0x3314, 0x3315, 0x3316, 0x3317, 0x3318, 0x3319, 0x331A, 0x331B,
	    0x331C, 0x331D, 0x331E, 0x331F, 0x3320, 0x3321, 0x3322, 0x3323,
	    0x3324, 0x3325, 0x3326, 0x3327, 0x3328, 0x3329, 0x332A, 0x332B,
	    0x332C, 0x332D, 0x332E, 0x332F, 0x3330, 0x3331, 0x3332, 0x3333,
	    0x3334, 0x3335, 0x3336, 0x3337, 0x3338, 0x3339, 0x333A, 0x333B,
	    0x333C, 0x333D, 0x333E, 0x333F, 0x3340, 0x3341, 0x3342, 0x3343,
	    0x3344, 0x3345, 0x3346, 0x3347, 0x3348, 0x3349, 0x334A, 0x334B,
	    0x334C, 0x334D, 0x334E, 0x334F, 0x3350, 0x3351, 0x3352, 0x3353,
	    0x3354, 0x3355, 0x3356, 0x3357, 0x3358, 0x3359, 0x335A, 0x335B,
	    0x335C, 0x335D, 0x335E, 0x335F, 0x3360, 0x3361, 0x3362, 0x3363,
	    0x3364, 0x3365, 0x3366, 0x3367, 0x3368, 0x3369, 0x336A, 0x336B,
	    0x336C, 0x336D, 0x336E, 0x336F, 0x3370, 0x3371, 0x3372, 0x3373,
	    0x3374, 0x3375, 0x3376, 0x3377, 0x3378, 0x3379, 0x337A, 0x337B,
	    0x337C, 0x337D, 0x337E, 0x337F, 0x3380, 0x3381, 0x3382, 0x3383,
	    0x3384, 0x3385, 0x3386, 0x3387, 0x3388, 0x3389, 0x338A, 0x338B,
	    0x338C, 0x338D, 0x338E, 0x338F, 0x3390, 0x3391, 0x3392, 0x3393,
	    0x3394, 0x3395, 0x3396, 0x3397, 0x3398, 0x3399, 0x339A, 0x339B,
	    0x339C, 0x339D, 0x339E, 0x339F, 0x33A0, 0x33A1, 0x33A2, 0x33A3,
	    0x33A4, 0x33A5, 0x33A6, 0x33A7, 0x33A8, 0x33A9, 0x33AA, 0x33AB,
	    0x33AC, 0x33AD, 0x33AE, 0x33AF, 0x33B0, 0x33B1, 0x33B2, 0x33B3,
	    0x33B4, 0x33B5, 0x33B6, 0x33B7, 0x33B8, 0x33B9, 0x33BA, 0x33BB,
	    0x33BC, 0x33BD, 0x33BE, 0x33BF, 0x33C0, 0x33C1, 0x33C2, 0x33C3,
	    0x33C4, 0x33C5, 0x33C6, 0x33C7, 0x33C8, 0x33C9, 0x33CA, 0x33CB,
	    0x33CC, 0x33CD, 0x33CE, 0x33CF, 0x33D0, 0x33D1, 0x33D2, 0x33D3,
	    0x33D4, 0x33D5, 0x33D6, 0x33D7, 0x33D8, 0x33D9, 0x33DA, 0x33DB,
	    0x33DC, 0x33DD, 0x33DE, 0x33DF, 0x33E0, 0x33E1, 0x33E2, 0x33E3,
	    0x33E4, 0x33E5, 0x33E6, 0x33E7, 0x33E8, 0x33E9, 0x33EA, 0x33EB,
	    0x33EC, 0x33ED, 0x33EE, 0x33EF, 0x33F0, 0x33F1, 0x33F2, 0x33F3,
	    0x33F4, 0x33F5, 0x33F6, 0x33F7, 0x33F8, 0x33F9, 0x33FA, 0x33FB,
	    0x33FC, 0x33FD, 0x33FE, 0x33FF, 0x4DC0, 0x4DC1, 0x4DC2, 0x4DC3,
	    0x4DC4, 0x4DC5, 0x4DC6, 0x4DC7, 0x4DC8, 0x4DC9, 0x4DCA, 0x4DCB,
	    0x4DCC, 0x4DCD, 0x4DCE, 0x4DCF, 0x4DD0, 0x4DD1, 0x4DD2, 0x4DD3,
	    0x4DD4, 0x4DD5, 0x4DD6, 0x4DD7, 0x4DD8, 0x4DD9, 0x4DDA, 0x4DDB,
	    0x4DDC, 0x4DDD, 0x4DDE, 0x4DDF, 0x4DE0, 0x4DE1, 0x4DE2, 0x4DE3,
	    0x4DE4, 0x4DE5, 0x4DE6, 0x4DE7, 0x4DE8, 0x4DE9, 0x4DEA, 0x4DEB,
	    0x4DEC, 0x4DED, 0x4DEE, 0x4DEF, 0x4DF0, 0x4DF1, 0x4DF2, 0x4DF3,
	    0x4DF4, 0x4DF5, 0x4DF6, 0x4DF7, 0x4DF8, 0x4DF9, 0x4DFA, 0x4DFB,
	    0x4DFC, 0x4DFD, 0x4DFE, 0x4DFF, 0xA490, 0xA491, 0xA492, 0xA493,
	    0xA494, 0xA495, 0xA496, 0xA497, 0xA498, 0xA499, 0xA49A, 0xA49B,
	    0xA49C, 0xA49D, 0xA49E, 0xA49F, 0xA4A0, 0xA4A1, 0xA4A2, 0xA4A3,
	    0xA4A4, 0xA4A5, 0xA4A6, 0xA4A7, 0xA4A8, 0xA4A9, 0xA4AA, 0xA4AB,
	    0xA4AC, 0xA4AD, 0xA4AE, 0xA4AF, 0xA4B0, 0xA4B1, 0xA4B2, 0xA4B3,
	    0xA4B4, 0xA4B5, 0xA4B6, 0xA4B7, 0xA4B8, 0xA4B9, 0xA4BA, 0xA4BB,
	    0xA4BC, 0xA4BD, 0xA4BE, 0xA4BF, 0xA4C0, 0xA4C1, 0xA4C2, 0xA4C3,
	    0xA4C4, 0xA4C5, 0xA4C6, 0xA700, 0xA701, 0xA702, 0xA703, 0xA704,
	    0xA705, 0xA706, 0xA707, 0xA708, 0xA709, 0xA70A, 0xA70B, 0xA70C,
	    0xA70D, 0xA70E, 0xA70F, 0xA710, 0xA711, 0xA712, 0xA713, 0xA714,
	    0xA715, 0xA716, 0xA720, 0xA721, 0xA789, 0xA78A, 0xA828, 0xA829,
	    0xA82A, 0xA82B, 0xA836, 0xA837, 0xA838, 0xA839, 0xAA77, 0xAA78,
	    0xAA79, 0xAB5B, 0xAB6A, 0xAB6B, 0xFB29, 0xFBB2, 0xFBB3, 0xFBB4,
	    0xFBB5, 0xFBB6, 0xFBB7, 0xFBB8, 0xFBB9, 0xFBBA, 0xFBBB, 0xFBBC,
	    0xFBBD, 0xFBBE, 0xFBBF, 0xFBC0, 0xFBC1, 0xFBC2, 0xFD40, 0xFD41,
	    0xFD42, 0xFD43, 0xFD44, 0xFD45, 0xFD46, 0xFD47, 0xFD48, 0xFD49,
	    0xFD4A, 0xFD4B, 0xFD4C, 0xFD4D, 0xFD4E, 0xFD4F, 0xFDCF, 0xFDFC,
	    0xFDFD, 0xFDFE, 0xFDFF, 0xFE62, 0xFE64, 0xFE65, 0xFE66, 0xFE69,
	    0xFF04, 0xFF0B, 0xFF1C, 0xFF1D, 0xFF1E, 0xFF3E, 0xFF40, 0xFF5C,
	    0xFF5E, 0xFFE0, 0xFFE1, 0xFFE2, 0xFFE3, 0xFFE4, 0xFFE5, 0xFFE6,
	    0xFFE8, 0xFFE9, 0xFFEA, 0xFFEB, 0xFFEC, 0xFFED, 0xFFEE, 0xFFFC,
	    0xFFFD, 0x10137, 0x10138, 0x10139, 0x1013A, 0x1013B, 0x1013C,
	    0x1013D, 0x1013E, 0x1013F, 0x10179, 0x1017A, 0x1017B, 0x1017C,
	    0x1017D, 0x1017E, 0x1017F, 0x10180, 0x10181, 0x10182, 0x10183,
	    0x10184, 0x10185, 0x10186, 0x10187, 0x10188, 0x10189, 0x1018C,
	    0x1018D, 0x1018E, 0x10190, 0x10191, 0x10192, 0x10193, 0x10194,
	    0x10195, 0x10196, 0x10197, 0x10198, 0x10199, 0x1019A, 0x1019B,
	    0x1019C, 0x101A0, 0x101D0, 0x101D1, 0x101D2, 0x101D3, 0x101D4,
	    0x101D5, 0x101D6, 0x101D7, 0x101D8, 0x101D9, 0x101DA, 0x101DB,
	    0x101DC, 0x101DD, 0x101DE, 0x101DF, 0x101E0, 0x101E1, 0x101E2,
	    0x101E3, 0x101E4, 0x101E5, 0x101E6, 0x101E7, 0x101E8, 0x101E9,
	    0x101EA, 0x101EB, 0x101EC, 0x101ED, 0x101EE, 0x101EF, 0x101F0,
	    0x101F1, 0x101F2, 0x101F3, 0x101F4, 0x101F5, 0x101F6, 0x101F7,
	    0x101F8, 0x101F9, 0x101FA, 0x101FB, 0x101FC, 0x10877, 0x10878,
	    0x10AC8, 0x10D8E, 0x10D8F, 0x1173F, 0x11FD5, 0x11FD6, 0x11FD7,
	    0x11FD8, 0x11FD9, 0x11FDA, 0x11FDB, 0x11FDC, 0x11FDD, 0x11FDE,
	    0x11FDF, 0x11FE0, 0x11FE1, 0x11FE2, 0x11FE3, 0x11FE4, 0x11FE5,
	    0x11FE6, 0x11FE7, 0x11FE8, 0x11FE9, 0x11FEA, 0x11FEB, 0x11FEC,
	    0x11FED, 0x11FEE, 0x11FEF, 0x11FF0, 0x11FF1, 0x16B3C, 0x16B3D,
	    0x16B3E, 0x16B3F, 0x16B45, 0x1BC9C, 0x1CC00, 0x1CC01, 0x1CC02,
	    0x1CC03, 0x1CC04, 0x1CC05, 0x1CC06, 0x1CC07, 0x1CC08, 0x1CC09,
	    0x1CC0A, 0x1CC0B, 0x1CC0C, 0x1CC0D, 0x1CC0E, 0x1CC0F, 0x1CC10,
	    0x1CC11, 0x1CC12, 0x1CC13, 0x1CC14, 0x1CC15, 0x1CC16, 0x1CC17,
	    0x1CC18, 0x1CC19, 0x1CC1A, 0x1CC1B, 0x1CC1C, 0x1CC1D, 0x1CC1E,
	    0x1CC1F, 0x1CC20, 0x1CC21, 0x1CC22, 0x1CC23, 0x1CC24, 0x1CC25,
	    0x1CC26, 0x1CC27, 0x1CC28, 0x1CC29, 0x1CC2A, 0x1CC2B, 0x1CC2C,
	    0x1CC2D, 0x1CC2E, 0x1CC2F, 0x1CC30, 0x1CC31, 0x1CC32, 0x1CC33,
	    0x1CC34, 0x1CC35, 0x1CC36, 0x1CC37, 0x1CC38, 0x1CC39, 0x1CC3A,
	    0x1CC3B, 0x1CC3C, 0x1CC3D, 0x1CC3E, 0x1CC3F, 0x1CC40, 0x1CC41,
	    0x1CC42, 0x1CC43, 0x1CC44, 0x1CC45, 0x1CC46, 0x1CC47, 0x1CC48,
	    0x1CC49, 0x1CC4A, 0x1CC4B, 0x1CC4C, 0x1CC4D, 0x1CC4E, 0x1CC4F,
	    0x1CC50, 0x1CC51, 0x1CC52, 0x1CC53, 0x1CC54, 0x1CC55, 0x1CC56,
	    0x1CC57, 0x1CC58, 0x1CC59, 0x1CC5A, 0x1CC5B, 0x1CC5C, 0x1CC5D,
	    0x1CC5E, 0x1CC5F, 0x1CC60, 0x1CC61, 0x1CC62, 0x1CC63, 0x1CC64,
	    0x1CC65, 0x1CC66, 0x1CC67, 0x1CC68, 0x1CC69, 0x1CC6A, 0x1CC6B,
	    0x1CC6C, 0x1CC6D, 0x1CC6E, 0x1CC6F, 0x1CC70, 0x1CC71, 0x1CC72,
	    0x1CC73, 0x1CC74, 0x1CC75, 0x1CC76, 0x1CC77, 0x1CC78, 0x1CC79,
	    0x1CC7A, 0x1CC7B, 0x1CC7C, 0x1CC7D, 0x1CC7E, 0x1CC7F, 0x1CC80,
	    0x1CC81, 0x1CC82, 0x1CC83, 0x1CC84, 0x1CC85, 0x1CC86, 0x1CC87,
	    0x1CC88, 0x1CC89, 0x1CC8A, 0x1CC8B, 0x1CC8C, 0x1CC8D, 0x1CC8E,
	    0x1CC8F, 0x1CC90, 0x1CC91, 0x1CC92, 0x1CC93, 0x1CC94, 0x1CC95,
	    0x1CC96, 0x1CC97, 0x1CC98, 0x1CC99, 0x1CC9A, 0x1CC9B, 0x1CC9C,
	    0x1CC9D, 0x1CC9E, 0x1CC9F, 0x1CCA0, 0x1CCA1, 0x1CCA2, 0x1CCA3,
	    0x1CCA4, 0x1CCA5, 0x1CCA6, 0x1CCA7, 0x1CCA8, 0x1CCA9, 0x1CCAA,
	    0x1CCAB, 0x1CCAC, 0x1CCAD, 0x1CCAE, 0x1CCAF, 0x1CCB0, 0x1CCB1,
	    0x1CCB2, 0x1CCB3, 0x1CCB4, 0x1CCB5, 0x1CCB6, 0x1CCB7, 0x1CCB8,
	    0x1CCB9, 0x1CCBA, 0x1CCBB, 0x1CCBC, 0x1CCBD, 0x1CCBE, 0x1CCBF,
	    0x1CCC0, 0x1CCC1, 0x1CCC2, 0x1CCC3, 0x1CCC4, 0x1CCC5, 0x1CCC6,
	    0x1CCC7, 0x1CCC8, 0x1CCC9, 0x1CCCA, 0x1CCCB, 0x1CCCC, 0x1CCCD,
	    0x1CCCE, 0x1CCCF, 0x1CCD0, 0x1CCD1, 0x1CCD2, 0x1CCD3, 0x1CCD4,
	    0x1CCD5, 0x1CCD6, 0x1CCD7, 0x1CCD8, 0x1CCD9, 0x1CCDA, 0x1CCDB,
	    0x1CCDC, 0x1CCDD, 0x1CCDE, 0x1CCDF, 0x1CCE0, 0x1CCE1, 0x1CCE2,
	    0x1CCE3, 0x1CCE4, 0x1CCE5, 0x1CCE6, 0x1CCE7, 0x1CCE8, 0x1CCE9,
	    0x1CCEA, 0x1CCEB, 0x1CCEC, 0x1CCED, 0x1CCEE, 0x1CCEF, 0x1CD00,
	    0x1CD01, 0x1CD02, 0x1CD03, 0x1CD04, 0x1CD05, 0x1CD06, 0x1CD07,
	    0x1CD08, 0x1CD09, 0x1CD0A, 0x1CD0B, 0x1CD0C, 0x1CD0D, 0x1CD0E,
	    0x1CD0F, 0x1CD10, 0x1CD11, 0x1CD12, 0x1CD13, 0x1CD14, 0x1CD15,
	    0x1CD16, 0x1CD17, 0x1CD18, 0x1CD19, 0x1CD1A, 0x1CD1B, 0x1CD1C,
	    0x1CD1D, 0x1CD1E, 0x1CD1F, 0x1CD20, 0x1CD21, 0x1CD22, 0x1CD23,
	    0x1CD24, 0x1CD25, 0x1CD26, 0x1CD27, 0x1CD28, 0x1CD29, 0x1CD2A,
	    0x1CD2B, 0x1CD2C, 0x1CD2D, 0x1CD2E, 0x1CD2F, 0x1CD30, 0x1CD31,
	    0x1CD32, 0x1CD33, 0x1CD34, 0x1CD35, 0x1CD36, 0x1CD37, 0x1CD38,
	    0x1CD39, 0x1CD3A, 0x1CD3B, 0x1CD3C, 0x1CD3D, 0x1CD3E, 0x1CD3F,
	    0x1CD40, 0x1CD41, 0x1CD42, 0x1CD43, 0x1CD44, 0x1CD45, 0x1CD46,
	    0x1CD47, 0x1CD48, 0x1CD49, 0x1CD4A, 0x1CD4B, 0x1CD4C, 0x1CD4D,
	    0x1CD4E, 0x1CD4F, 0x1CD50, 0x1CD51, 0x1CD52, 0x1CD53, 0x1CD54,
	    0x1CD55, 0x1CD56, 0x1CD57, 0x1CD58, 0x1CD59, 0x1CD5A, 0x1CD5B,
	    0x1CD5C, 0x1CD5D, 0x1CD5E, 0x1CD5F, 0x1CD60, 0x1CD61, 0x1CD62,
	    0x1CD63, 0x1CD64, 0x1CD65, 0x1CD66, 0x1CD67, 0x1CD68, 0x1CD69,
	    0x1CD6A, 0x1CD6B, 0x1CD6C, 0x1CD6D, 0x1CD6E, 0x1CD6F, 0x1CD70,
	    0x1CD71, 0x1CD72, 0x1CD73, 0x1CD74, 0x1CD75, 0x1CD76, 0x1CD77,
	    0x1CD78, 0x1CD79, 0x1CD7A, 0x1CD7B, 0x1CD7C, 0x1CD7D, 0x1CD7E,
	    0x1CD7F, 0x1CD80, 0x1CD81, 0x1CD82, 0x1CD83, 0x1CD84, 0x1CD85,
	    0x1CD86, 0x1CD87, 0x1CD88, 0x1CD89, 0x1CD8A, 0x1CD8B, 0x1CD8C,
	    0x1CD8D, 0x1CD8E, 0x1CD8F, 0x1CD90, 0x1CD91, 0x1CD92, 0x1CD93,
	    0x1CD94, 0x1CD95, 0x1CD96, 0x1CD97, 0x1CD98, 0x1CD99, 0x1CD9A,
	    0x1CD9B, 0x1CD9C, 0x1CD9D, 0x1CD9E, 0x1CD9F, 0x1CDA0, 0x1CDA1,
	    0x1CDA2, 0x1CDA3, 0x1CDA4, 0x1CDA5, 0x1CDA6, 0x1CDA7, 0x1CDA8,
	    0x1CDA9, 0x1CDAA, 0x1CDAB, 0x1CDAC, 0x1CDAD, 0x1CDAE, 0x1CDAF,
	    0x1CDB0, 0x1CDB1, 0x1CDB2, 0x1CDB3, 0x1CDB4, 0x1CDB5, 0x1CDB6,
	    0x1CDB7, 0x1CDB8, 0x1CDB9, 0x1CDBA, 0x1CDBB, 0x1CDBC, 0x1CDBD,
	    0x1CDBE, 0x1CDBF, 0x1CDC0, 0x1CDC1, 0x1CDC2, 0x1CDC3, 0x1CDC4,
	    0x1CDC5, 0x1CDC6, 0x1CDC7, 0x1CDC8, 0x1CDC9, 0x1CDCA, 0x1CDCB,
	    0x1CDCC, 0x1CDCD, 0x1CDCE, 0x1CDCF, 0x1CDD0, 0x1CDD1, 0x1CDD2,
	    0x1CDD3, 0x1CDD4, 0x1CDD5, 0x1CDD6, 0x1CDD7, 0x1CDD8, 0x1CDD9,
	    0x1CDDA, 0x1CDDB, 0x1CDDC, 0x1CDDD, 0x1CDDE, 0x1CDDF, 0x1CDE0,
	    0x1CDE1, 0x1CDE2, 0x1CDE3, 0x1CDE4, 0x1CDE5, 0x1CDE6, 0x1CDE7,
	    0x1CDE8, 0x1CDE9, 0x1CDEA, 0x1CDEB, 0x1CDEC, 0x1CDED, 0x1CDEE,
	    0x1CDEF, 0x1CDF0, 0x1CDF1, 0x1CDF2, 0x1CDF3, 0x1CDF4, 0x1CDF5,
	    0x1CDF6, 0x1CDF7, 0x1CDF8, 0x1CDF9, 0x1CDFA, 0x1CDFB, 0x1CDFC,
	    0x1CDFD, 0x1CDFE, 0x1CDFF, 0x1CE00, 0x1CE01, 0x1CE02, 0x1CE03,
	    0x1CE04, 0x1CE05, 0x1CE06, 0x1CE07, 0x1CE08, 0x1CE09, 0x1CE0A,
	    0x1CE0B, 0x1CE0C, 0x1CE0D, 0x1CE0E, 0x1CE0F, 0x1CE10, 0x1CE11,
	    0x1CE12, 0x1CE13, 0x1CE14, 0x1CE15, 0x1CE16, 0x1CE17, 0x1CE18,
	    0x1CE19, 0x1CE1A, 0x1CE1B, 0x1CE1C, 0x1CE1D, 0x1CE1E, 0x1CE1F,
	    0x1CE20, 0x1CE21, 0x1CE22, 0x1CE23, 0x1CE24, 0x1CE25, 0x1CE26,
	    0x1CE27, 0x1CE28, 0x1CE29, 0x1CE2A, 0x1CE2B, 0x1CE2C, 0x1CE2D,
	    0x1CE2E, 0x1CE2F, 0x1CE30, 0x1CE31, 0x1CE32, 0x1CE33, 0x1CE34,
	    0x1CE35, 0x1CE36, 0x1CE37, 0x1CE38, 0x1CE39, 0x1CE3A, 0x1CE3B,
	    0x1CE3C, 0x1CE3D, 0x1CE3E, 0x1CE3F, 0x1CE40, 0x1CE41, 0x1CE42,
	    0x1CE43, 0x1CE44, 0x1CE45, 0x1CE46, 0x1CE47, 0x1CE48, 0x1CE49,
	    0x1CE4A, 0x1CE4B, 0x1CE4C, 0x1CE4D, 0x1CE4E, 0x1CE4F, 0x1CE50,
	    0x1CE51, 0x1CE52, 0x1CE53, 0x1CE54, 0x1CE55, 0x1CE56, 0x1CE57,
	    0x1CE58, 0x1CE59, 0x1CE5A, 0x1CE5B, 0x1CE5C, 0x1CE5D, 0x1CE5E,
	    0x1CE5F, 0x1CE60, 0x1CE61, 0x1CE62, 0x1CE63, 0x1CE64, 0x1CE65,
	    0x1CE66, 0x1CE67, 0x1CE68, 0x1CE69, 0x1CE6A, 0x1CE6B, 0x1CE6C,
	    0x1CE6D, 0x1CE6E, 0x1CE6F, 0x1CE70, 0x1CE71, 0x1CE72, 0x1CE73,
	    0x1CE74, 0x1CE75, 0x1CE76, 0x1CE77, 0x1CE78, 0x1CE79, 0x1CE7A,
	    0x1CE7B, 0x1CE7C, 0x1CE7D, 0x1CE7E, 0x1CE7F, 0x1CE80, 0x1CE81,
	    0x1CE82, 0x1CE83, 0x1CE84, 0x1CE85, 0x1CE86, 0x1CE87, 0x1CE88,
	    0x1CE89, 0x1CE8A, 0x1CE8B, 0x1CE8C, 0x1CE8D, 0x1CE8E, 0x1CE8F,
	    0x1CE90, 0x1CE91, 0x1CE92, 0x1CE93, 0x1CE94, 0x1CE95, 0x1CE96,
	    0x1CE97, 0x1CE98, 0x1CE99, 0x1CE9A, 0x1CE9B, 0x1CE9C, 0x1CE9D,
	    0x1CE9E, 0x1CE9F, 0x1CEA0, 0x1CEA1, 0x1CEA2, 0x1CEA3, 0x1CEA4,
	    0x1CEA5, 0x1CEA6, 0x1CEA7, 0x1CEA8, 0x1CEA9, 0x1CEAA, 0x1CEAB,
	    0x1CEAC, 0x1CEAD, 0x1CEAE, 0x1CEAF, 0x1CEB0, 0x1CEB1, 0x1CEB2,
	    0x1CEB3, 0x1CF50, 0x1CF51, 0x1CF52, 0x1CF53, 0x1CF54, 0x1CF55,
	    0x1CF56, 0x1CF57, 0x1CF58, 0x1CF59, 0x1CF5A, 0x1CF5B, 0x1CF5C,
	    0x1CF5D, 0x1CF5E, 0x1CF5F, 0x1CF60, 0x1CF61, 0x1CF62, 0x1CF63,
	    0x1CF64, 0x1CF65, 0x1CF66, 0x1CF67, 0x1CF68, 0x1CF69, 0x1CF6A,
	    0x1CF6B, 0x1CF6C, 0x1CF6D, 0x1CF6E, 0x1CF6F, 0x1CF70, 0x1CF71,
	    0x1CF72, 0x1CF73, 0x1CF74, 0x1CF75, 0x1CF76, 0x1CF77, 0x1CF78,
	    0x1CF79, 0x1CF7A, 0x1CF7B, 0x1CF7C, 0x1CF7D, 0x1CF7E, 0x1CF7F,
	    0x1CF80, 0x1CF81, 0x1CF82, 0x1CF83, 0x1CF84, 0x1CF85, 0x1CF86,
	    0x1CF87, 0x1CF88, 0x1CF89, 0x1CF8A, 0x1CF8B, 0x1CF8C, 0x1CF8D,
	    0x1CF8E, 0x1CF8F, 0x1CF90, 0x1CF91, 0x1CF92, 0x1CF93, 0x1CF94,
	    0x1CF95, 0x1CF96, 0x1CF97, 0x1CF98, 0x1CF99, 0x1CF9A, 0x1CF9B,
	    0x1CF9C, 0x1CF9D, 0x1CF9E, 0x1CF9F, 0x1CFA0, 0x1CFA1, 0x1CFA2,
	    0x1CFA3, 0x1CFA4, 0x1CFA5, 0x1CFA6, 0x1CFA7, 0x1CFA8, 0x1CFA9,
	    0x1CFAA, 0x1CFAB, 0x1CFAC, 0x1CFAD, 0x1CFAE, 0x1CFAF, 0x1CFB0,
	    0x1CFB1, 0x1CFB2, 0x1CFB3, 0x1CFB4, 0x1CFB5, 0x1CFB6, 0x1CFB7,
	    0x1CFB8, 0x1CFB9, 0x1CFBA, 0x1CFBB, 0x1CFBC, 0x1CFBD, 0x1CFBE,
	    0x1CFBF, 0x1CFC0, 0x1CFC1, 0x1CFC2, 0x1CFC3, 0x1D000, 0x1D001,
	    0x1D002, 0x1D003, 0x1D004, 0x1D005, 0x1D006, 0x1D007, 0x1D008,
	    0x1D009, 0x1D00A, 0x1D00B, 0x1D00C, 0x1D00D, 0x1D00E, 0x1D00F,
	    0x1D010, 0x1D011, 0x1D012, 0x1D013, 0x1D014, 0x1D015, 0x1D016,
	    0x1D017, 0x1D018, 0x1D019, 0x1D01A, 0x1D01B, 0x1D01C, 0x1D01D,
	    0x1D01E, 0x1D01F, 0x1D020, 0x1D021, 0x1D022, 0x1D023, 0x1D024,
	    0x1D025, 0x1D026, 0x1D027, 0x1D028, 0x1D029, 0x1D02A, 0x1D02B,
	    0x1D02C, 0x1D02D, 0x1D02E, 0x1D02F, 0x1D030, 0x1D031, 0x1D032,
	    0x1D033, 0x1D034, 0x1D035, 0x1D036, 0x1D037, 0x1D038, 0x1D039,
	    0x1D03A, 0x1D03B, 0x1D03C, 0x1D03D, 0x1D03E, 0x1D03F, 0x1D040,
	    0x1D041, 0x1D042, 0x1D043, 0x1D044, 0x1D045, 0x1D046, 0x1D047,
	    0x1D048, 0x1D049, 0x1D04A, 0x1D04B, 0x1D04C, 0x1D04D, 0x1D04E,
	    0x1D04F, 0x1D050, 0x1D051, 0x1D052, 0x1D053, 0x1D054, 0x1D055,
	    0x1D056, 0x1D057, 0x1D058, 0x1D059, 0x1D05A, 0x1D05B, 0x1D05C,
	    0x1D05D, 0x1D05E, 0x1D05F, 0x1D060, 0x1D061, 0x1D062, 0x1D063,
	    0x1D064, 0x1D065, 0x1D066, 0x1D067, 0x1D068, 0x1D069, 0x1D06A,
	    0x1D06B, 0x1D06C, 0x1D06D, 0x1D06E, 0x1D06F, 0x1D070, 0x1D071,
	    0x1D072, 0x1D073, 0x1D074, 0x1D075, 0x1D076, 0x1D077, 0x1D078,
	    0x1D079, 0x1D07A, 0x1D07B, 0x1D07C, 0x1D07D, 0x1D07E, 0x1D07F,
	    0x1D080, 0x1D081, 0x1D082, 0x1D083, 0x1D084, 0x1D085, 0x1D086,
	    0x1D087, 0x1D088, 0x1D089, 0x1D08A, 0x1D08B, 0x1D08C, 0x1D08D,
	    0x1D08E, 0x1D08F, 0x1D090, 0x1D091, 0x1D092, 0x1D093, 0x1D094,
	    0x1D095, 0x1D096, 0x1D097, 0x1D098, 0x1D099, 0x1D09A, 0x1D09B,
	    0x1D09C, 0x1D09D, 0x1D09E, 0x1D09F, 0x1D0A0, 0x1D0A1, 0x1D0A2,
	    0x1D0A3, 0x1D0A4, 0x1D0A5, 0x1D0A6, 0x1D0A7, 0x1D0A8, 0x1D0A9,
	    0x1D0AA, 0x1D0AB, 0x1D0AC, 0x1D0AD, 0x1D0AE, 0x1D0AF, 0x1D0B0,
	    0x1D0B1, 0x1D0B2, 0x1D0B3, 0x1D0B4, 0x1D0B5, 0x1D0B6, 0x1D0B7,
	    0x1D0B8, 0x1D0B9, 0x1D0BA, 0x1D0BB, 0x1D0BC, 0x1D0BD, 0x1D0BE,
	    0x1D0BF, 0x1D0C0, 0x1D0C1, 0x1D0C2, 0x1D0C3, 0x1D0C4, 0x1D0C5,
	    0x1D0C6, 0x1D0C7, 0x1D0C8, 0x1D0C9, 0x1D0CA, 0x1D0CB, 0x1D0CC,
	    0x1D0CD, 0x1D0CE, 0x1D0CF, 0x1D0D0, 0x1D0D1, 0x1D0D2, 0x1D0D3,
	    0x1D0D4, 0x1D0D5, 0x1D0D6, 0x1D0D7, 0x1D0D8, 0x1D0D9, 0x1D0DA,
	    0x1D0DB, 0x1D0DC, 0x1D0DD, 0x1D0DE, 0x1D0DF, 0x1D0E0, 0x1D0E1,
	    0x1D0E2, 0x1D0E3, 0x1D0E4, 0x1D0E5, 0x1D0E6, 0x1D0E7, 0x1D0E8,
	    0x1D0E9, 0x1D0EA, 0x1D0EB, 0x1D0EC, 0x1D0ED, 0x1D0EE, 0x1D0EF,
	    0x1D0F0, 0x1D0F1, 0x1D0F2, 0x1D0F3, 0x1D0F4, 0x1D0F5, 0x1D100,
	    0x1D101, 0x1D102, 0x1D103, 0x1D104, 0x1D105, 0x1D106, 0x1D107,
	    0x1D108, 0x1D109, 0x1D10A, 0x1D10B, 0x1D10C, 0x1D10D, 0x1D10E,
	    0x1D10F, 0x1D110, 0x1D111, 0x1D112, 0x1D113, 0x1D114, 0x1D115,
	    0x1D116, 0x1D117, 0x1D118, 0x1D119, 0x1D11A, 0x1D11B, 0x1D11C,
	    0x1D11D, 0x1D11E, 0x1D11F, 0x1D120, 0x1D121, 0x1D122, 0x1D123,
	    0x1D124, 0x1D125, 0x1D126, 0x1D129, 0x1D12A, 0x1D12B, 0x1D12C,
	    0x1D12D, 0x1D12E, 0x1D12F, 0x1D130, 0x1D131, 0x1D132, 0x1D133,
	    0x1D134, 0x1D135, 0x1D136, 0x1D137, 0x1D138, 0x1D139, 0x1D13A,
	    0x1D13B, 0x1D13C, 0x1D13D, 0x1D13E, 0x1D13F, 0x1D140, 0x1D141,
	    0x1D142, 0x1D143, 0x1D144, 0x1D145, 0x1D146, 0x1D147, 0x1D148,
	    0x1D149, 0x1D14A, 0x1D14B, 0x1D14C, 0x1D14D, 0x1D14E, 0x1D14F,
	    0x1D150, 0x1D151, 0x1D152, 0x1D153, 0x1D154, 0x1D155, 0x1D156,
	    0x1D157, 0x1D158, 0x1D159, 0x1D15A, 0x1D15B, 0x1D15C, 0x1D15D,
	    0x1D15E, 0x1D15F, 0x1D160, 0x1D161, 0x1D162, 0x1D163, 0x1D164,
	    0x1D16A, 0x1D16B, 0x1D16C, 0x1D183, 0x1D184, 0x1D18C, 0x1D18D,
	    0x1D18E, 0x1D18F, 0x1D190, 0x1D191, 0x1D192, 0x1D193, 0x1D194,
	    0x1D195, 0x1D196, 0x1D197, 0x1D198, 0x1D199, 0x1D19A, 0x1D19B,
	    0x1D19C, 0x1D19D, 0x1D19E, 0x1D19F, 0x1D1A0, 0x1D1A1, 0x1D1A2,
	    0x1D1A3, 0x1D1A4, 0x1D1A5, 0x1D1A6, 0x1D1A7, 0x1D1A8, 0x1D1A9,
	    0x1D1AE, 0x1D1AF, 0x1D1B0, 0x1D1B1, 0x1D1B2, 0x1D1B3, 0x1D1B4,
	    0x1D1B5, 0x1D1B6, 0x1D1B7, 0x1D1B8, 0x1D1B9, 0x1D1BA, 0x1D1BB,
	    0x1D1BC, 0x1D1BD, 0x1D1BE, 0x1D1BF, 0x1D1C0, 0x1D1C1, 0x1D1C2,
	    0x1D1C3, 0x1D1C4, 0x1D1C5, 0x1D1C6, 0x1D1C7, 0x1D1C8, 0x1D1C9,
	    0x1D1CA, 0x1D1CB, 0x1D1CC, 0x1D1CD, 0x1D1CE, 0x1D1CF, 0x1D1D0,
	    0x1D1D1, 0x1D1D2, 0x1D1D3, 0x1D1D4, 0x1D1D5, 0x1D1D6, 0x1D1D7,
	    0x1D1D8, 0x1D1D9, 0x1D1DA, 0x1D1DB, 0x1D1DC, 0x1D1DD, 0x1D1DE,
	    0x1D1DF, 0x1D1E0, 0x1D1E1, 0x1D1E2, 0x1D1E3, 0x1D1E4, 0x1D1E5,
	    0x1D1E6, 0x1D1E7, 0x1D1E8, 0x1D1E9, 0x1D1EA, 0x1D200, 0x1D201,
	    0x1D202, 0x1D203, 0x1D204, 0x1D205, 0x1D206, 0x1D207, 0x1D208,
	    0x1D209, 0x1D20A, 0x1D20B, 0x1D20C, 0x1D20D, 0x1D20E, 0x1D20F,
	    0x1D210, 0x1D211, 0x1D212, 0x1D213, 0x1D214, 0x1D215, 0x1D216,
	    0x1D217, 0x1D218, 0x1D219, 0x1D21A, 0x1D21B, 0x1D21C, 0x1D21D,
	    0x1D21E, 0x1D21F, 0x1D220, 0x1D221, 0x1D222, 0x1D223, 0x1D224,
	    0x1D225, 0x1D226, 0x1D227, 0x1D228, 0x1D229, 0x1D22A, 0x1D22B,
	    0x1D22C, 0x1D22D, 0x1D22E, 0x1D22F, 0x1D230, 0x1D231, 0x1D232,
	    0x1D233, 0x1D234, 0x1D235, 0x1D236, 0x1D237, 0x1D238, 0x1D239,
	    0x1D23A, 0x1D23B, 0x1D23C, 0x1D23D, 0x1D23E, 0x1D23F, 0x1D240,
	    0x1D241, 0x1D245, 0x1D300, 0x1D301, 0x1D302, 0x1D303, 0x1D304,
	    0x1D305, 0x1D306, 0x1D307, 0x1D308, 0x1D309, 0x1D30A, 0x1D30B,
	    0x1D30C, 0x1D30D, 0x1D30E, 0x1D30F, 0x1D310, 0x1D311, 0x1D312,
	    0x1D313, 0x1D314, 0x1D315, 0x1D316, 0x1D317, 0x1D318, 0x1D319,
	    0x1D31A, 0x1D31B, 0x1D31C, 0x1D31D, 0x1D31E, 0x1D31F, 0x1D320,
	    0x1D321, 0x1D322, 0x1D323, 0x1D324, 0x1D325, 0x1D326, 0x1D327,
	    0x1D328, 0x1D329, 0x1D32A, 0x1D32B, 0x1D32C, 0x1D32D, 0x1D32E,
	    0x1D32F, 0x1D330, 0x1D331, 0x1D332, 0x1D333, 0x1D334, 0x1D335,
	    0x1D336, 0x1D337, 0x1D338, 0x1D339, 0x1D33A, 0x1D33B, 0x1D33C,
	    0x1D33D, 0x1D33E, 0x1D33F, 0x1D340, 0x1D341, 0x1D342, 0x1D343,
	    0x1D344, 0x1D345, 0x1D346, 0x1D347, 0x1D348, 0x1D349, 0x1D34A,
	    0x1D34B, 0x1D34C, 0x1D34D, 0x1D34E, 0x1D34F, 0x1D350, 0x1D351,
	    0x1D352, 0x1D353, 0x1D354, 0x1D355, 0x1D356, 0x1D6C1, 0x1D6DB,
	    0x1D6FB, 0x1D715, 0x1D735, 0x1D74F, 0x1D76F, 0x1D789, 0x1D7A9,
	    0x1D7C3, 0x1D800, 0x1D801, 0x1D802, 0x1D803, 0x1D804, 0x1D805,
	    0x1D806, 0x1D807, 0x1D808, 0x1D809, 0x1D80A, 0x1D80B, 0x1D80C,
	    0x1D80D, 0x1D80E, 0x1D80F, 0x1D810, 0x1D811, 0x1D812, 0x1D813,
	    0x1D814, 0x1D815, 0x1D816, 0x1D817, 0x1D818, 0x1D819, 0x1D81A,
	    0x1D81B, 0x1D81C, 0x1D81D, 0x1D81E, 0x1D81F, 0x1D820, 0x1D821,
	    0x1D822, 0x1D823, 0x1D824, 0x1D825, 0x1D826, 0x1D827, 0x1D828,
	    0x1D829, 0x1D82A, 0x1D82B, 0x1D82C, 0x1D82D, 0x1D82E, 0x1D82F,
	    0x1D830, 0x1D831, 0x1D832, 0x1D833, 0x1D834, 0x1D835, 0x1D836,
	    0x1D837, 0x1D838, 0x1D839, 0x1D83A, 0x1D83B, 0x1D83C, 0x1D83D,
	    0x1D83E, 0x1D83F, 0x1D840, 0x1D841, 0x1D842, 0x1D843, 0x1D844,
	    0x1D845, 0x1D846, 0x1D847, 0x1D848, 0x1D849, 0x1D84A, 0x1D84B,
	    0x1D84C, 0x1D84D, 0x1D84E, 0x1D84F, 0x1D850, 0x1D851, 0x1D852,
	    0x1D853, 0x1D854, 0x1D855, 0x1D856, 0x1D857, 0x1D858, 0x1D859,
	    0x1D85A, 0x1D85B, 0x1D85C, 0x1D85D, 0x1D85E, 0x1D85F, 0x1D860,
	    0x1D861, 0x1D862, 0x1D863, 0x1D864, 0x1D865, 0x1D866, 0x1D867,
	    0x1D868, 0x1D869, 0x1D86A, 0x1D86B, 0x1D86C, 0x1D86D, 0x1D86E,
	    0x1D86F, 0x1D870, 0x1D871, 0x1D872, 0x1D873, 0x1D874, 0x1D875,
	    0x1D876, 0x1D877, 0x1D878, 0x1D879, 0x1D87A, 0x1D87B, 0x1D87C,
	    0x1D87D, 0x1D87E, 0x1D87F, 0x1D880, 0x1D881, 0x1D882, 0x1D883,
	    0x1D884, 0x1D885, 0x1D886, 0x1D887, 0x1D888, 0x1D889, 0x1D88A,
	    0x1D88B, 0x1D88C, 0x1D88D, 0x1D88E, 0x1D88F, 0x1D890, 0x1D891,
	    0x1D892, 0x1D893, 0x1D894, 0x1D895, 0x1D896, 0x1D897, 0x1D898,
	    0x1D899, 0x1D89A, 0x1D89B, 0x1D89C, 0x1D89D, 0x1D89E, 0x1D89F,
	    0x1D8A0, 0x1D8A1, 0x1D8A2, 0x1D8A3, 0x1D8A4, 0x1D8A5, 0x1D8A6,
	    0x1D8A7, 0x1D8A8, 0x1D8A9, 0x1D8AA, 0x1D8AB, 0x1D8AC, 0x1D8AD,
	    0x1D8AE, 0x1D8AF, 0x1D8B0, 0x1D8B1, 0x1D8B2, 0x1D8B3, 0x1D8B4,
	    0x1D8B5, 0x1D8B6, 0x1D8B7, 0x1D8B8, 0x1D8B9, 0x1D8BA, 0x1D8BB,
	    0x1D8BC, 0x1D8BD, 0x1D8BE, 0x1D8BF, 0x1D8C0, 0x1D8C1, 0x1D8C2,
	    0x1D8C3, 0x1D8C4, 0x1D8C5, 0x1D8C6, 0x1D8C7, 0x1D8C8, 0x1D8C9,
	    0x1D8CA, 0x1D8CB, 0x1D8CC, 0x1D8CD, 0x1D8CE, 0x1D8CF, 0x1D8D0,
	    0x1D8D1, 0x1D8D2, 0x1D8D3, 0x1D8D4, 0x1D8D5, 0x1D8D6, 0x1D8D7,
	    0x1D8D8, 0x1D8D9, 0x1D8DA, 0x1D8DB, 0x1D8DC, 0x1D8DD, 0x1D8DE,
	    0x1D8DF, 0x1D8E0, 0x1D8E1, 0x1D8E2, 0x1D8E3, 0x1D8E4, 0x1D8E5,
	    0x1D8E6, 0x1D8E7, 0x1D8E8, 0x1D8E9, 0x1D8EA, 0x1D8EB, 0x1D8EC,
	    0x1D8ED, 0x1D8EE, 0x1D8EF, 0x1D8F0, 0x1D8F1, 0x1D8F2, 0x1D8F3,
	    0x1D8F4, 0x1D8F5, 0x1D8F6, 0x1D8F7, 0x1D8F8, 0x1D8F9, 0x1D8FA,
	    0x1D8FB, 0x1D8FC, 0x1D8FD, 0x1D8FE, 0x1D8FF, 0x1D900, 0x1D901,
	    0x1D902, 0x1D903, 0x1D904, 0x1D905, 0x1D906, 0x1D907, 0x1D908,
	    0x1D909, 0x1D90A, 0x1D90B, 0x1D90C, 0x1D90D, 0x1D90E, 0x1D90F,
	    0x1D910, 0x1D911, 0x1D912, 0x1D913, 0x1D914, 0x1D915, 0x1D916,
	    0x1D917, 0x1D918, 0x1D919, 0x1D91A, 0x1D91B, 0x1D91C, 0x1D91D,
	    0x1D91E, 0x1D91F, 0x1D920, 0x1D921, 0x1D922, 0x1D923, 0x1D924,
	    0x1D925, 0x1D926, 0x1D927, 0x1D928, 0x1D929, 0x1D92A, 0x1D92B,
	    0x1D92C, 0x1D92D, 0x1D92E, 0x1D92F, 0x1D930, 0x1D931, 0x1D932,
	    0x1D933, 0x1D934, 0x1D935, 0x1D936, 0x1D937, 0x1D938, 0x1D939,
	    0x1D93A, 0x1D93B, 0x1D93C, 0x1D93D, 0x1D93E, 0x1D93F, 0x1D940,
	    0x1D941, 0x1D942, 0x1D943, 0x1D944, 0x1D945, 0x1D946, 0x1D947,
	    0x1D948, 0x1D949, 0x1D94A, 0x1D94B, 0x1D94C, 0x1D94D, 0x1D94E,
	    0x1D94F, 0x1D950, 0x1D951, 0x1D952, 0x1D953, 0x1D954, 0x1D955,
	    0x1D956, 0x1D957, 0x1D958, 0x1D959, 0x1D95A, 0x1D95B, 0x1D95C,
	    0x1D95D, 0x1D95E, 0x1D95F, 0x1D960, 0x1D961, 0x1D962, 0x1D963,
	    0x1D964, 0x1D965, 0x1D966, 0x1D967, 0x1D968, 0x1D969, 0x1D96A,
	    0x1D96B, 0x1D96C, 0x1D96D, 0x1D96E, 0x1D96F, 0x1D970, 0x1D971,
	    0x1D972, 0x1D973, 0x1D974, 0x1D975, 0x1D976, 0x1D977, 0x1D978,
	    0x1D979, 0x1D97A, 0x1D97B, 0x1D97C, 0x1D97D, 0x1D97E, 0x1D97F,
	    0x1D980, 0x1D981, 0x1D982, 0x1D983, 0x1D984, 0x1D985, 0x1D986,
	    0x1D987, 0x1D988, 0x1D989, 0x1D98A, 0x1D98B, 0x1D98C, 0x1D98D,
	    0x1D98E, 0x1D98F, 0x1D990, 0x1D991, 0x1D992, 0x1D993, 0x1D994,
	    0x1D995, 0x1D996, 0x1D997, 0x1D998, 0x1D999, 0x1D99A, 0x1D99B,
	    0x1D99C, 0x1D99D, 0x1D99E, 0x1D99F, 0x1D9A0, 0x1D9A1, 0x1D9A2,
	    0x1D9A3, 0x1D9A4, 0x1D9A5, 0x1D9A6, 0x1D9A7, 0x1D9A8, 0x1D9A9,
	    0x1D9AA, 0x1D9AB, 0x1D9AC, 0x1D9AD, 0x1D9AE, 0x1D9AF, 0x1D9B0,
	    0x1D9B1, 0x1D9B2, 0x1D9B3, 0x1D9B4, 0x1D9B5, 0x1D9B6, 0x1D9B7,
	    0x1D9B8, 0x1D9B9, 0x1D9BA, 0x1D9BB, 0x1D9BC, 0x1D9BD, 0x1D9BE,
	    0x1D9BF, 0x1D9C0, 0x1D9C1, 0x1D9C2, 0x1D9C3, 0x1D9C4, 0x1D9C5,
	    0x1D9C6, 0x1D9C7, 0x1D9C8, 0x1D9C9, 0x1D9CA, 0x1D9CB, 0x1D9CC,
	    0x1D9CD, 0x1D9CE, 0x1D9CF, 0x1D9D0, 0x1D9D1, 0x1D9D2, 0x1D9D3,
	    0x1D9D4, 0x1D9D5, 0x1D9D6, 0x1D9D7, 0x1D9D8, 0x1D9D9, 0x1D9DA,
	    0x1D9DB, 0x1D9DC, 0x1D9DD, 0x1D9DE, 0x1D9DF, 0x1D9E0, 0x1D9E1,
	    0x1D9E2, 0x1D9E3, 0x1D9E4, 0x1D9E5, 0x1D9E6, 0x1D9E7, 0x1D9E8,
	    0x1D9E9, 0x1D9EA, 0x1D9EB, 0x1D9EC, 0x1D9ED, 0x1D9EE, 0x1D9EF,
	    0x1D9F0, 0x1D9F1, 0x1D9F2, 0x1D9F3, 0x1D9F4, 0x1D9F5, 0x1D9F6,
	    0x1D9F7, 0x1D9F8, 0x1D9F9, 0x1D9FA, 0x1D9FB, 0x1D9FC, 0x1D9FD,
	    0x1D9FE, 0x1D9FF, 0x1DA37, 0x1DA38, 0x1DA39, 0x1DA3A, 0x1DA6D,
	    0x1DA6E, 0x1DA6F, 0x1DA70, 0x1DA71, 0x1DA72, 0x1DA73, 0x1DA74,
	    0x1DA76, 0x1DA77, 0x1DA78, 0x1DA79, 0x1DA7A, 0x1DA7B, 0x1DA7C,
	    0x1DA7D, 0x1DA7E, 0x1DA7F, 0x1DA80, 0x1DA81, 0x1DA82, 0x1DA83,
	    0x1DA85, 0x1DA86, 0x1E14F, 0x1E2FF, 0x1ECAC, 0x1ECB0, 0x1ED2E,
	    0x1EEF0, 0x1EEF1, 0x1F000, 0x1F001, 0x1F002, 0x1F003, 0x1F004,
	    0x1F005, 0x1F006, 0x1F007, 0x1F008, 0x1F009, 0x1F00A, 0x1F00B,
	    0x1F00C, 0x1F00D, 0x1F00E, 0x1F00F, 0x1F010, 0x1F011, 0x1F012,
	    0x1F013, 0x1F014, 0x1F015, 0x1F016, 0x1F017, 0x1F018, 0x1F019,
	    0x1F01A, 0x1F01B, 0x1F01C, 0x1F01D, 0x1F01E, 0x1F01F, 0x1F020,
	    0x1F021, 0x1F022, 0x1F023, 0x1F024, 0x1F025, 0x1F026, 0x1F027,
	    0x1F028, 0x1F029, 0x1F02A, 0x1F02B, 0x1F030, 0x1F031, 0x1F032,
	    0x1F033, 0x1F034, 0x1F035, 0x1F036, 0x1F037, 0x1F038, 0x1F039,
	    0x1F03A, 0x1F03B, 0x1F03C, 0x1F03D, 0x1F03E, 0x1F03F, 0x1F040,
	    0x1F041, 0x1F042, 0x1F043, 0x1F044, 0x1F045, 0x1F046, 0x1F047,
	    0x1F048, 0x1F049, 0x1F04A, 0x1F04B, 0x1F04C, 0x1F04D, 0x1F04E,
	    0x1F04F, 0x1F050, 0x1F051, 0x1F052, 0x1F053, 0x1F054, 0x1F055,
	    0x1F056, 0x1F057, 0x1F058, 0x1F059, 0x1F05A, 0x1F05B, 0x1F05C,
	    0x1F05D, 0x1F05E, 0x1F05F, 0x1F060, 0x1F061, 0x1F062, 0x1F063,
	    0x1F064, 0x1F065, 0x1F066, 0x1F067, 0x1F068, 0x1F069, 0x1F06A,
	    0x1F06B, 0x1F06C, 0x1F06D, 0x1F06E, 0x1F06F, 0x1F070, 0x1F071,
	    0x1F072, 0x1F073, 0x1F074, 0x1F075, 0x1F076, 0x1F077, 0x1F078,
	    0x1F079, 0x1F07A, 0x1F07B, 0x1F07C, 0x1F07D, 0x1F07E, 0x1F07F,
	    0x1F080, 0x1F081, 0x1F082, 0x1F083, 0x1F084, 0x1F085, 0x1F086,
	    0x1F087, 0x1F088, 0x1F089, 0x1F08A, 0x1F08B, 0x1F08C, 0x1F08D,
	    0x1F08E, 0x1F08F, 0x1F090, 0x1F091, 0x1F092, 0x1F093, 0x1F0A0,
	    0x1F0A1, 0x1F0A2, 0x1F0A3, 0x1F0A4, 0x1F0A5, 0x1F0A6, 0x1F0A7,
	    0x1F0A8, 0x1F0A9, 0x1F0AA, 0x1F0AB, 0x1F0AC, 0x1F0AD, 0x1F0AE,
	    0x1F0B1, 0x1F0B2, 0x1F0B3, 0x1F0B4, 0x1F0B5, 0x1F0B6, 0x1F0B7,
	    0x1F0B8, 0x1F0B9, 0x1F0BA, 0x1F0BB, 0x1F0BC, 0x1F0BD, 0x1F0BE,
	    0x1F0BF, 0x1F0C1, 0x1F0C2, 0x1F0C3, 0x1F0C4, 0x1F0C5, 0x1F0C6,
	    0x1F0C7, 0x1F0C8, 0x1F0C9, 0x1F0CA, 0x1F0CB, 0x1F0CC, 0x1F0CD,
	    0x1F0CE, 0x1F0CF, 0x1F0D1, 0x1F0D2, 0x1F0D3, 0x1F0D4, 0x1F0D5,
	    0x1F0D6, 0x1F0D7, 0x1F0D8, 0x1F0D9, 0x1F0DA, 0x1F0DB, 0x1F0DC,
	    0x1F0DD, 0x1F0DE, 0x1F0DF, 0x1F0E0, 0x1F0E1, 0x1F0E2, 0x1F0E3,
	    0x1F0E4, 0x1F0E5, 0x1F0E6, 0x1F0E7, 0x1F0E8, 0x1F0E9, 0x1F0EA,
	    0x1F0EB, 0x1F0EC, 0x1F0ED, 0x1F0EE, 0x1F0EF, 0x1F0F0, 0x1F0F1,
	    0x1F0F2, 0x1F0F3, 0x1F0F4, 0x1F0F5, 0x1F10D, 0x1F10E, 0x1F10F,
	    0x1F110, 0x1F111, 0x1F112, 0x1F113, 0x1F114, 0x1F115, 0x1F116,
	    0x1F117, 0x1F118, 0x1F119, 0x1F11A, 0x1F11B, 0x1F11C, 0x1F11D,
	    0x1F11E, 0x1F11F, 0x1F120, 0x1F121, 0x1F122, 0x1F123, 0x1F124,
	    0x1F125, 0x1F126, 0x1F127, 0x1F128, 0x1F129, 0x1F12A, 0x1F12B,
	    0x1F12C, 0x1F12D, 0x1F12E, 0x1F12F, 0x1F130, 0x1F131, 0x1F132,
	    0x1F133, 0x1F134, 0x1F135, 0x1F136, 0x1F137, 0x1F138, 0x1F139,
	    0x1F13A, 0x1F13B, 0x1F13C, 0x1F13D, 0x1F13E, 0x1F13F, 0x1F140,
	    0x1F141, 0x1F142, 0x1F143, 0x1F144, 0x1F145, 0x1F146, 0x1F147,
	    0x1F148, 0x1F149, 0x1F14A, 0x1F14B, 0x1F14C, 0x1F14D, 0x1F14E,
	    0x1F14F, 0x1F150, 0x1F151, 0x1F152, 0x1F153, 0x1F154, 0x1F155,
	    0x1F156, 0x1F157, 0x1F158, 0x1F159, 0x1F15A, 0x1F15B, 0x1F15C,
	    0x1F15D, 0x1F15E, 0x1F15F, 0x1F160, 0x1F161, 0x1F162, 0x1F163,
	    0x1F164, 0x1F165, 0x1F166, 0x1F167, 0x1F168, 0x1F169, 0x1F16A,
	    0x1F16B, 0x1F16C, 0x1F16D, 0x1F16E, 0x1F16F, 0x1F170, 0x1F171,
	    0x1F172, 0x1F173, 0x1F174, 0x1F175, 0x1F176, 0x1F177, 0x1F178,
	    0x1F179, 0x1F17A, 0x1F17B, 0x1F17C, 0x1F17D, 0x1F17E, 0x1F17F,
	    0x1F180, 0x1F181, 0x1F182, 0x1F183, 0x1F184, 0x1F185, 0x1F186,
	    0x1F187, 0x1F188, 0x1F189, 0x1F18A, 0x1F18B, 0x1F18C, 0x1F18D,
	    0x1F18E, 0x1F18F, 0x1F190, 0x1F191, 0x1F192, 0x1F193, 0x1F194,
	    0x1F195, 0x1F196, 0x1F197, 0x1F198, 0x1F199, 0x1F19A, 0x1F19B,
	    0x1F19C, 0x1F19D, 0x1F19E, 0x1F19F, 0x1F1A0, 0x1F1A1, 0x1F1A2,
	    0x1F1A3, 0x1F1A4, 0x1F1A5, 0x1F1A6, 0x1F1A7, 0x1F1A8, 0x1F1A9,
	    0x1F1AA, 0x1F1AB, 0x1F1AC, 0x1F1AD, 0x1F1E6, 0x1F1E7, 0x1F1E8,
	    0x1F1E9, 0x1F1EA, 0x1F1EB, 0x1F1EC, 0x1F1ED, 0x1F1EE, 0x1F1EF,
	    0x1F1F0, 0x1F1F1, 0x1F1F2, 0x1F1F3, 0x1F1F4, 0x1F1F5, 0x1F1F6,
	    0x1F1F7, 0x1F1F8, 0x1F1F9, 0x1F1FA, 0x1F1FB, 0x1F1FC, 0x1F1FD,
	    0x1F1FE, 0x1F1FF, 0x1F200, 0x1F201, 0x1F202, 0x1F210, 0x1F211,
	    0x1F212, 0x1F213, 0x1F214, 0x1F215, 0x1F216, 0x1F217, 0x1F218,
	    0x1F219, 0x1F21A, 0x1F21B, 0x1F21C, 0x1F21D, 0x1F21E, 0x1F21F,
	    0x1F220, 0x1F221, 0x1F222, 0x1F223, 0x1F224, 0x1F225, 0x1F226,
	    0x1F227, 0x1F228, 0x1F229, 0x1F22A, 0x1F22B, 0x1F22C, 0x1F22D,
	    0x1F22E, 0x1F22F, 0x1F230, 0x1F231, 0x1F232, 0x1F233, 0x1F234,
	    0x1F235, 0x1F236, 0x1F237, 0x1F238, 0x1F239, 0x1F23A, 0x1F23B,
	    0x1F240, 0x1F241, 0x1F242, 0x1F243, 0x1F244, 0x1F245, 0x1F246,
	    0x1F247, 0x1F248, 0x1F250, 0x1F251, 0x1F260, 0x1F261, 0x1F262,
	    0x1F263, 0x1F264, 0x1F265, 0x1F300, 0x1F301, 0x1F302, 0x1F303,
	    0x1F304, 0x1F305, 0x1F306, 0x1F307, 0x1F308, 0x1F309, 0x1F30A,
	    0x1F30B, 0x1F30C, 0x1F30D, 0x1F30E, 0x1F30F, 0x1F310, 0x1F311,
	    0x1F312, 0x1F313, 0x1F314, 0x1F315, 0x1F316, 0x1F317, 0x1F318,
	    0x1F319, 0x1F31A, 0x1F31B, 0x1F31C, 0x1F31D, 0x1F31E, 0x1F31F,
	    0x1F320, 0x1F321, 0x1F322, 0x1F323, 0x1F324, 0x1F325, 0x1F326,
	    0x1F327, 0x1F328, 0x1F329, 0x1F32A, 0x1F32B, 0x1F32C, 0x1F32D,
	    0x1F32E, 0x1F32F, 0x1F330, 0x1F331, 0x1F332, 0x1F333, 0x1F334,
	    0x1F335, 0x1F336, 0x1F337, 0x1F338, 0x1F339, 0x1F33A, 0x1F33B,
	    0x1F33C, 0x1F33D, 0x1F33E, 0x1F33F, 0x1F340, 0x1F341, 0x1F342,
	    0x1F343, 0x1F344, 0x1F345, 0x1F346, 0x1F347, 0x1F348, 0x1F349,
	    0x1F34A, 0x1F34B, 0x1F34C, 0x1F34D, 0x1F34E, 0x1F34F, 0x1F350,
	    0x1F351, 0x1F352, 0x1F353, 0x1F354, 0x1F355, 0x1F356, 0x1F357,
	    0x1F358, 0x1F359, 0x1F35A, 0x1F35B, 0x1F35C, 0x1F35D, 0x1F35E,
	    0x1F35F, 0x1F360, 0x1F361, 0x1F362, 0x1F363, 0x1F364, 0x1F365,
	    0x1F366, 0x1F367, 0x1F368, 0x1F369, 0x1F36A, 0x1F36B, 0x1F36C,
	    0x1F36D, 0x1F36E, 0x1F36F, 0x1F370, 0x1F371, 0x1F372, 0x1F373,
	    0x1F374, 0x1F375, 0x1F376, 0x1F377, 0x1F378, 0x1F379, 0x1F37A,
	    0x1F37B, 0x1F37C, 0x1F37D, 0x1F37E, 0x1F37F, 0x1F380, 0x1F381,
	    0x1F382, 0x1F383, 0x1F384, 0x1F385, 0x1F386, 0x1F387, 0x1F388,
	    0x1F389, 0x1F38A, 0x1F38B, 0x1F38C, 0x1F38D, 0x1F38E, 0x1F38F,
	    0x1F390, 0x1F391, 0x1F392, 0x1F393, 0x1F394, 0x1F395, 0x1F396,
	    0x1F397, 0x1F398, 0x1F399, 0x1F39A, 0x1F39B, 0x1F39C, 0x1F39D,
	    0x1F39E, 0x1F39F, 0x1F3A0, 0x1F3A1, 0x1F3A2, 0x1F3A3, 0x1F3A4,
	    0x1F3A5, 0x1F3A6, 0x1F3A7, 0x1F3A8, 0x1F3A9, 0x1F3AA, 0x1F3AB,
	    0x1F3AC, 0x1F3AD, 0x1F3AE, 0x1F3AF, 0x1F3B0, 0x1F3B1, 0x1F3B2,
	    0x1F3B3, 0x1F3B4, 0x1F3B5, 0x1F3B6, 0x1F3B7, 0x1F3B8, 0x1F3B9,
	    0x1F3BA, 0x1F3BB, 0x1F3BC, 0x1F3BD, 0x1F3BE, 0x1F3BF, 0x1F3C0,
	    0x1F3C1, 0x1F3C2, 0x1F3C3, 0x1F3C4, 0x1F3C5, 0x1F3C6, 0x1F3C7,
	    0x1F3C8, 0x1F3C9, 0x1F3CA, 0x1F3CB, 0x1F3CC, 0x1F3CD, 0x1F3CE,
	    0x1F3CF, 0x1F3D0, 0x1F3D1, 0x1F3D2, 0x1F3D3, 0x1F3D4, 0x1F3D5,
	    0x1F3D6, 0x1F3D7, 0x1F3D8, 0x1F3D9, 0x1F3DA, 0x1F3DB, 0x1F3DC,
	    0x1F3DD, 0x1F3DE, 0x1F3DF, 0x1F3E0, 0x1F3E1, 0x1F3E2, 0x1F3E3,
	    0x1F3E4, 0x1F3E5, 0x1F3E6, 0x1F3E7, 0x1F3E8, 0x1F3E9, 0x1F3EA,
	    0x1F3EB, 0x1F3EC, 0x1F3ED, 0x1F3EE, 0x1F3EF, 0x1F3F0, 0x1F3F1,
	    0x1F3F2, 0x1F3F3, 0x1F3F4, 0x1F3F5, 0x1F3F6, 0x1F3F7, 0x1F3F8,
	    0x1F3F9, 0x1F3FA, 0x1F3FB, 0x1F3FC, 0x1F3FD, 0x1F3FE, 0x1F3FF,
	    0x1F400, 0x1F401, 0x1F402, 0x1F403, 0x1F404, 0x1F405, 0x1F406,
	    0x1F407, 0x1F408, 0x1F409, 0x1F40A, 0x1F40B, 0x1F40C, 0x1F40D,
	    0x1F40E, 0x1F40F, 0x1F410, 0x1F411, 0x1F412, 0x1F413, 0x1F414,
	    0x1F415, 0x1F416, 0x1F417, 0x1F418, 0x1F419, 0x1F41A, 0x1F41B,
	    0x1F41C, 0x1F41D, 0x1F41E, 0x1F41F, 0x1F420, 0x1F421, 0x1F422,
	    0x1F423, 0x1F424, 0x1F425, 0x1F426, 0x1F427, 0x1F428, 0x1F429,
	    0x1F42A, 0x1F42B, 0x1F42C, 0x1F42D, 0x1F42E, 0x1F42F, 0x1F430,
	    0x1F431, 0x1F432, 0x1F433, 0x1F434, 0x1F435, 0x1F436, 0x1F437,
	    0x1F438, 0x1F439, 0x1F43A, 0x1F43B, 0x1F43C, 0x1F43D, 0x1F43E,
	    0x1F43F, 0x1F440, 0x1F441, 0x1F442, 0x1F443, 0x1F444, 0x1F445,
	    0x1F446, 0x1F447, 0x1F448, 0x1F449, 0x1F44A, 0x1F44B, 0x1F44C,
	    0x1F44D, 0x1F44E, 0x1F44F, 0x1F450, 0x1F451, 0x1F452, 0x1F453,
	    0x1F454, 0x1F455, 0x1F456, 0x1F457, 0x1F458, 0x1F459, 0x1F45A,
	    0x1F45B, 0x1F45C, 0x1F45D, 0x1F45E, 0x1F45F, 0x1F460, 0x1F461,
	    0x1F462, 0x1F463, 0x1F464, 0x1F465, 0x1F466, 0x1F467, 0x1F468,
	    0x1F469, 0x1F46A, 0x1F46B, 0x1F46C, 0x1F46D, 0x1F46E, 0x1F46F,
	    0x1F470, 0x1F471, 0x1F472, 0x1F473, 0x1F474, 0x1F475, 0x1F476,
	    0x1F477, 0x1F478, 0x1F479, 0x1F47A, 0x1F47B, 0x1F47C, 0x1F47D,
	    0x1F47E, 0x1F47F, 0x1F480, 0x1F481, 0x1F482, 0x1F483, 0x1F484,
	    0x1F485, 0x1F486, 0x1F487, 0x1F488, 0x1F489, 0x1F48A, 0x1F48B,
	    0x1F48C, 0x1F48D, 0x1F48E, 0x1F48F, 0x1F490, 0x1F491, 0x1F492,
	    0x1F493, 0x1F494, 0x1F495, 0x1F496, 0x1F497, 0x1F498, 0x1F499,
	    0x1F49A, 0x1F49B, 0x1F49C, 0x1F49D, 0x1F49E, 0x1F49F, 0x1F4A0,
	    0x1F4A1, 0x1F4A2, 0x1F4A3, 0x1F4A4, 0x1F4A5, 0x1F4A6, 0x1F4A7,
	    0x1F4A8, 0x1F4A9, 0x1F4AA, 0x1F4AB, 0x1F4AC, 0x1F4AD, 0x1F4AE,
	    0x1F4AF, 0x1F4B0, 0x1F4B1, 0x1F4B2, 0x1F4B3, 0x1F4B4, 0x1F4B5,
	    0x1F4B6, 0x1F4B7, 0x1F4B8, 0x1F4B9, 0x1F4BA, 0x1F4BB, 0x1F4BC,
	    0x1F4BD, 0x1F4BE, 0x1F4BF, 0x1F4C0, 0x1F4C1, 0x1F4C2, 0x1F4C3,
	    0x1F4C4, 0x1F4C5, 0x1F4C6, 0x1F4C7, 0x1F4C8, 0x1F4C9, 0x1F4CA,
	    0x1F4CB, 0x1F4CC, 0x1F4CD, 0x1F4CE, 0x1F4CF, 0x1F4D0, 0x1F4D1,
	    0x1F4D2, 0x1F4D3, 0x1F4D4, 0x1F4D5, 0x1F4D6, 0x1F4D7, 0x1F4D8,
	    0x1F4D9, 0x1F4DA, 0x1F4DB, 0x1F4DC, 0x1F4DD, 0x1F4DE, 0x1F4DF,
	    0x1F4E0, 0x1F4E1, 0x1F4E2, 0x1F4E3, 0x1F4E4, 0x1F4E5, 0x1F4E6,
	    0x1F4E7, 0x1F4E8, 0x1F4E9, 0x1F4EA, 0x1F4EB, 0x1F4EC, 0x1F4ED,
	    0x1F4EE, 0x1F4EF, 0x1F4F0, 0x1F4F1, 0x1F4F2, 0x1F4F3, 0x1F4F4,
	    0x1F4F5, 0x1F4F6, 0x1F4F7, 0x1F4F8, 0x1F4F9, 0x1F4FA, 0x1F4FB,
	    0x1F4FC, 0x1F4FD, 0x1F4FE, 0x1F4FF, 0x1F500, 0x1F501, 0x1F502,
	    0x1F503, 0x1F504, 0x1F505, 0x1F506, 0x1F507, 0x1F508, 0x1F509,
	    0x1F50A, 0x1F50B, 0x1F50C, 0x1F50D, 0x1F50E, 0x1F50F, 0x1F510,
	    0x1F511, 0x1F512, 0x1F513, 0x1F514, 0x1F515, 0x1F516, 0x1F517,
	    0x1F518, 0x1F519, 0x1F51A, 0x1F51B, 0x1F51C, 0x1F51D, 0x1F51E,
	    0x1F51F, 0x1F520, 0x1F521, 0x1F522, 0x1F523, 0x1F524, 0x1F525,
	    0x1F526, 0x1F527, 0x1F528, 0x1F529, 0x1F52A, 0x1F52B, 0x1F52C,
	    0x1F52D, 0x1F52E, 0x1F52F, 0x1F530, 0x1F531, 0x1F532, 0x1F533,
	    0x1F534, 0x1F535, 0x1F536, 0x1F537, 0x1F538, 0x1F539, 0x1F53A,
	    0x1F53B, 0x1F53C, 0x1F53D, 0x1F53E, 0x1F53F, 0x1F540, 0x1F541,
	    0x1F542, 0x1F543, 0x1F544, 0x1F545, 0x1F546, 0x1F547, 0x1F548,
	    0x1F549, 0x1F54A, 0x1F54B, 0x1F54C, 0x1F54D, 0x1F54E, 0x1F54F,
	    0x1F550, 0x1F551, 0x1F552, 0x1F553, 0x1F554, 0x1F555, 0x1F556,
	    0x1F557, 0x1F558, 0x1F559, 0x1F55A, 0x1F55B, 0x1F55C, 0x1F55D,
	    0x1F55E, 0x1F55F, 0x1F560, 0x1F561, 0x1F562, 0x1F563, 0x1F564,
	    0x1F565, 0x1F566, 0x1F567, 0x1F568, 0x1F569, 0x1F56A, 0x1F56B,
	    0x1F56C, 0x1F56D, 0x1F56E, 0x1F56F, 0x1F570, 0x1F571, 0x1F572,
	    0x1F573, 0x1F574, 0x1F575, 0x1F576, 0x1F577, 0x1F578, 0x1F579,
	    0x1F57A, 0x1F57B, 0x1F57C, 0x1F57D, 0x1F57E, 0x1F57F, 0x1F580,
	    0x1F581, 0x1F582, 0x1F583, 0x1F584, 0x1F585, 0x1F586, 0x1F587,
	    0x1F588, 0x1F589, 0x1F58A, 0x1F58B, 0x1F58C, 0x1F58D, 0x1F58E,
	    0x1F58F, 0x1F590, 0x1F591, 0x1F592, 0x1F593, 0x1F594, 0x1F595,
	    0x1F596, 0x1F597, 0x1F598, 0x1F599, 0x1F59A, 0x1F59B, 0x1F59C,
	    0x1F59D, 0x1F59E, 0x1F59F, 0x1F5A0, 0x1F5A1, 0x1F5A2, 0x1F5A3,
	    0x1F5A4, 0x1F5A5, 0x1F5A6, 0x1F5A7, 0x1F5A8, 0x1F5A9, 0x1F5AA,
	    0x1F5AB, 0x1F5AC, 0x1F5AD, 0x1F5AE, 0x1F5AF, 0x1F5B0, 0x1F5B1,
	    0x1F5B2, 0x1F5B3, 0x1F5B4, 0x1F5B5, 0x1F5B6, 0x1F5B7, 0x1F5B8,
	    0x1F5B9, 0x1F5BA, 0x1F5BB, 0x1F5BC, 0x1F5BD, 0x1F5BE, 0x1F5BF,
	    0x1F5C0, 0x1F5C1, 0x1F5C2, 0x1F5C3, 0x1F5C4, 0x1F5C5, 0x1F5C6,
	    0x1F5C7, 0x1F5C8, 0x1F5C9, 0x1F5CA, 0x1F5CB, 0x1F5CC, 0x1F5CD,
	    0x1F5CE, 0x1F5CF, 0x1F5D0, 0x1F5D1, 0x1F5D2, 0x1F5D3, 0x1F5D4,
	    0x1F5D5, 0x1F5D6, 0x1F5D7, 0x1F5D8, 0x1F5D9, 0x1F5DA, 0x1F5DB,
	    0x1F5DC, 0x1F5DD, 0x1F5DE, 0x1F5DF, 0x1F5E0, 0x1F5E1, 0x1F5E2,
	    0x1F5E3, 0x1F5E4, 0x1F5E5, 0x1F5E6, 0x1F5E7, 0x1F5E8, 0x1F5E9,
	    0x1F5EA, 0x1F5EB, 0x1F5EC, 0x1F5ED, 0x1F5EE, 0x1F5EF, 0x1F5F0,
	    0x1F5F1, 0x1F5F2, 0x1F5F3, 0x1F5F4, 0x1F5F5, 0x1F5F6, 0x1F5F7,
	    0x1F5F8, 0x1F5F9, 0x1F5FA, 0x1F5FB, 0x1F5FC, 0x1F5FD, 0x1F5FE,
	    0x1F5FF, 0x1F600, 0x1F601, 0x1F602, 0x1F603, 0x1F604, 0x1F605,
	    0x1F606, 0x1F607, 0x1F608, 0x1F609, 0x1F60A, 0x1F60B, 0x1F60C,
	    0x1F60D, 0x1F60E, 0x1F60F, 0x1F610, 0x1F611, 0x1F612, 0x1F613,
	    0x1F614, 0x1F615, 0x1F616, 0x1F617, 0x1F618, 0x1F619, 0x1F61A,
	    0x1F61B, 0x1F61C, 0x1F61D, 0x1F61E, 0x1F61F, 0x1F620, 0x1F621,
	    0x1F622, 0x1F623, 0x1F624, 0x1F625, 0x1F626, 0x1F627, 0x1F628,
	    0x1F629, 0x1F62A, 0x1F62B, 0x1F62C, 0x1F62D, 0x1F62E, 0x1F62F,
	    0x1F630, 0x1F631, 0x1F632, 0x1F633, 0x1F634, 0x1F635, 0x1F636,
	    0x1F637, 0x1F638, 0x1F639, 0x1F63A, 0x1F63B, 0x1F63C, 0x1F63D,
	    0x1F63E, 0x1F63F, 0x1F640, 0x1F641, 0x1F642, 0x1F643, 0x1F644,
	    0x1F645, 0x1F646, 0x1F647, 0x1F648, 0x1F649, 0x1F64A, 0x1F64B,
	    0x1F64C, 0x1F64D, 0x1F64E, 0x1F64F, 0x1F650, 0x1F651, 0x1F652,
	    0x1F653, 0x1F654, 0x1F655, 0x1F656, 0x1F657, 0x1F658, 0x1F659,
	    0x1F65A, 0x1F65B, 0x1F65C, 0x1F65D, 0x1F65E, 0x1F65F, 0x1F660,
	    0x1F661, 0x1F662, 0x1F663, 0x1F664, 0x1F665, 0x1F666, 0x1F667,
	    0x1F668, 0x1F669, 0x1F66A, 0x1F66B, 0x1F66C, 0x1F66D, 0x1F66E,
	    0x1F66F, 0x1F670, 0x1F671, 0x1F672, 0x1F673, 0x1F674, 0x1F675,
	    0x1F676, 0x1F677, 0x1F678, 0x1F679, 0x1F67A, 0x1F67B, 0x1F67C,
	    0x1F67D, 0x1F67E, 0x1F67F, 0x1F680, 0x1F681, 0x1F682, 0x1F683,
	    0x1F684, 0x1F685, 0x1F686, 0x1F687, 0x1F688, 0x1F689, 0x1F68A,
	    0x1F68B, 0x1F68C, 0x1F68D, 0x1F68E, 0x1F68F, 0x1F690, 0x1F691,
	    0x1F692, 0x1F693, 0x1F694, 0x1F695, 0x1F696, 0x1F697, 0x1F698,
	    0x1F699, 0x1F69A, 0x1F69B, 0x1F69C, 0x1F69D, 0x1F69E, 0x1F69F,
	    0x1F6A0, 0x1F6A1, 0x1F6A2, 0x1F6A3, 0x1F6A4, 0x1F6A5, 0x1F6A6,
	    0x1F6A7, 0x1F6A8, 0x1F6A9, 0x1F6AA, 0x1F6AB, 0x1F6AC, 0x1F6AD,
	    0x1F6AE, 0x1F6AF, 0x1F6B0, 0x1F6B1, 0x1F6B2, 0x1F6B3, 0x1F6B4,
	    0x1F6B5, 0x1F6B6, 0x1F6B7, 0x1F6B8, 0x1F6B9, 0x1F6BA, 0x1F6BB,
	    0x1F6BC, 0x1F6BD, 0x1F6BE, 0x1F6BF, 0x1F6C0, 0x1F6C1, 0x1F6C2,
	    0x1F6C3, 0x1F6C4, 0x1F6C5, 0x1F6C6, 0x1F6C7, 0x1F6C8, 0x1F6C9,
	    0x1F6CA, 0x1F6CB, 0x1F6CC, 0x1F6CD, 0x1F6CE, 0x1F6CF, 0x1F6D0,
	    0x1F6D1, 0x1F6D2, 0x1F6D3, 0x1F6D4, 0x1F6D5, 0x1F6D6, 0x1F6D7,
	    0x1F6DC, 0x1F6DD, 0x1F6DE, 0x1F6DF, 0x1F6E0, 0x1F6E1, 0x1F6E2,
	    0x1F6E3, 0x1F6E4, 0x1F6E5, 0x1F6E6, 0x1F6E7, 0x1F6E8, 0x1F6E9,
	    0x1F6EA, 0x1F6EB, 0x1F6EC, 0x1F6F0, 0x1F6F1, 0x1F6F2, 0x1F6F3,
	    0x1F6F4, 0x1F6F5, 0x1F6F6, 0x1F6F7, 0x1F6F8, 0x1F6F9, 0x1F6FA,
	    0x1F6FB, 0x1F6FC, 0x1F700, 0x1F701, 0x1F702, 0x1F703, 0x1F704,
	    0x1F705, 0x1F706, 0x1F707, 0x1F708, 0x1F709, 0x1F70A, 0x1F70B,
	    0x1F70C, 0x1F70D, 0x1F70E, 0x1F70F, 0x1F710, 0x1F711, 0x1F712,
	    0x1F713, 0x1F714, 0x1F715, 0x1F716, 0x1F717, 0x1F718, 0x1F719,
	    0x1F71A, 0x1F71B, 0x1F71C, 0x1F71D, 0x1F71E, 0x1F71F, 0x1F720,
	    0x1F721, 0x1F722, 0x1F723, 0x1F724, 0x1F725, 0x1F726, 0x1F727,
	    0x1F728, 0x1F729, 0x1F72A, 0x1F72B, 0x1F72C, 0x1F72D, 0x1F72E,
	    0x1F72F, 0x1F730, 0x1F731, 0x1F732, 0x1F733, 0x1F734, 0x1F735,
	    0x1F736, 0x1F737, 0x1F738, 0x1F739, 0x1F73A, 0x1F73B, 0x1F73C,
	    0x1F73D, 0x1F73E, 0x1F73F, 0x1F740, 0x1F741, 0x1F742, 0x1F743,
	    0x1F744, 0x1F745, 0x1F746, 0x1F747, 0x1F748, 0x1F749, 0x1F74A,
	    0x1F74B, 0x1F74C, 0x1F74D, 0x1F74E, 0x1F74F, 0x1F750, 0x1F751,
	    0x1F752, 0x1F753, 0x1F754, 0x1F755, 0x1F756, 0x1F757, 0x1F758,
	    0x1F759, 0x1F75A, 0x1F75B, 0x1F75C, 0x1F75D, 0x1F75E, 0x1F75F,
	    0x1F760, 0x1F761, 0x1F762, 0x1F763, 0x1F764, 0x1F765, 0x1F766,
	    0x1F767, 0x1F768, 0x1F769, 0x1F76A, 0x1F76B, 0x1F76C, 0x1F76D,
	    0x1F76E, 0x1F76F, 0x1F770, 0x1F771, 0x1F772, 0x1F773, 0x1F774,
	    0x1F775, 0x1F776, 0x1F77B, 0x1F77C, 0x1F77D, 0x1F77E, 0x1F77F,
	    0x1F780, 0x1F781, 0x1F782, 0x1F783, 0x1F784, 0x1F785, 0x1F786,
	    0x1F787, 0x1F788, 0x1F789, 0x1F78A, 0x1F78B, 0x1F78C, 0x1F78D,
	    0x1F78E, 0x1F78F, 0x1F790, 0x1F791, 0x1F792, 0x1F793, 0x1F794,
	    0x1F795, 0x1F796, 0x1F797, 0x1F798, 0x1F799, 0x1F79A, 0x1F79B,
	    0x1F79C, 0x1F79D, 0x1F79E, 0x1F79F, 0x1F7A0, 0x1F7A1, 0x1F7A2,
	    0x1F7A3, 0x1F7A4, 0x1F7A5, 0x1F7A6, 0x1F7A7, 0x1F7A8, 0x1F7A9,
	    0x1F7AA, 0x1F7AB, 0x1F7AC, 0x1F7AD, 0x1F7AE, 0x1F7AF, 0x1F7B0,
	    0x1F7B1, 0x1F7B2, 0x1F7B3, 0x1F7B4, 0x1F7B5, 0x1F7B6, 0x1F7B7,
	    0x1F7B8, 0x1F7B9, 0x1F7BA, 0x1F7BB, 0x1F7BC, 0x1F7BD, 0x1F7BE,
	    0x1F7BF, 0x1F7C0, 0x1F7C1, 0x1F7C2, 0x1F7C3, 0x1F7C4, 0x1F7C5,
	    0x1F7C6, 0x1F7C7, 0x1F7C8, 0x1F7C9, 0x1F7CA, 0x1F7CB, 0x1F7CC,
	    0x1F7CD, 0x1F7CE, 0x1F7CF, 0x1F7D0, 0x1F7D1, 0x1F7D2, 0x1F7D3,
	    0x1F7D4, 0x1F7D5, 0x1F7D6, 0x1F7D7, 0x1F7D8, 0x1F7D9, 0x1F7E0,
	    0x1F7E1, 0x1F7E2, 0x1F7E3, 0x1F7E4, 0x1F7E5, 0x1F7E6, 0x1F7E7,
	    0x1F7E8, 0x1F7E9, 0x1F7EA, 0x1F7EB, 0x1F7F0, 0x1F800, 0x1F801,
	    0x1F802, 0x1F803, 0x1F804, 0x1F805, 0x1F806, 0x1F807, 0x1F808,
	    0x1F809, 0x1F80A, 0x1F80B, 0x1F810, 0x1F811, 0x1F812, 0x1F813,
	    0x1F814, 0x1F815, 0x1F816, 0x1F817, 0x1F818, 0x1F819, 0x1F81A,
	    0x1F81B, 0x1F81C, 0x1F81D, 0x1F81E, 0x1F81F, 0x1F820, 0x1F821,
	    0x1F822, 0x1F823, 0x1F824, 0x1F825, 0x1F826, 0x1F827, 0x1F828,
	    0x1F829, 0x1F82A, 0x1F82B, 0x1F82C, 0x1F82D, 0x1F82E, 0x1F82F,
	    0x1F830, 0x1F831, 0x1F832, 0x1F833, 0x1F834, 0x1F835, 0x1F836,
	    0x1F837, 0x1F838, 0x1F839, 0x1F83A, 0x1F83B, 0x1F83C, 0x1F83D,
	    0x1F83E, 0x1F83F, 0x1F840, 0x1F841, 0x1F842, 0x1F843, 0x1F844,
	    0x1F845, 0x1F846, 0x1F847, 0x1F850, 0x1F851, 0x1F852, 0x1F853,
	    0x1F854, 0x1F855, 0x1F856, 0x1F857, 0x1F858, 0x1F859, 0x1F860,
	    0x1F861, 0x1F862, 0x1F863, 0x1F864, 0x1F865, 0x1F866, 0x1F867,
	    0x1F868, 0x1F869, 0x1F86A, 0x1F86B, 0x1F86C, 0x1F86D, 0x1F86E,
	    0x1F86F, 0x1F870, 0x1F871, 0x1F872, 0x1F873, 0x1F874, 0x1F875,
	    0x1F876, 0x1F877, 0x1F878, 0x1F879, 0x1F87A, 0x1F87B, 0x1F87C,
	    0x1F87D, 0x1F87E, 0x1F87F, 0x1F880, 0x1F881, 0x1F882, 0x1F883,
	    0x1F884, 0x1F885, 0x1F886, 0x1F887, 0x1F890, 0x1F891, 0x1F892,
	    0x1F893, 0x1F894, 0x1F895, 0x1F896, 0x1F897, 0x1F898, 0x1F899,
	    0x1F89A, 0x1F89B, 0x1F89C, 0x1F89D, 0x1F89E, 0x1F89F, 0x1F8A0,
	    0x1F8A1, 0x1F8A2, 0x1F8A3, 0x1F8A4, 0x1F8A5, 0x1F8A6, 0x1F8A7,
	    0x1F8A8, 0x1F8A9, 0x1F8AA, 0x1F8AB, 0x1F8AC, 0x1F8AD, 0x1F8B0,
	    0x1F8B1, 0x1F8B2, 0x1F8B3, 0x1F8B4, 0x1F8B5, 0x1F8B6, 0x1F8B7,
	    0x1F8B8, 0x1F8B9, 0x1F8BA, 0x1F8BB, 0x1F8C0, 0x1F8C1, 0x1F900,
	    0x1F901, 0x1F902, 0x1F903, 0x1F904, 0x1F905, 0x1F906, 0x1F907,
	    0x1F908, 0x1F909, 0x1F90A, 0x1F90B, 0x1F90C, 0x1F90D, 0x1F90E,
	    0x1F90F, 0x1F910, 0x1F911, 0x1F912, 0x1F913, 0x1F914, 0x1F915,
	    0x1F916, 0x1F917, 0x1F918, 0x1F919, 0x1F91A, 0x1F91B, 0x1F91C,
	    0x1F91D, 0x1F91E, 0x1F91F, 0x1F920, 0x1F921, 0x1F922, 0x1F923,
	    0x1F924, 0x1F925, 0x1F926, 0x1F927, 0x1F928, 0x1F929, 0x1F92A,
	    0x1F92B, 0x1F92C, 0x1F92D, 0x1F92E, 0x1F92F, 0x1F930, 0x1F931,
	    0x1F932, 0x1F933, 0x1F934, 0x1F935, 0x1F936, 0x1F937, 0x1F938,
	    0x1F939, 0x1F93A, 0x1F93B, 0x1F93C, 0x1F93D, 0x1F93E, 0x1F93F,
	    0x1F940, 0x1F941, 0x1F942, 0x1F943, 0x1F944, 0x1F945, 0x1F946,
	    0x1F947, 0x1F948, 0x1F949, 0x1F94A, 0x1F94B, 0x1F94C, 0x1F94D,
	    0x1F94E, 0x1F94F, 0x1F950, 0x1F951, 0x1F952, 0x1F953, 0x1F954,
	    0x1F955, 0x1F956, 0x1F957, 0x1F958, 0x1F959, 0x1F95A, 0x1F95B,
	    0x1F95C, 0x1F95D, 0x1F95E, 0x1F95F, 0x1F960, 0x1F961, 0x1F962,
	    0x1F963, 0x1F964, 0x1F965, 0x1F966, 0x1F967, 0x1F968, 0x1F969,
	    0x1F96A, 0x1F96B, 0x1F96C, 0x1F96D, 0x1F96E, 0x1F96F, 0x1F970,
	    0x1F971, 0x1F972, 0x1F973, 0x1F974, 0x1F975, 0x1F976, 0x1F977,
	    0x1F978, 0x1F979, 0x1F97A, 0x1F97B, 0x1F97C, 0x1F97D, 0x1F97E,
	    0x1F97F, 0x1F980, 0x1F981, 0x1F982, 0x1F983, 0x1F984, 0x1F985,
	    0x1F986, 0x1F987, 0x1F988, 0x1F989, 0x1F98A, 0x1F98B, 0x1F98C,
	    0x1F98D, 0x1F98E, 0x1F98F, 0x1F990, 0x1F991, 0x1F992, 0x1F993,
	    0x1F994, 0x1F995, 0x1F996, 0x1F997, 0x1F998, 0x1F999, 0x1F99A,
	    0x1F99B, 0x1F99C, 0x1F99D, 0x1F99E, 0x1F99F, 0x1F9A0, 0x1F9A1,
	    0x1F9A2, 0x1F9A3, 0x1F9A4, 0x1F9A5, 0x1F9A6, 0x1F9A7, 0x1F9A8,
	    0x1F9A9, 0x1F9AA, 0x1F9AB, 0x1F9AC, 0x1F9AD, 0x1F9AE, 0x1F9AF,
	    0x1F9B0, 0x1F9B1, 0x1F9B2, 0x1F9B3, 0x1F9B4, 0x1F9B5, 0x1F9B6,
	    0x1F9B7, 0x1F9B8, 0x1F9B9, 0x1F9BA, 0x1F9BB, 0x1F9BC, 0x1F9BD,
	    0x1F9BE, 0x1F9BF, 0x1F9C0, 0x1F9C1, 0x1F9C2, 0x1F9C3, 0x1F9C4,
	    0x1F9C5, 0x1F9C6, 0x1F9C7, 0x1F9C8, 0x1F9C9, 0x1F9CA, 0x1F9CB,
	    0x1F9CC, 0x1F9CD, 0x1F9CE, 0x1F9CF, 0x1F9D0, 0x1F9D1, 0x1F9D2,
	    0x1F9D3, 0x1F9D4, 0x1F9D5, 0x1F9D6, 0x1F9D7, 0x1F9D8, 0x1F9D9,
	    0x1F9DA, 0x1F9DB, 0x1F9DC, 0x1F9DD, 0x1F9DE, 0x1F9DF, 0x1F9E0,
	    0x1F9E1, 0x1F9E2, 0x1F9E3, 0x1F9E4, 0x1F9E5, 0x1F9E6, 0x1F9E7,
	    0x1F9E8, 0x1F9E9, 0x1F9EA, 0x1F9EB, 0x1F9EC, 0x1F9ED, 0x1F9EE,
	    0x1F9EF, 0x1F9F0, 0x1F9F1, 0x1F9F2, 0x1F9F3, 0x1F9F4, 0x1F9F5,
	    0x1F9F6, 0x1F9F7, 0x1F9F8, 0x1F9F9, 0x1F9FA, 0x1F9FB, 0x1F9FC,
	    0x1F9FD, 0x1F9FE, 0x1F9FF, 0x1FA00, 0x1FA01, 0x1FA02, 0x1FA03,
	    0x1FA04, 0x1FA05, 0x1FA06, 0x1FA07, 0x1FA08, 0x1FA09, 0x1FA0A,
	    0x1FA0B, 0x1FA0C, 0x1FA0D, 0x1FA0E, 0x1FA0F, 0x1FA10, 0x1FA11,
	    0x1FA12, 0x1FA13, 0x1FA14, 0x1FA15, 0x1FA16, 0x1FA17, 0x1FA18,
	    0x1FA19, 0x1FA1A, 0x1FA1B, 0x1FA1C, 0x1FA1D, 0x1FA1E, 0x1FA1F,
	    0x1FA20, 0x1FA21, 0x1FA22, 0x1FA23, 0x1FA24, 0x1FA25, 0x1FA26,
	    0x1FA27, 0x1FA28, 0x1FA29, 0x1FA2A, 0x1FA2B, 0x1FA2C, 0x1FA2D,
	    0x1FA2E, 0x1FA2F, 0x1FA30, 0x1FA31, 0x1FA32, 0x1FA33, 0x1FA34,
	    0x1FA35, 0x1FA36, 0x1FA37, 0x1FA38, 0x1FA39, 0x1FA3A, 0x1FA3B,
	    0x1FA3C, 0x1FA3D, 0x1FA3E, 0x1FA3F, 0x1FA40, 0x1FA41, 0x1FA42,
	    0x1FA43, 0x1FA44, 0x1FA45, 0x1FA46, 0x1FA47, 0x1FA48, 0x1FA49,
	    0x1FA4A, 0x1FA4B, 0x1FA4C, 0x1FA4D, 0x1FA4E, 0x1FA4F, 0x1FA50,
	    0x1FA51, 0x1FA52, 0x1FA53, 0x1FA60, 0x1FA61, 0x1FA62, 0x1FA63,
	    0x1FA64, 0x1FA65, 0x1FA66, 0x1FA67, 0x1FA68, 0x1FA69, 0x1FA6A,
	    0x1FA6B, 0x1FA6C, 0x1FA6D, 0x1FA70, 0x1FA71, 0x1FA72, 0x1FA73,
	    0x1FA74, 0x1FA75, 0x1FA76, 0x1FA77, 0x1FA78, 0x1FA79, 0x1FA7A,
	    0x1FA7B, 0x1FA7C, 0x1FA80, 0x1FA81, 0x1FA82, 0x1FA83, 0x1FA84,
	    0x1FA85, 0x1FA86, 0x1FA87, 0x1FA88, 0x1FA89, 0x1FA8F, 0x1FA90,
	    0x1FA91, 0x1FA92, 0x1FA93, 0x1FA94, 0x1FA95, 0x1FA96, 0x1FA97,
	    0x1FA98, 0x1FA99, 0x1FA9A, 0x1FA9B, 0x1FA9C, 0x1FA9D, 0x1FA9E,
	    0x1FA9F, 0x1FAA0, 0x1FAA1, 0x1FAA2, 0x1FAA3, 0x1FAA4, 0x1FAA5,
	    0x1FAA6, 0x1FAA7, 0x1FAA8, 0x1FAA9, 0x1FAAA, 0x1FAAB, 0x1FAAC,
	    0x1FAAD, 0x1FAAE, 0x1FAAF, 0x1FAB0, 0x1FAB1, 0x1FAB2, 0x1FAB3,
	    0x1FAB4, 0x1FAB5, 0x1FAB6, 0x1FAB7, 0x1FAB8, 0x1FAB9, 0x1FABA,
	    0x1FABB, 0x1FABC, 0x1FABD, 0x1FABE, 0x1FABF, 0x1FAC0, 0x1FAC1,
	    0x1FAC2, 0x1FAC3, 0x1FAC4, 0x1FAC5, 0x1FAC6, 0x1FACE, 0x1FACF,
	    0x1FAD0, 0x1FAD1, 0x1FAD2, 0x1FAD3, 0x1FAD4, 0x1FAD5, 0x1FAD6,
	    0x1FAD7, 0x1FAD8, 0x1FAD9, 0x1FADA, 0x1FADB, 0x1FADC, 0x1FADF,
	    0x1FAE0, 0x1FAE1, 0x1FAE2, 0x1FAE3, 0x1FAE4, 0x1FAE5, 0x1FAE6,
	    0x1FAE7, 0x1FAE8, 0x1FAE9, 0x1FAF0, 0x1FAF1, 0x1FAF2, 0x1FAF3,
	    0x1FAF4, 0x1FAF5, 0x1FAF6, 0x1FAF7, 0x1FAF8, 0x1FB00, 0x1FB01,
	    0x1FB02, 0x1FB03, 0x1FB04, 0x1FB05, 0x1FB06, 0x1FB07, 0x1FB08,
	    0x1FB09, 0x1FB0A, 0x1FB0B, 0x1FB0C, 0x1FB0D, 0x1FB0E, 0x1FB0F,
	    0x1FB10, 0x1FB11, 0x1FB12, 0x1FB13, 0x1FB14, 0x1FB15, 0x1FB16,
	    0x1FB17, 0x1FB18, 0x1FB19, 0x1FB1A, 0x1FB1B, 0x1FB1C, 0x1FB1D,
	    0x1FB1E, 0x1FB1F, 0x1FB20, 0x1FB21, 0x1FB22, 0x1FB23, 0x1FB24,
	    0x1FB25, 0x1FB26, 0x1FB27, 0x1FB28, 0x1FB29, 0x1FB2A, 0x1FB2B,
	    0x1FB2C, 0x1FB2D, 0x1FB2E, 0x1FB2F, 0x1FB30, 0x1FB31, 0x1FB32,
	    0x1FB33, 0x1FB34, 0x1FB35, 0x1FB36, 0x1FB37, 0x1FB38, 0x1FB39,
	    0x1FB3A, 0x1FB3B, 0x1FB3C, 0x1FB3D, 0x1FB3E, 0x1FB3F, 0x1FB40,
	    0x1FB41, 0x1FB42, 0x1FB43, 0x1FB44, 0x1FB45, 0x1FB46, 0x1FB47,
	    0x1FB48, 0x1FB49, 0x1FB4A, 0x1FB4B, 0x1FB4C, 0x1FB4D, 0x1FB4E,
	    0x1FB4F, 0x1FB50, 0x1FB51, 0x1FB52, 0x1FB53, 0x1FB54, 0x1FB55,
	    0x1FB56, 0x1FB57, 0x1FB58, 0x1FB59, 0x1FB5A, 0x1FB5B, 0x1FB5C,
	    0x1FB5D, 0x1FB5E, 0x1FB5F, 0x1FB60, 0x1FB61, 0x1FB62, 0x1FB63,
	    0x1FB64, 0x1FB65, 0x1FB66, 0x1FB67, 0x1FB68, 0x1FB69, 0x1FB6A,
	    0x1FB6B, 0x1FB6C, 0x1FB6D, 0x1FB6E, 0x1FB6F, 0x1FB70, 0x1FB71,
	    0x1FB72, 0x1FB73, 0x1FB74, 0x1FB75, 0x1FB76, 0x1FB77, 0x1FB78,
	    0x1FB79, 0x1FB7A, 0x1FB7B, 0x1FB7C, 0x1FB7D, 0x1FB7E, 0x1FB7F,
	    0x1FB80, 0x1FB81, 0x1FB82, 0x1FB83, 0x1FB84, 0x1FB85, 0x1FB86,
	    0x1FB87, 0x1FB88, 0x1FB89, 0x1FB8A, 0x1FB8B, 0x1FB8C, 0x1FB8D,
	    0x1FB8E, 0x1FB8F, 0x1FB90, 0x1FB91, 0x1FB92, 0x1FB94, 0x1FB95,
	    0x1FB96, 0x1FB97, 0x1FB98, 0x1FB99, 0x1FB9A, 0x1FB9B, 0x1FB9C,
	    0x1FB9D, 0x1FB9E, 0x1FB9F, 0x1FBA0, 0x1FBA1, 0x1FBA2, 0x1FBA3,
	    0x1FBA4, 0x1FBA5, 0x1FBA6, 0x1FBA7, 0x1FBA8, 0x1FBA9, 0x1FBAA,
	    0x1FBAB, 0x1FBAC, 0x1FBAD, 0x1FBAE, 0x1FBAF, 0x1FBB0, 0x1FBB1,
	    0x1FBB2, 0x1FBB3, 0x1FBB4, 0x1FBB5, 0x1FBB6, 0x1FBB7, 0x1FBB8,
	    0x1FBB9, 0x1FBBA, 0x1FBBB, 0x1FBBC, 0x1FBBD, 0x1FBBE, 0x1FBBF,
	    0x1FBC0, 0x1FBC1, 0x1FBC2, 0x1FBC3, 0x1FBC4, 0x1FBC5, 0x1FBC6,
	    0x1FBC7, 0x1FBC8, 0x1FBC9, 0x1FBCA, 0x1FBCB, 0x1FBCC, 0x1FBCD,
	    0x1FBCE, 0x1FBCF, 0x1FBD0, 0x1FBD1, 0x1FBD2, 0x1FBD3, 0x1FBD4,
	    0x1FBD5, 0x1FBD6, 0x1FBD7, 0x1FBD8, 0x1FBD9, 0x1FBDA, 0x1FBDB,
	    0x1FBDC, 0x1FBDD, 0x1FBDE, 0x1FBDF, 0x1FBE0, 0x1FBE1, 0x1FBE2,
	    0x1FBE3, 0x1FBE4, 0x1FBE5, 0x1FBE6, 0x1FBE7, 0x1FBE8, 0x1FBE9,
	    0x1FBEA, 0x1FBEB, 0x1FBEC, 0x1FBED, 0x1FBEE, 0x1FBEF
	};
        // Version 3.9; ICU version: 74.1; Unicode/Emoji version: 16.0.0;
    };

    // Produced from https://util.unicode.org/UnicodeJsps/list-unicodeset.jsp,
    // using "[:Ll:]" for the Input field, using the categories found at
    // https://www.fileformat.info/info/unicode/category/index.htm
    template<>
    struct char_set<lower_case_chars>
    {
        static constexpr uint32_t chars[] = {
        0x61,    0x62,    0x63,    0x64,    0x65,    0x66,    0x67,    0x68,
        0x69,    0x6A,    0x6B,    0x6C,    0x6D,    0x6E,    0x6F,    0x70,
        0x71,    0x72,    0x73,    0x74,    0x75,    0x76,    0x77,    0x78,
        0x79,    0x7A,    0xB5,    0xDF,    0xE0,    0xE1,    0xE2,    0xE3,
        0xE4,    0xE5,    0xE6,    0xE7,    0xE8,    0xE9,    0xEA,    0xEB,
        0xEC,    0xED,    0xEE,    0xEF,    0xF0,    0xF1,    0xF2,    0xF3,
        0xF4,    0xF5,    0xF6,    0xF8,    0xF9,    0xFA,    0xFB,    0xFC,
        0xFD,    0xFE,    0xFF,    0x101,   0x103,   0x105,   0x107,   0x109,
        0x10B,   0x10D,   0x10F,   0x111,   0x113,   0x115,   0x117,   0x119,
        0x11B,   0x11D,   0x11F,   0x121,   0x123,   0x125,   0x127,   0x129,
        0x12B,   0x12D,   0x12F,   0x131,   0x133,   0x135,   0x137,   0x138,
        0x13A,   0x13C,   0x13E,   0x140,   0x142,   0x144,   0x146,   0x148,
        0x149,   0x14B,   0x14D,   0x14F,   0x151,   0x153,   0x155,   0x157,
        0x159,   0x15B,   0x15D,   0x15F,   0x161,   0x163,   0x165,   0x167,
        0x169,   0x16B,   0x16D,   0x16F,   0x171,   0x173,   0x175,   0x177,
        0x17A,   0x17C,   0x17E,   0x17F,   0x180,   0x183,   0x185,   0x188,
        0x18C,   0x18D,   0x192,   0x195,   0x199,   0x19A,   0x19B,   0x19E,
        0x1A1,   0x1A3,   0x1A5,   0x1A8,   0x1AA,   0x1AB,   0x1AD,   0x1B0,
        0x1B4,   0x1B6,   0x1B9,   0x1BA,   0x1BD,   0x1BE,   0x1BF,   0x1C6,
        0x1C9,   0x1CC,   0x1CE,   0x1D0,   0x1D2,   0x1D4,   0x1D6,   0x1D8,
        0x1DA,   0x1DC,   0x1DD,   0x1DF,   0x1E1,   0x1E3,   0x1E5,   0x1E7,
        0x1E9,   0x1EB,   0x1ED,   0x1EF,   0x1F0,   0x1F3,   0x1F5,   0x1F9,
        0x1FB,   0x1FD,   0x1FF,   0x201,   0x203,   0x205,   0x207,   0x209,
        0x20B,   0x20D,   0x20F,   0x211,   0x213,   0x215,   0x217,   0x219,
        0x21B,   0x21D,   0x21F,   0x221,   0x223,   0x225,   0x227,   0x229,
        0x22B,   0x22D,   0x22F,   0x231,   0x233,   0x234,   0x235,   0x236,
        0x237,   0x238,   0x239,   0x23C,   0x23F,   0x240,   0x242,   0x247,
        0x249,   0x24B,   0x24D,   0x24F,   0x250,   0x251,   0x252,   0x253,
        0x254,   0x255,   0x256,   0x257,   0x258,   0x259,   0x25A,   0x25B,
        0x25C,   0x25D,   0x25E,   0x25F,   0x260,   0x261,   0x262,   0x263,
        0x264,   0x265,   0x266,   0x267,   0x268,   0x269,   0x26A,   0x26B,
        0x26C,   0x26D,   0x26E,   0x26F,   0x270,   0x271,   0x272,   0x273,
        0x274,   0x275,   0x276,   0x277,   0x278,   0x279,   0x27A,   0x27B,
        0x27C,   0x27D,   0x27E,   0x27F,   0x280,   0x281,   0x282,   0x283,
        0x284,   0x285,   0x286,   0x287,   0x288,   0x289,   0x28A,   0x28B,
        0x28C,   0x28D,   0x28E,   0x28F,   0x290,   0x291,   0x292,   0x293,
        0x295,   0x296,   0x297,   0x298,   0x299,   0x29A,   0x29B,   0x29C,
        0x29D,   0x29E,   0x29F,   0x2A0,   0x2A1,   0x2A2,   0x2A3,   0x2A4,
        0x2A5,   0x2A6,   0x2A7,   0x2A8,   0x2A9,   0x2AA,   0x2AB,   0x2AC,
        0x2AD,   0x2AE,   0x2AF,   0x371,   0x373,   0x377,   0x37B,   0x37C,
        0x37D,   0x390,   0x3AC,   0x3AD,   0x3AE,   0x3AF,   0x3B0,   0x3B1,
        0x3B2,   0x3B3,   0x3B4,   0x3B5,   0x3B6,   0x3B7,   0x3B8,   0x3B9,
        0x3BA,   0x3BB,   0x3BC,   0x3BD,   0x3BE,   0x3BF,   0x3C0,   0x3C1,
        0x3C2,   0x3C3,   0x3C4,   0x3C5,   0x3C6,   0x3C7,   0x3C8,   0x3C9,
        0x3CA,   0x3CB,   0x3CC,   0x3CD,   0x3CE,   0x3D0,   0x3D1,   0x3D5,
        0x3D6,   0x3D7,   0x3D9,   0x3DB,   0x3DD,   0x3DF,   0x3E1,   0x3E3,
        0x3E5,   0x3E7,   0x3E9,   0x3EB,   0x3ED,   0x3EF,   0x3F0,   0x3F1,
        0x3F2,   0x3F3,   0x3F5,   0x3F8,   0x3FB,   0x3FC,   0x430,   0x431,
        0x432,   0x433,   0x434,   0x435,   0x436,   0x437,   0x438,   0x439,
        0x43A,   0x43B,   0x43C,   0x43D,   0x43E,   0x43F,   0x440,   0x441,
        0x442,   0x443,   0x444,   0x445,   0x446,   0x447,   0x448,   0x449,
        0x44A,   0x44B,   0x44C,   0x44D,   0x44E,   0x44F,   0x450,   0x451,
        0x452,   0x453,   0x454,   0x455,   0x456,   0x457,   0x458,   0x459,
        0x45A,   0x45B,   0x45C,   0x45D,   0x45E,   0x45F,   0x461,   0x463,
        0x465,   0x467,   0x469,   0x46B,   0x46D,   0x46F,   0x471,   0x473,
        0x475,   0x477,   0x479,   0x47B,   0x47D,   0x47F,   0x481,   0x48B,
        0x48D,   0x48F,   0x491,   0x493,   0x495,   0x497,   0x499,   0x49B,
        0x49D,   0x49F,   0x4A1,   0x4A3,   0x4A5,   0x4A7,   0x4A9,   0x4AB,
        0x4AD,   0x4AF,   0x4B1,   0x4B3,   0x4B5,   0x4B7,   0x4B9,   0x4BB,
        0x4BD,   0x4BF,   0x4C2,   0x4C4,   0x4C6,   0x4C8,   0x4CA,   0x4CC,
        0x4CE,   0x4CF,   0x4D1,   0x4D3,   0x4D5,   0x4D7,   0x4D9,   0x4DB,
        0x4DD,   0x4DF,   0x4E1,   0x4E3,   0x4E5,   0x4E7,   0x4E9,   0x4EB,
        0x4ED,   0x4EF,   0x4F1,   0x4F3,   0x4F5,   0x4F7,   0x4F9,   0x4FB,
        0x4FD,   0x4FF,   0x501,   0x503,   0x505,   0x507,   0x509,   0x50B,
        0x50D,   0x50F,   0x511,   0x513,   0x515,   0x517,   0x519,   0x51B,
        0x51D,   0x51F,   0x521,   0x523,   0x525,   0x527,   0x529,   0x52B,
        0x52D,   0x52F,   0x560,   0x561,   0x562,   0x563,   0x564,   0x565,
        0x566,   0x567,   0x568,   0x569,   0x56A,   0x56B,   0x56C,   0x56D,
        0x56E,   0x56F,   0x570,   0x571,   0x572,   0x573,   0x574,   0x575,
        0x576,   0x577,   0x578,   0x579,   0x57A,   0x57B,   0x57C,   0x57D,
        0x57E,   0x57F,   0x580,   0x581,   0x582,   0x583,   0x584,   0x585,
        0x586,   0x587,   0x588,   0x10D0,  0x10D1,  0x10D2,  0x10D3,  0x10D4,
        0x10D5,  0x10D6,  0x10D7,  0x10D8,  0x10D9,  0x10DA,  0x10DB,  0x10DC,
        0x10DD,  0x10DE,  0x10DF,  0x10E0,  0x10E1,  0x10E2,  0x10E3,  0x10E4,
        0x10E5,  0x10E6,  0x10E7,  0x10E8,  0x10E9,  0x10EA,  0x10EB,  0x10EC,
        0x10ED,  0x10EE,  0x10EF,  0x10F0,  0x10F1,  0x10F2,  0x10F3,  0x10F4,
        0x10F5,  0x10F6,  0x10F7,  0x10F8,  0x10F9,  0x10FA,  0x10FD,  0x10FE,
        0x10FF,  0x13F8,  0x13F9,  0x13FA,  0x13FB,  0x13FC,  0x13FD,  0x1C80,
        0x1C81,  0x1C82,  0x1C83,  0x1C84,  0x1C85,  0x1C86,  0x1C87,  0x1C88,
        0x1D00,  0x1D01,  0x1D02,  0x1D03,  0x1D04,  0x1D05,  0x1D06,  0x1D07,
        0x1D08,  0x1D09,  0x1D0A,  0x1D0B,  0x1D0C,  0x1D0D,  0x1D0E,  0x1D0F,
        0x1D10,  0x1D11,  0x1D12,  0x1D13,  0x1D14,  0x1D15,  0x1D16,  0x1D17,
        0x1D18,  0x1D19,  0x1D1A,  0x1D1B,  0x1D1C,  0x1D1D,  0x1D1E,  0x1D1F,
        0x1D20,  0x1D21,  0x1D22,  0x1D23,  0x1D24,  0x1D25,  0x1D26,  0x1D27,
        0x1D28,  0x1D29,  0x1D2A,  0x1D2B,  0x1D6B,  0x1D6C,  0x1D6D,  0x1D6E,
        0x1D6F,  0x1D70,  0x1D71,  0x1D72,  0x1D73,  0x1D74,  0x1D75,  0x1D76,
        0x1D77,  0x1D79,  0x1D7A,  0x1D7B,  0x1D7C,  0x1D7D,  0x1D7E,  0x1D7F,
        0x1D80,  0x1D81,  0x1D82,  0x1D83,  0x1D84,  0x1D85,  0x1D86,  0x1D87,
        0x1D88,  0x1D89,  0x1D8A,  0x1D8B,  0x1D8C,  0x1D8D,  0x1D8E,  0x1D8F,
        0x1D90,  0x1D91,  0x1D92,  0x1D93,  0x1D94,  0x1D95,  0x1D96,  0x1D97,
        0x1D98,  0x1D99,  0x1D9A,  0x1E01,  0x1E03,  0x1E05,  0x1E07,  0x1E09,
        0x1E0B,  0x1E0D,  0x1E0F,  0x1E11,  0x1E13,  0x1E15,  0x1E17,  0x1E19,
        0x1E1B,  0x1E1D,  0x1E1F,  0x1E21,  0x1E23,  0x1E25,  0x1E27,  0x1E29,
        0x1E2B,  0x1E2D,  0x1E2F,  0x1E31,  0x1E33,  0x1E35,  0x1E37,  0x1E39,
        0x1E3B,  0x1E3D,  0x1E3F,  0x1E41,  0x1E43,  0x1E45,  0x1E47,  0x1E49,
        0x1E4B,  0x1E4D,  0x1E4F,  0x1E51,  0x1E53,  0x1E55,  0x1E57,  0x1E59,
        0x1E5B,  0x1E5D,  0x1E5F,  0x1E61,  0x1E63,  0x1E65,  0x1E67,  0x1E69,
        0x1E6B,  0x1E6D,  0x1E6F,  0x1E71,  0x1E73,  0x1E75,  0x1E77,  0x1E79,
        0x1E7B,  0x1E7D,  0x1E7F,  0x1E81,  0x1E83,  0x1E85,  0x1E87,  0x1E89,
        0x1E8B,  0x1E8D,  0x1E8F,  0x1E91,  0x1E93,  0x1E95,  0x1E96,  0x1E97,
        0x1E98,  0x1E99,  0x1E9A,  0x1E9B,  0x1E9C,  0x1E9D,  0x1E9F,  0x1EA1,
        0x1EA3,  0x1EA5,  0x1EA7,  0x1EA9,  0x1EAB,  0x1EAD,  0x1EAF,  0x1EB1,
        0x1EB3,  0x1EB5,  0x1EB7,  0x1EB9,  0x1EBB,  0x1EBD,  0x1EBF,  0x1EC1,
        0x1EC3,  0x1EC5,  0x1EC7,  0x1EC9,  0x1ECB,  0x1ECD,  0x1ECF,  0x1ED1,
        0x1ED3,  0x1ED5,  0x1ED7,  0x1ED9,  0x1EDB,  0x1EDD,  0x1EDF,  0x1EE1,
        0x1EE3,  0x1EE5,  0x1EE7,  0x1EE9,  0x1EEB,  0x1EED,  0x1EEF,  0x1EF1,
        0x1EF3,  0x1EF5,  0x1EF7,  0x1EF9,  0x1EFB,  0x1EFD,  0x1EFF,  0x1F00,
        0x1F01,  0x1F02,  0x1F03,  0x1F04,  0x1F05,  0x1F06,  0x1F07,  0x1F10,
        0x1F11,  0x1F12,  0x1F13,  0x1F14,  0x1F15,  0x1F20,  0x1F21,  0x1F22,
        0x1F23,  0x1F24,  0x1F25,  0x1F26,  0x1F27,  0x1F30,  0x1F31,  0x1F32,
        0x1F33,  0x1F34,  0x1F35,  0x1F36,  0x1F37,  0x1F40,  0x1F41,  0x1F42,
        0x1F43,  0x1F44,  0x1F45,  0x1F50,  0x1F51,  0x1F52,  0x1F53,  0x1F54,
        0x1F55,  0x1F56,  0x1F57,  0x1F60,  0x1F61,  0x1F62,  0x1F63,  0x1F64,
        0x1F65,  0x1F66,  0x1F67,  0x1F70,  0x1F71,  0x1F72,  0x1F73,  0x1F74,
        0x1F75,  0x1F76,  0x1F77,  0x1F78,  0x1F79,  0x1F7A,  0x1F7B,  0x1F7C,
        0x1F7D,  0x1F80,  0x1F81,  0x1F82,  0x1F83,  0x1F84,  0x1F85,  0x1F86,
        0x1F87,  0x1F90,  0x1F91,  0x1F92,  0x1F93,  0x1F94,  0x1F95,  0x1F96,
        0x1F97,  0x1FA0,  0x1FA1,  0x1FA2,  0x1FA3,  0x1FA4,  0x1FA5,  0x1FA6,
        0x1FA7,  0x1FB0,  0x1FB1,  0x1FB2,  0x1FB3,  0x1FB4,  0x1FB6,  0x1FB7,
        0x1FBE,  0x1FC2,  0x1FC3,  0x1FC4,  0x1FC6,  0x1FC7,  0x1FD0,  0x1FD1,
        0x1FD2,  0x1FD3,  0x1FD6,  0x1FD7,  0x1FE0,  0x1FE1,  0x1FE2,  0x1FE3,
        0x1FE4,  0x1FE5,  0x1FE6,  0x1FE7,  0x1FF2,  0x1FF3,  0x1FF4,  0x1FF6,
        0x1FF7,  0x210A,  0x210E,  0x210F,  0x2113,  0x212F,  0x2134,  0x2139,
        0x213C,  0x213D,  0x2146,  0x2147,  0x2148,  0x2149,  0x214E,  0x2184,
        0x2C30,  0x2C31,  0x2C32,  0x2C33,  0x2C34,  0x2C35,  0x2C36,  0x2C37,
        0x2C38,  0x2C39,  0x2C3A,  0x2C3B,  0x2C3C,  0x2C3D,  0x2C3E,  0x2C3F,
        0x2C40,  0x2C41,  0x2C42,  0x2C43,  0x2C44,  0x2C45,  0x2C46,  0x2C47,
        0x2C48,  0x2C49,  0x2C4A,  0x2C4B,  0x2C4C,  0x2C4D,  0x2C4E,  0x2C4F,
        0x2C50,  0x2C51,  0x2C52,  0x2C53,  0x2C54,  0x2C55,  0x2C56,  0x2C57,
        0x2C58,  0x2C59,  0x2C5A,  0x2C5B,  0x2C5C,  0x2C5D,  0x2C5E,  0x2C5F,
        0x2C61,  0x2C65,  0x2C66,  0x2C68,  0x2C6A,  0x2C6C,  0x2C71,  0x2C73,
        0x2C74,  0x2C76,  0x2C77,  0x2C78,  0x2C79,  0x2C7A,  0x2C7B,  0x2C81,
        0x2C83,  0x2C85,  0x2C87,  0x2C89,  0x2C8B,  0x2C8D,  0x2C8F,  0x2C91,
        0x2C93,  0x2C95,  0x2C97,  0x2C99,  0x2C9B,  0x2C9D,  0x2C9F,  0x2CA1,
        0x2CA3,  0x2CA5,  0x2CA7,  0x2CA9,  0x2CAB,  0x2CAD,  0x2CAF,  0x2CB1,
        0x2CB3,  0x2CB5,  0x2CB7,  0x2CB9,  0x2CBB,  0x2CBD,  0x2CBF,  0x2CC1,
        0x2CC3,  0x2CC5,  0x2CC7,  0x2CC9,  0x2CCB,  0x2CCD,  0x2CCF,  0x2CD1,
        0x2CD3,  0x2CD5,  0x2CD7,  0x2CD9,  0x2CDB,  0x2CDD,  0x2CDF,  0x2CE1,
        0x2CE3,  0x2CE4,  0x2CEC,  0x2CEE,  0x2CF3,  0x2D00,  0x2D01,  0x2D02,
        0x2D03,  0x2D04,  0x2D05,  0x2D06,  0x2D07,  0x2D08,  0x2D09,  0x2D0A,
        0x2D0B,  0x2D0C,  0x2D0D,  0x2D0E,  0x2D0F,  0x2D10,  0x2D11,  0x2D12,
        0x2D13,  0x2D14,  0x2D15,  0x2D16,  0x2D17,  0x2D18,  0x2D19,  0x2D1A,
        0x2D1B,  0x2D1C,  0x2D1D,  0x2D1E,  0x2D1F,  0x2D20,  0x2D21,  0x2D22,
        0x2D23,  0x2D24,  0x2D25,  0x2D27,  0x2D2D,  0xA641,  0xA643,  0xA645,
        0xA647,  0xA649,  0xA64B,  0xA64D,  0xA64F,  0xA651,  0xA653,  0xA655,
        0xA657,  0xA659,  0xA65B,  0xA65D,  0xA65F,  0xA661,  0xA663,  0xA665,
        0xA667,  0xA669,  0xA66B,  0xA66D,  0xA681,  0xA683,  0xA685,  0xA687,
        0xA689,  0xA68B,  0xA68D,  0xA68F,  0xA691,  0xA693,  0xA695,  0xA697,
        0xA699,  0xA69B,  0xA723,  0xA725,  0xA727,  0xA729,  0xA72B,  0xA72D,
        0xA72F,  0xA730,  0xA731,  0xA733,  0xA735,  0xA737,  0xA739,  0xA73B,
        0xA73D,  0xA73F,  0xA741,  0xA743,  0xA745,  0xA747,  0xA749,  0xA74B,
        0xA74D,  0xA74F,  0xA751,  0xA753,  0xA755,  0xA757,  0xA759,  0xA75B,
        0xA75D,  0xA75F,  0xA761,  0xA763,  0xA765,  0xA767,  0xA769,  0xA76B,
        0xA76D,  0xA76F,  0xA771,  0xA772,  0xA773,  0xA774,  0xA775,  0xA776,
        0xA777,  0xA778,  0xA77A,  0xA77C,  0xA77F,  0xA781,  0xA783,  0xA785,
        0xA787,  0xA78C,  0xA78E,  0xA791,  0xA793,  0xA794,  0xA795,  0xA797,
        0xA799,  0xA79B,  0xA79D,  0xA79F,  0xA7A1,  0xA7A3,  0xA7A5,  0xA7A7,
        0xA7A9,  0xA7AF,  0xA7B5,  0xA7B7,  0xA7B9,  0xA7BB,  0xA7BD,  0xA7BF,
        0xA7C1,  0xA7C3,  0xA7C8,  0xA7CA,  0xA7D1,  0xA7D3,  0xA7D5,  0xA7D7,
        0xA7D9,  0xA7F6,  0xA7FA,  0xAB30,  0xAB31,  0xAB32,  0xAB33,  0xAB34,
        0xAB35,  0xAB36,  0xAB37,  0xAB38,  0xAB39,  0xAB3A,  0xAB3B,  0xAB3C,
        0xAB3D,  0xAB3E,  0xAB3F,  0xAB40,  0xAB41,  0xAB42,  0xAB43,  0xAB44,
        0xAB45,  0xAB46,  0xAB47,  0xAB48,  0xAB49,  0xAB4A,  0xAB4B,  0xAB4C,
        0xAB4D,  0xAB4E,  0xAB4F,  0xAB50,  0xAB51,  0xAB52,  0xAB53,  0xAB54,
        0xAB55,  0xAB56,  0xAB57,  0xAB58,  0xAB59,  0xAB5A,  0xAB60,  0xAB61,
        0xAB62,  0xAB63,  0xAB64,  0xAB65,  0xAB66,  0xAB67,  0xAB68,  0xAB70,
        0xAB71,  0xAB72,  0xAB73,  0xAB74,  0xAB75,  0xAB76,  0xAB77,  0xAB78,
        0xAB79,  0xAB7A,  0xAB7B,  0xAB7C,  0xAB7D,  0xAB7E,  0xAB7F,  0xAB80,
        0xAB81,  0xAB82,  0xAB83,  0xAB84,  0xAB85,  0xAB86,  0xAB87,  0xAB88,
        0xAB89,  0xAB8A,  0xAB8B,  0xAB8C,  0xAB8D,  0xAB8E,  0xAB8F,  0xAB90,
        0xAB91,  0xAB92,  0xAB93,  0xAB94,  0xAB95,  0xAB96,  0xAB97,  0xAB98,
        0xAB99,  0xAB9A,  0xAB9B,  0xAB9C,  0xAB9D,  0xAB9E,  0xAB9F,  0xABA0,
        0xABA1,  0xABA2,  0xABA3,  0xABA4,  0xABA5,  0xABA6,  0xABA7,  0xABA8,
        0xABA9,  0xABAA,  0xABAB,  0xABAC,  0xABAD,  0xABAE,  0xABAF,  0xABB0,
        0xABB1,  0xABB2,  0xABB3,  0xABB4,  0xABB5,  0xABB6,  0xABB7,  0xABB8,
        0xABB9,  0xABBA,  0xABBB,  0xABBC,  0xABBD,  0xABBE,  0xABBF,  0xFB00,
        0xFB01,  0xFB02,  0xFB03,  0xFB04,  0xFB05,  0xFB06,  0xFB13,  0xFB14,
        0xFB15,  0xFB16,  0xFB17,  0xFF41,  0xFF42,  0xFF43,  0xFF44,  0xFF45,
        0xFF46,  0xFF47,  0xFF48,  0xFF49,  0xFF4A,  0xFF4B,  0xFF4C,  0xFF4D,
        0xFF4E,  0xFF4F,  0xFF50,  0xFF51,  0xFF52,  0xFF53,  0xFF54,  0xFF55,
        0xFF56,  0xFF57,  0xFF58,  0xFF59,  0xFF5A,  0x10428, 0x10429, 0x1042A,
        0x1042B, 0x1042C, 0x1042D, 0x1042E, 0x1042F, 0x10430, 0x10431, 0x10432,
        0x10433, 0x10434, 0x10435, 0x10436, 0x10437, 0x10438, 0x10439, 0x1043A,
        0x1043B, 0x1043C, 0x1043D, 0x1043E, 0x1043F, 0x10440, 0x10441, 0x10442,
        0x10443, 0x10444, 0x10445, 0x10446, 0x10447, 0x10448, 0x10449, 0x1044A,
        0x1044B, 0x1044C, 0x1044D, 0x1044E, 0x1044F, 0x104D8, 0x104D9, 0x104DA,
        0x104DB, 0x104DC, 0x104DD, 0x104DE, 0x104DF, 0x104E0, 0x104E1, 0x104E2,
        0x104E3, 0x104E4, 0x104E5, 0x104E6, 0x104E7, 0x104E8, 0x104E9, 0x104EA,
        0x104EB, 0x104EC, 0x104ED, 0x104EE, 0x104EF, 0x104F0, 0x104F1, 0x104F2,
        0x104F3, 0x104F4, 0x104F5, 0x104F6, 0x104F7, 0x104F8, 0x104F9, 0x104FA,
        0x104FB, 0x10597, 0x10598, 0x10599, 0x1059A, 0x1059B, 0x1059C, 0x1059D,
        0x1059E, 0x1059F, 0x105A0, 0x105A1, 0x105A3, 0x105A4, 0x105A5, 0x105A6,
        0x105A7, 0x105A8, 0x105A9, 0x105AA, 0x105AB, 0x105AC, 0x105AD, 0x105AE,
        0x105AF, 0x105B0, 0x105B1, 0x105B3, 0x105B4, 0x105B5, 0x105B6, 0x105B7,
        0x105B8, 0x105B9, 0x105BB, 0x105BC, 0x10CC0, 0x10CC1, 0x10CC2, 0x10CC3,
        0x10CC4, 0x10CC5, 0x10CC6, 0x10CC7, 0x10CC8, 0x10CC9, 0x10CCA, 0x10CCB,
        0x10CCC, 0x10CCD, 0x10CCE, 0x10CCF, 0x10CD0, 0x10CD1, 0x10CD2, 0x10CD3,
        0x10CD4, 0x10CD5, 0x10CD6, 0x10CD7, 0x10CD8, 0x10CD9, 0x10CDA, 0x10CDB,
        0x10CDC, 0x10CDD, 0x10CDE, 0x10CDF, 0x10CE0, 0x10CE1, 0x10CE2, 0x10CE3,
        0x10CE4, 0x10CE5, 0x10CE6, 0x10CE7, 0x10CE8, 0x10CE9, 0x10CEA, 0x10CEB,
        0x10CEC, 0x10CED, 0x10CEE, 0x10CEF, 0x10CF0, 0x10CF1, 0x10CF2, 0x118C0,
        0x118C1, 0x118C2, 0x118C3, 0x118C4, 0x118C5, 0x118C6, 0x118C7, 0x118C8,
        0x118C9, 0x118CA, 0x118CB, 0x118CC, 0x118CD, 0x118CE, 0x118CF, 0x118D0,
        0x118D1, 0x118D2, 0x118D3, 0x118D4, 0x118D5, 0x118D6, 0x118D7, 0x118D8,
        0x118D9, 0x118DA, 0x118DB, 0x118DC, 0x118DD, 0x118DE, 0x118DF, 0x16E60,
        0x16E61, 0x16E62, 0x16E63, 0x16E64, 0x16E65, 0x16E66, 0x16E67, 0x16E68,
        0x16E69, 0x16E6A, 0x16E6B, 0x16E6C, 0x16E6D, 0x16E6E, 0x16E6F, 0x16E70,
        0x16E71, 0x16E72, 0x16E73, 0x16E74, 0x16E75, 0x16E76, 0x16E77, 0x16E78,
        0x16E79, 0x16E7A, 0x16E7B, 0x16E7C, 0x16E7D, 0x16E7E, 0x16E7F, 0x1D41A,
        0x1D41B, 0x1D41C, 0x1D41D, 0x1D41E, 0x1D41F, 0x1D420, 0x1D421, 0x1D422,
        0x1D423, 0x1D424, 0x1D425, 0x1D426, 0x1D427, 0x1D428, 0x1D429, 0x1D42A,
        0x1D42B, 0x1D42C, 0x1D42D, 0x1D42E, 0x1D42F, 0x1D430, 0x1D431, 0x1D432,
        0x1D433, 0x1D44E, 0x1D44F, 0x1D450, 0x1D451, 0x1D452, 0x1D453, 0x1D454,
        0x1D456, 0x1D457, 0x1D458, 0x1D459, 0x1D45A, 0x1D45B, 0x1D45C, 0x1D45D,
        0x1D45E, 0x1D45F, 0x1D460, 0x1D461, 0x1D462, 0x1D463, 0x1D464, 0x1D465,
        0x1D466, 0x1D467, 0x1D482, 0x1D483, 0x1D484, 0x1D485, 0x1D486, 0x1D487,
        0x1D488, 0x1D489, 0x1D48A, 0x1D48B, 0x1D48C, 0x1D48D, 0x1D48E, 0x1D48F,
        0x1D490, 0x1D491, 0x1D492, 0x1D493, 0x1D494, 0x1D495, 0x1D496, 0x1D497,
        0x1D498, 0x1D499, 0x1D49A, 0x1D49B, 0x1D4B6, 0x1D4B7, 0x1D4B8, 0x1D4B9,
        0x1D4BB, 0x1D4BD, 0x1D4BE, 0x1D4BF, 0x1D4C0, 0x1D4C1, 0x1D4C2, 0x1D4C3,
        0x1D4C5, 0x1D4C6, 0x1D4C7, 0x1D4C8, 0x1D4C9, 0x1D4CA, 0x1D4CB, 0x1D4CC,
        0x1D4CD, 0x1D4CE, 0x1D4CF, 0x1D4EA, 0x1D4EB, 0x1D4EC, 0x1D4ED, 0x1D4EE,
        0x1D4EF, 0x1D4F0, 0x1D4F1, 0x1D4F2, 0x1D4F3, 0x1D4F4, 0x1D4F5, 0x1D4F6,
        0x1D4F7, 0x1D4F8, 0x1D4F9, 0x1D4FA, 0x1D4FB, 0x1D4FC, 0x1D4FD, 0x1D4FE,
        0x1D4FF, 0x1D500, 0x1D501, 0x1D502, 0x1D503, 0x1D51E, 0x1D51F, 0x1D520,
        0x1D521, 0x1D522, 0x1D523, 0x1D524, 0x1D525, 0x1D526, 0x1D527, 0x1D528,
        0x1D529, 0x1D52A, 0x1D52B, 0x1D52C, 0x1D52D, 0x1D52E, 0x1D52F, 0x1D530,
        0x1D531, 0x1D532, 0x1D533, 0x1D534, 0x1D535, 0x1D536, 0x1D537, 0x1D552,
        0x1D553, 0x1D554, 0x1D555, 0x1D556, 0x1D557, 0x1D558, 0x1D559, 0x1D55A,
        0x1D55B, 0x1D55C, 0x1D55D, 0x1D55E, 0x1D55F, 0x1D560, 0x1D561, 0x1D562,
        0x1D563, 0x1D564, 0x1D565, 0x1D566, 0x1D567, 0x1D568, 0x1D569, 0x1D56A,
        0x1D56B, 0x1D586, 0x1D587, 0x1D588, 0x1D589, 0x1D58A, 0x1D58B, 0x1D58C,
        0x1D58D, 0x1D58E, 0x1D58F, 0x1D590, 0x1D591, 0x1D592, 0x1D593, 0x1D594,
        0x1D595, 0x1D596, 0x1D597, 0x1D598, 0x1D599, 0x1D59A, 0x1D59B, 0x1D59C,
        0x1D59D, 0x1D59E, 0x1D59F, 0x1D5BA, 0x1D5BB, 0x1D5BC, 0x1D5BD, 0x1D5BE,
        0x1D5BF, 0x1D5C0, 0x1D5C1, 0x1D5C2, 0x1D5C3, 0x1D5C4, 0x1D5C5, 0x1D5C6,
        0x1D5C7, 0x1D5C8, 0x1D5C9, 0x1D5CA, 0x1D5CB, 0x1D5CC, 0x1D5CD, 0x1D5CE,
        0x1D5CF, 0x1D5D0, 0x1D5D1, 0x1D5D2, 0x1D5D3, 0x1D5EE, 0x1D5EF, 0x1D5F0,
        0x1D5F1, 0x1D5F2, 0x1D5F3, 0x1D5F4, 0x1D5F5, 0x1D5F6, 0x1D5F7, 0x1D5F8,
        0x1D5F9, 0x1D5FA, 0x1D5FB, 0x1D5FC, 0x1D5FD, 0x1D5FE, 0x1D5FF, 0x1D600,
        0x1D601, 0x1D602, 0x1D603, 0x1D604, 0x1D605, 0x1D606, 0x1D607, 0x1D622,
        0x1D623, 0x1D624, 0x1D625, 0x1D626, 0x1D627, 0x1D628, 0x1D629, 0x1D62A,
        0x1D62B, 0x1D62C, 0x1D62D, 0x1D62E, 0x1D62F, 0x1D630, 0x1D631, 0x1D632,
        0x1D633, 0x1D634, 0x1D635, 0x1D636, 0x1D637, 0x1D638, 0x1D639, 0x1D63A,
        0x1D63B, 0x1D656, 0x1D657, 0x1D658, 0x1D659, 0x1D65A, 0x1D65B, 0x1D65C,
        0x1D65D, 0x1D65E, 0x1D65F, 0x1D660, 0x1D661, 0x1D662, 0x1D663, 0x1D664,
        0x1D665, 0x1D666, 0x1D667, 0x1D668, 0x1D669, 0x1D66A, 0x1D66B, 0x1D66C,
        0x1D66D, 0x1D66E, 0x1D66F, 0x1D68A, 0x1D68B, 0x1D68C, 0x1D68D, 0x1D68E,
        0x1D68F, 0x1D690, 0x1D691, 0x1D692, 0x1D693, 0x1D694, 0x1D695, 0x1D696,
        0x1D697, 0x1D698, 0x1D699, 0x1D69A, 0x1D69B, 0x1D69C, 0x1D69D, 0x1D69E,
        0x1D69F, 0x1D6A0, 0x1D6A1, 0x1D6A2, 0x1D6A3, 0x1D6A4, 0x1D6A5, 0x1D6C2,
        0x1D6C3, 0x1D6C4, 0x1D6C5, 0x1D6C6, 0x1D6C7, 0x1D6C8, 0x1D6C9, 0x1D6CA,
        0x1D6CB, 0x1D6CC, 0x1D6CD, 0x1D6CE, 0x1D6CF, 0x1D6D0, 0x1D6D1, 0x1D6D2,
        0x1D6D3, 0x1D6D4, 0x1D6D5, 0x1D6D6, 0x1D6D7, 0x1D6D8, 0x1D6D9, 0x1D6DA,
        0x1D6DC, 0x1D6DD, 0x1D6DE, 0x1D6DF, 0x1D6E0, 0x1D6E1, 0x1D6FC, 0x1D6FD,
        0x1D6FE, 0x1D6FF, 0x1D700, 0x1D701, 0x1D702, 0x1D703, 0x1D704, 0x1D705,
        0x1D706, 0x1D707, 0x1D708, 0x1D709, 0x1D70A, 0x1D70B, 0x1D70C, 0x1D70D,
        0x1D70E, 0x1D70F, 0x1D710, 0x1D711, 0x1D712, 0x1D713, 0x1D714, 0x1D716,
        0x1D717, 0x1D718, 0x1D719, 0x1D71A, 0x1D71B, 0x1D736, 0x1D737, 0x1D738,
        0x1D739, 0x1D73A, 0x1D73B, 0x1D73C, 0x1D73D, 0x1D73E, 0x1D73F, 0x1D740,
        0x1D741, 0x1D742, 0x1D743, 0x1D744, 0x1D745, 0x1D746, 0x1D747, 0x1D748,
        0x1D749, 0x1D74A, 0x1D74B, 0x1D74C, 0x1D74D, 0x1D74E, 0x1D750, 0x1D751,
        0x1D752, 0x1D753, 0x1D754, 0x1D755, 0x1D770, 0x1D771, 0x1D772, 0x1D773,
        0x1D774, 0x1D775, 0x1D776, 0x1D777, 0x1D778, 0x1D779, 0x1D77A, 0x1D77B,
        0x1D77C, 0x1D77D, 0x1D77E, 0x1D77F, 0x1D780, 0x1D781, 0x1D782, 0x1D783,
        0x1D784, 0x1D785, 0x1D786, 0x1D787, 0x1D788, 0x1D78A, 0x1D78B, 0x1D78C,
        0x1D78D, 0x1D78E, 0x1D78F, 0x1D7AA, 0x1D7AB, 0x1D7AC, 0x1D7AD, 0x1D7AE,
        0x1D7AF, 0x1D7B0, 0x1D7B1, 0x1D7B2, 0x1D7B3, 0x1D7B4, 0x1D7B5, 0x1D7B6,
        0x1D7B7, 0x1D7B8, 0x1D7B9, 0x1D7BA, 0x1D7BB, 0x1D7BC, 0x1D7BD, 0x1D7BE,
        0x1D7BF, 0x1D7C0, 0x1D7C1, 0x1D7C2, 0x1D7C4, 0x1D7C5, 0x1D7C6, 0x1D7C7,
        0x1D7C8, 0x1D7C9, 0x1D7CB, 0x1DF00, 0x1DF01, 0x1DF02, 0x1DF03, 0x1DF04,
        0x1DF05, 0x1DF06, 0x1DF07, 0x1DF08, 0x1DF09, 0x1DF0B, 0x1DF0C, 0x1DF0D,
        0x1DF0E, 0x1DF0F, 0x1DF10, 0x1DF11, 0x1DF12, 0x1DF13, 0x1DF14, 0x1DF15,
        0x1DF16, 0x1DF17, 0x1DF18, 0x1DF19, 0x1DF1A, 0x1DF1B, 0x1DF1C, 0x1DF1D,
        0x1DF1E, 0x1DF25, 0x1DF26, 0x1DF27, 0x1DF28, 0x1DF29, 0x1DF2A, 0x1E922,
        0x1E923, 0x1E924, 0x1E925, 0x1E926, 0x1E927, 0x1E928, 0x1E929, 0x1E92A,
        0x1E92B, 0x1E92C, 0x1E92D, 0x1E92E, 0x1E92F, 0x1E930, 0x1E931, 0x1E932,
        0x1E933, 0x1E934, 0x1E935, 0x1E936, 0x1E937, 0x1E938, 0x1E939, 0x1E93A,
        0x1E93B, 0x1E93C, 0x1E93D, 0x1E93E, 0x1E93F, 0x1E940, 0x1E941, 0x1E942,
        0x1E943};
    };

    // Produced from https://util.unicode.org/UnicodeJsps/list-unicodeset.jsp,
    // using "[:Lu:]" for the Input field, using the categories found at
    // https://www.fileformat.info/info/unicode/category/index.htm
    template<>
    struct char_set<upper_case_chars>
    {
        static constexpr uint32_t chars[] = {
            0x41,    0x42,    0x43,    0x44,    0x45,    0x46,    0x47,
            0x48,    0x49,    0x4A,    0x4B,    0x4C,    0x4D,    0x4E,
            0x4F,    0x50,    0x51,    0x52,    0x53,    0x54,    0x55,
            0x56,    0x57,    0x58,    0x59,    0x5A,    0xC0,    0xC1,
            0xC2,    0xC3,    0xC4,    0xC5,    0xC6,    0xC7,    0xC8,
            0xC9,    0xCA,    0xCB,    0xCC,    0xCD,    0xCE,    0xCF,
            0xD0,    0xD1,    0xD2,    0xD3,    0xD4,    0xD5,    0xD6,
            0xD8,    0xD9,    0xDA,    0xDB,    0xDC,    0xDD,    0xDE,
            0x100,   0x102,   0x104,   0x106,   0x108,   0x10A,   0x10C,
            0x10E,   0x110,   0x112,   0x114,   0x116,   0x118,   0x11A,
            0x11C,   0x11E,   0x120,   0x122,   0x124,   0x126,   0x128,
            0x12A,   0x12C,   0x12E,   0x130,   0x132,   0x134,   0x136,
            0x139,   0x13B,   0x13D,   0x13F,   0x141,   0x143,   0x145,
            0x147,   0x14A,   0x14C,   0x14E,   0x150,   0x152,   0x154,
            0x156,   0x158,   0x15A,   0x15C,   0x15E,   0x160,   0x162,
            0x164,   0x166,   0x168,   0x16A,   0x16C,   0x16E,   0x170,
            0x172,   0x174,   0x176,   0x178,   0x179,   0x17B,   0x17D,
            0x181,   0x182,   0x184,   0x186,   0x187,   0x189,   0x18A,
            0x18B,   0x18E,   0x18F,   0x190,   0x191,   0x193,   0x194,
            0x196,   0x197,   0x198,   0x19C,   0x19D,   0x19F,   0x1A0,
            0x1A2,   0x1A4,   0x1A6,   0x1A7,   0x1A9,   0x1AC,   0x1AE,
            0x1AF,   0x1B1,   0x1B2,   0x1B3,   0x1B5,   0x1B7,   0x1B8,
            0x1BC,   0x1C4,   0x1C7,   0x1CA,   0x1CD,   0x1CF,   0x1D1,
            0x1D3,   0x1D5,   0x1D7,   0x1D9,   0x1DB,   0x1DE,   0x1E0,
            0x1E2,   0x1E4,   0x1E6,   0x1E8,   0x1EA,   0x1EC,   0x1EE,
            0x1F1,   0x1F4,   0x1F6,   0x1F7,   0x1F8,   0x1FA,   0x1FC,
            0x1FE,   0x200,   0x202,   0x204,   0x206,   0x208,   0x20A,
            0x20C,   0x20E,   0x210,   0x212,   0x214,   0x216,   0x218,
            0x21A,   0x21C,   0x21E,   0x220,   0x222,   0x224,   0x226,
            0x228,   0x22A,   0x22C,   0x22E,   0x230,   0x232,   0x23A,
            0x23B,   0x23D,   0x23E,   0x241,   0x243,   0x244,   0x245,
            0x246,   0x248,   0x24A,   0x24C,   0x24E,   0x370,   0x372,
            0x376,   0x37F,   0x386,   0x388,   0x389,   0x38A,   0x38C,
            0x38E,   0x38F,   0x391,   0x392,   0x393,   0x394,   0x395,
            0x396,   0x397,   0x398,   0x399,   0x39A,   0x39B,   0x39C,
            0x39D,   0x39E,   0x39F,   0x3A0,   0x3A1,   0x3A3,   0x3A4,
            0x3A5,   0x3A6,   0x3A7,   0x3A8,   0x3A9,   0x3AA,   0x3AB,
            0x3CF,   0x3D2,   0x3D3,   0x3D4,   0x3D8,   0x3DA,   0x3DC,
            0x3DE,   0x3E0,   0x3E2,   0x3E4,   0x3E6,   0x3E8,   0x3EA,
            0x3EC,   0x3EE,   0x3F4,   0x3F7,   0x3F9,   0x3FA,   0x3FD,
            0x3FE,   0x3FF,   0x400,   0x401,   0x402,   0x403,   0x404,
            0x405,   0x406,   0x407,   0x408,   0x409,   0x40A,   0x40B,
            0x40C,   0x40D,   0x40E,   0x40F,   0x410,   0x411,   0x412,
            0x413,   0x414,   0x415,   0x416,   0x417,   0x418,   0x419,
            0x41A,   0x41B,   0x41C,   0x41D,   0x41E,   0x41F,   0x420,
            0x421,   0x422,   0x423,   0x424,   0x425,   0x426,   0x427,
            0x428,   0x429,   0x42A,   0x42B,   0x42C,   0x42D,   0x42E,
            0x42F,   0x460,   0x462,   0x464,   0x466,   0x468,   0x46A,
            0x46C,   0x46E,   0x470,   0x472,   0x474,   0x476,   0x478,
            0x47A,   0x47C,   0x47E,   0x480,   0x48A,   0x48C,   0x48E,
            0x490,   0x492,   0x494,   0x496,   0x498,   0x49A,   0x49C,
            0x49E,   0x4A0,   0x4A2,   0x4A4,   0x4A6,   0x4A8,   0x4AA,
            0x4AC,   0x4AE,   0x4B0,   0x4B2,   0x4B4,   0x4B6,   0x4B8,
            0x4BA,   0x4BC,   0x4BE,   0x4C0,   0x4C1,   0x4C3,   0x4C5,
            0x4C7,   0x4C9,   0x4CB,   0x4CD,   0x4D0,   0x4D2,   0x4D4,
            0x4D6,   0x4D8,   0x4DA,   0x4DC,   0x4DE,   0x4E0,   0x4E2,
            0x4E4,   0x4E6,   0x4E8,   0x4EA,   0x4EC,   0x4EE,   0x4F0,
            0x4F2,   0x4F4,   0x4F6,   0x4F8,   0x4FA,   0x4FC,   0x4FE,
            0x500,   0x502,   0x504,   0x506,   0x508,   0x50A,   0x50C,
            0x50E,   0x510,   0x512,   0x514,   0x516,   0x518,   0x51A,
            0x51C,   0x51E,   0x520,   0x522,   0x524,   0x526,   0x528,
            0x52A,   0x52C,   0x52E,   0x531,   0x532,   0x533,   0x534,
            0x535,   0x536,   0x537,   0x538,   0x539,   0x53A,   0x53B,
            0x53C,   0x53D,   0x53E,   0x53F,   0x540,   0x541,   0x542,
            0x543,   0x544,   0x545,   0x546,   0x547,   0x548,   0x549,
            0x54A,   0x54B,   0x54C,   0x54D,   0x54E,   0x54F,   0x550,
            0x551,   0x552,   0x553,   0x554,   0x555,   0x556,   0x10A0,
            0x10A1,  0x10A2,  0x10A3,  0x10A4,  0x10A5,  0x10A6,  0x10A7,
            0x10A8,  0x10A9,  0x10AA,  0x10AB,  0x10AC,  0x10AD,  0x10AE,
            0x10AF,  0x10B0,  0x10B1,  0x10B2,  0x10B3,  0x10B4,  0x10B5,
            0x10B6,  0x10B7,  0x10B8,  0x10B9,  0x10BA,  0x10BB,  0x10BC,
            0x10BD,  0x10BE,  0x10BF,  0x10C0,  0x10C1,  0x10C2,  0x10C3,
            0x10C4,  0x10C5,  0x10C7,  0x10CD,  0x13A0,  0x13A1,  0x13A2,
            0x13A3,  0x13A4,  0x13A5,  0x13A6,  0x13A7,  0x13A8,  0x13A9,
            0x13AA,  0x13AB,  0x13AC,  0x13AD,  0x13AE,  0x13AF,  0x13B0,
            0x13B1,  0x13B2,  0x13B3,  0x13B4,  0x13B5,  0x13B6,  0x13B7,
            0x13B8,  0x13B9,  0x13BA,  0x13BB,  0x13BC,  0x13BD,  0x13BE,
            0x13BF,  0x13C0,  0x13C1,  0x13C2,  0x13C3,  0x13C4,  0x13C5,
            0x13C6,  0x13C7,  0x13C8,  0x13C9,  0x13CA,  0x13CB,  0x13CC,
            0x13CD,  0x13CE,  0x13CF,  0x13D0,  0x13D1,  0x13D2,  0x13D3,
            0x13D4,  0x13D5,  0x13D6,  0x13D7,  0x13D8,  0x13D9,  0x13DA,
            0x13DB,  0x13DC,  0x13DD,  0x13DE,  0x13DF,  0x13E0,  0x13E1,
            0x13E2,  0x13E3,  0x13E4,  0x13E5,  0x13E6,  0x13E7,  0x13E8,
            0x13E9,  0x13EA,  0x13EB,  0x13EC,  0x13ED,  0x13EE,  0x13EF,
            0x13F0,  0x13F1,  0x13F2,  0x13F3,  0x13F4,  0x13F5,  0x1C90,
            0x1C91,  0x1C92,  0x1C93,  0x1C94,  0x1C95,  0x1C96,  0x1C97,
            0x1C98,  0x1C99,  0x1C9A,  0x1C9B,  0x1C9C,  0x1C9D,  0x1C9E,
            0x1C9F,  0x1CA0,  0x1CA1,  0x1CA2,  0x1CA3,  0x1CA4,  0x1CA5,
            0x1CA6,  0x1CA7,  0x1CA8,  0x1CA9,  0x1CAA,  0x1CAB,  0x1CAC,
            0x1CAD,  0x1CAE,  0x1CAF,  0x1CB0,  0x1CB1,  0x1CB2,  0x1CB3,
            0x1CB4,  0x1CB5,  0x1CB6,  0x1CB7,  0x1CB8,  0x1CB9,  0x1CBA,
            0x1CBD,  0x1CBE,  0x1CBF,  0x1E00,  0x1E02,  0x1E04,  0x1E06,
            0x1E08,  0x1E0A,  0x1E0C,  0x1E0E,  0x1E10,  0x1E12,  0x1E14,
            0x1E16,  0x1E18,  0x1E1A,  0x1E1C,  0x1E1E,  0x1E20,  0x1E22,
            0x1E24,  0x1E26,  0x1E28,  0x1E2A,  0x1E2C,  0x1E2E,  0x1E30,
            0x1E32,  0x1E34,  0x1E36,  0x1E38,  0x1E3A,  0x1E3C,  0x1E3E,
            0x1E40,  0x1E42,  0x1E44,  0x1E46,  0x1E48,  0x1E4A,  0x1E4C,
            0x1E4E,  0x1E50,  0x1E52,  0x1E54,  0x1E56,  0x1E58,  0x1E5A,
            0x1E5C,  0x1E5E,  0x1E60,  0x1E62,  0x1E64,  0x1E66,  0x1E68,
            0x1E6A,  0x1E6C,  0x1E6E,  0x1E70,  0x1E72,  0x1E74,  0x1E76,
            0x1E78,  0x1E7A,  0x1E7C,  0x1E7E,  0x1E80,  0x1E82,  0x1E84,
            0x1E86,  0x1E88,  0x1E8A,  0x1E8C,  0x1E8E,  0x1E90,  0x1E92,
            0x1E94,  0x1E9E,  0x1EA0,  0x1EA2,  0x1EA4,  0x1EA6,  0x1EA8,
            0x1EAA,  0x1EAC,  0x1EAE,  0x1EB0,  0x1EB2,  0x1EB4,  0x1EB6,
            0x1EB8,  0x1EBA,  0x1EBC,  0x1EBE,  0x1EC0,  0x1EC2,  0x1EC4,
            0x1EC6,  0x1EC8,  0x1ECA,  0x1ECC,  0x1ECE,  0x1ED0,  0x1ED2,
            0x1ED4,  0x1ED6,  0x1ED8,  0x1EDA,  0x1EDC,  0x1EDE,  0x1EE0,
            0x1EE2,  0x1EE4,  0x1EE6,  0x1EE8,  0x1EEA,  0x1EEC,  0x1EEE,
            0x1EF0,  0x1EF2,  0x1EF4,  0x1EF6,  0x1EF8,  0x1EFA,  0x1EFC,
            0x1EFE,  0x1F08,  0x1F09,  0x1F0A,  0x1F0B,  0x1F0C,  0x1F0D,
            0x1F0E,  0x1F0F,  0x1F18,  0x1F19,  0x1F1A,  0x1F1B,  0x1F1C,
            0x1F1D,  0x1F28,  0x1F29,  0x1F2A,  0x1F2B,  0x1F2C,  0x1F2D,
            0x1F2E,  0x1F2F,  0x1F38,  0x1F39,  0x1F3A,  0x1F3B,  0x1F3C,
            0x1F3D,  0x1F3E,  0x1F3F,  0x1F48,  0x1F49,  0x1F4A,  0x1F4B,
            0x1F4C,  0x1F4D,  0x1F59,  0x1F5B,  0x1F5D,  0x1F5F,  0x1F68,
            0x1F69,  0x1F6A,  0x1F6B,  0x1F6C,  0x1F6D,  0x1F6E,  0x1F6F,
            0x1FB8,  0x1FB9,  0x1FBA,  0x1FBB,  0x1FC8,  0x1FC9,  0x1FCA,
            0x1FCB,  0x1FD8,  0x1FD9,  0x1FDA,  0x1FDB,  0x1FE8,  0x1FE9,
            0x1FEA,  0x1FEB,  0x1FEC,  0x1FF8,  0x1FF9,  0x1FFA,  0x1FFB,
            0x2102,  0x2107,  0x210B,  0x210C,  0x210D,  0x2110,  0x2111,
            0x2112,  0x2115,  0x2119,  0x211A,  0x211B,  0x211C,  0x211D,
            0x2124,  0x2126,  0x2128,  0x212A,  0x212B,  0x212C,  0x212D,
            0x2130,  0x2131,  0x2132,  0x2133,  0x213E,  0x213F,  0x2145,
            0x2183,  0x2C00,  0x2C01,  0x2C02,  0x2C03,  0x2C04,  0x2C05,
            0x2C06,  0x2C07,  0x2C08,  0x2C09,  0x2C0A,  0x2C0B,  0x2C0C,
            0x2C0D,  0x2C0E,  0x2C0F,  0x2C10,  0x2C11,  0x2C12,  0x2C13,
            0x2C14,  0x2C15,  0x2C16,  0x2C17,  0x2C18,  0x2C19,  0x2C1A,
            0x2C1B,  0x2C1C,  0x2C1D,  0x2C1E,  0x2C1F,  0x2C20,  0x2C21,
            0x2C22,  0x2C23,  0x2C24,  0x2C25,  0x2C26,  0x2C27,  0x2C28,
            0x2C29,  0x2C2A,  0x2C2B,  0x2C2C,  0x2C2D,  0x2C2E,  0x2C2F,
            0x2C60,  0x2C62,  0x2C63,  0x2C64,  0x2C67,  0x2C69,  0x2C6B,
            0x2C6D,  0x2C6E,  0x2C6F,  0x2C70,  0x2C72,  0x2C75,  0x2C7E,
            0x2C7F,  0x2C80,  0x2C82,  0x2C84,  0x2C86,  0x2C88,  0x2C8A,
            0x2C8C,  0x2C8E,  0x2C90,  0x2C92,  0x2C94,  0x2C96,  0x2C98,
            0x2C9A,  0x2C9C,  0x2C9E,  0x2CA0,  0x2CA2,  0x2CA4,  0x2CA6,
            0x2CA8,  0x2CAA,  0x2CAC,  0x2CAE,  0x2CB0,  0x2CB2,  0x2CB4,
            0x2CB6,  0x2CB8,  0x2CBA,  0x2CBC,  0x2CBE,  0x2CC0,  0x2CC2,
            0x2CC4,  0x2CC6,  0x2CC8,  0x2CCA,  0x2CCC,  0x2CCE,  0x2CD0,
            0x2CD2,  0x2CD4,  0x2CD6,  0x2CD8,  0x2CDA,  0x2CDC,  0x2CDE,
            0x2CE0,  0x2CE2,  0x2CEB,  0x2CED,  0x2CF2,  0xA640,  0xA642,
            0xA644,  0xA646,  0xA648,  0xA64A,  0xA64C,  0xA64E,  0xA650,
            0xA652,  0xA654,  0xA656,  0xA658,  0xA65A,  0xA65C,  0xA65E,
            0xA660,  0xA662,  0xA664,  0xA666,  0xA668,  0xA66A,  0xA66C,
            0xA680,  0xA682,  0xA684,  0xA686,  0xA688,  0xA68A,  0xA68C,
            0xA68E,  0xA690,  0xA692,  0xA694,  0xA696,  0xA698,  0xA69A,
            0xA722,  0xA724,  0xA726,  0xA728,  0xA72A,  0xA72C,  0xA72E,
            0xA732,  0xA734,  0xA736,  0xA738,  0xA73A,  0xA73C,  0xA73E,
            0xA740,  0xA742,  0xA744,  0xA746,  0xA748,  0xA74A,  0xA74C,
            0xA74E,  0xA750,  0xA752,  0xA754,  0xA756,  0xA758,  0xA75A,
            0xA75C,  0xA75E,  0xA760,  0xA762,  0xA764,  0xA766,  0xA768,
            0xA76A,  0xA76C,  0xA76E,  0xA779,  0xA77B,  0xA77D,  0xA77E,
            0xA780,  0xA782,  0xA784,  0xA786,  0xA78B,  0xA78D,  0xA790,
            0xA792,  0xA796,  0xA798,  0xA79A,  0xA79C,  0xA79E,  0xA7A0,
            0xA7A2,  0xA7A4,  0xA7A6,  0xA7A8,  0xA7AA,  0xA7AB,  0xA7AC,
            0xA7AD,  0xA7AE,  0xA7B0,  0xA7B1,  0xA7B2,  0xA7B3,  0xA7B4,
            0xA7B6,  0xA7B8,  0xA7BA,  0xA7BC,  0xA7BE,  0xA7C0,  0xA7C2,
            0xA7C4,  0xA7C5,  0xA7C6,  0xA7C7,  0xA7C9,  0xA7D0,  0xA7D6,
            0xA7D8,  0xA7F5,  0xFF21,  0xFF22,  0xFF23,  0xFF24,  0xFF25,
            0xFF26,  0xFF27,  0xFF28,  0xFF29,  0xFF2A,  0xFF2B,  0xFF2C,
            0xFF2D,  0xFF2E,  0xFF2F,  0xFF30,  0xFF31,  0xFF32,  0xFF33,
            0xFF34,  0xFF35,  0xFF36,  0xFF37,  0xFF38,  0xFF39,  0xFF3A,
            0x10400, 0x10401, 0x10402, 0x10403, 0x10404, 0x10405, 0x10406,
            0x10407, 0x10408, 0x10409, 0x1040A, 0x1040B, 0x1040C, 0x1040D,
            0x1040E, 0x1040F, 0x10410, 0x10411, 0x10412, 0x10413, 0x10414,
            0x10415, 0x10416, 0x10417, 0x10418, 0x10419, 0x1041A, 0x1041B,
            0x1041C, 0x1041D, 0x1041E, 0x1041F, 0x10420, 0x10421, 0x10422,
            0x10423, 0x10424, 0x10425, 0x10426, 0x10427, 0x104B0, 0x104B1,
            0x104B2, 0x104B3, 0x104B4, 0x104B5, 0x104B6, 0x104B7, 0x104B8,
            0x104B9, 0x104BA, 0x104BB, 0x104BC, 0x104BD, 0x104BE, 0x104BF,
            0x104C0, 0x104C1, 0x104C2, 0x104C3, 0x104C4, 0x104C5, 0x104C6,
            0x104C7, 0x104C8, 0x104C9, 0x104CA, 0x104CB, 0x104CC, 0x104CD,
            0x104CE, 0x104CF, 0x104D0, 0x104D1, 0x104D2, 0x104D3, 0x10570,
            0x10571, 0x10572, 0x10573, 0x10574, 0x10575, 0x10576, 0x10577,
            0x10578, 0x10579, 0x1057A, 0x1057C, 0x1057D, 0x1057E, 0x1057F,
            0x10580, 0x10581, 0x10582, 0x10583, 0x10584, 0x10585, 0x10586,
            0x10587, 0x10588, 0x10589, 0x1058A, 0x1058C, 0x1058D, 0x1058E,
            0x1058F, 0x10590, 0x10591, 0x10592, 0x10594, 0x10595, 0x10C80,
            0x10C81, 0x10C82, 0x10C83, 0x10C84, 0x10C85, 0x10C86, 0x10C87,
            0x10C88, 0x10C89, 0x10C8A, 0x10C8B, 0x10C8C, 0x10C8D, 0x10C8E,
            0x10C8F, 0x10C90, 0x10C91, 0x10C92, 0x10C93, 0x10C94, 0x10C95,
            0x10C96, 0x10C97, 0x10C98, 0x10C99, 0x10C9A, 0x10C9B, 0x10C9C,
            0x10C9D, 0x10C9E, 0x10C9F, 0x10CA0, 0x10CA1, 0x10CA2, 0x10CA3,
            0x10CA4, 0x10CA5, 0x10CA6, 0x10CA7, 0x10CA8, 0x10CA9, 0x10CAA,
            0x10CAB, 0x10CAC, 0x10CAD, 0x10CAE, 0x10CAF, 0x10CB0, 0x10CB1,
            0x10CB2, 0x118A0, 0x118A1, 0x118A2, 0x118A3, 0x118A4, 0x118A5,
            0x118A6, 0x118A7, 0x118A8, 0x118A9, 0x118AA, 0x118AB, 0x118AC,
            0x118AD, 0x118AE, 0x118AF, 0x118B0, 0x118B1, 0x118B2, 0x118B3,
            0x118B4, 0x118B5, 0x118B6, 0x118B7, 0x118B8, 0x118B9, 0x118BA,
            0x118BB, 0x118BC, 0x118BD, 0x118BE, 0x118BF, 0x16E40, 0x16E41,
            0x16E42, 0x16E43, 0x16E44, 0x16E45, 0x16E46, 0x16E47, 0x16E48,
            0x16E49, 0x16E4A, 0x16E4B, 0x16E4C, 0x16E4D, 0x16E4E, 0x16E4F,
            0x16E50, 0x16E51, 0x16E52, 0x16E53, 0x16E54, 0x16E55, 0x16E56,
            0x16E57, 0x16E58, 0x16E59, 0x16E5A, 0x16E5B, 0x16E5C, 0x16E5D,
            0x16E5E, 0x16E5F, 0x1D400, 0x1D401, 0x1D402, 0x1D403, 0x1D404,
            0x1D405, 0x1D406, 0x1D407, 0x1D408, 0x1D409, 0x1D40A, 0x1D40B,
            0x1D40C, 0x1D40D, 0x1D40E, 0x1D40F, 0x1D410, 0x1D411, 0x1D412,
            0x1D413, 0x1D414, 0x1D415, 0x1D416, 0x1D417, 0x1D418, 0x1D419,
            0x1D434, 0x1D435, 0x1D436, 0x1D437, 0x1D438, 0x1D439, 0x1D43A,
            0x1D43B, 0x1D43C, 0x1D43D, 0x1D43E, 0x1D43F, 0x1D440, 0x1D441,
            0x1D442, 0x1D443, 0x1D444, 0x1D445, 0x1D446, 0x1D447, 0x1D448,
            0x1D449, 0x1D44A, 0x1D44B, 0x1D44C, 0x1D44D, 0x1D468, 0x1D469,
            0x1D46A, 0x1D46B, 0x1D46C, 0x1D46D, 0x1D46E, 0x1D46F, 0x1D470,
            0x1D471, 0x1D472, 0x1D473, 0x1D474, 0x1D475, 0x1D476, 0x1D477,
            0x1D478, 0x1D479, 0x1D47A, 0x1D47B, 0x1D47C, 0x1D47D, 0x1D47E,
            0x1D47F, 0x1D480, 0x1D481, 0x1D49C, 0x1D49E, 0x1D49F, 0x1D4A2,
            0x1D4A5, 0x1D4A6, 0x1D4A9, 0x1D4AA, 0x1D4AB, 0x1D4AC, 0x1D4AE,
            0x1D4AF, 0x1D4B0, 0x1D4B1, 0x1D4B2, 0x1D4B3, 0x1D4B4, 0x1D4B5,
            0x1D4D0, 0x1D4D1, 0x1D4D2, 0x1D4D3, 0x1D4D4, 0x1D4D5, 0x1D4D6,
            0x1D4D7, 0x1D4D8, 0x1D4D9, 0x1D4DA, 0x1D4DB, 0x1D4DC, 0x1D4DD,
            0x1D4DE, 0x1D4DF, 0x1D4E0, 0x1D4E1, 0x1D4E2, 0x1D4E3, 0x1D4E4,
            0x1D4E5, 0x1D4E6, 0x1D4E7, 0x1D4E8, 0x1D4E9, 0x1D504, 0x1D505,
            0x1D507, 0x1D508, 0x1D509, 0x1D50A, 0x1D50D, 0x1D50E, 0x1D50F,
            0x1D510, 0x1D511, 0x1D512, 0x1D513, 0x1D514, 0x1D516, 0x1D517,
            0x1D518, 0x1D519, 0x1D51A, 0x1D51B, 0x1D51C, 0x1D538, 0x1D539,
            0x1D53B, 0x1D53C, 0x1D53D, 0x1D53E, 0x1D540, 0x1D541, 0x1D542,
            0x1D543, 0x1D544, 0x1D546, 0x1D54A, 0x1D54B, 0x1D54C, 0x1D54D,
            0x1D54E, 0x1D54F, 0x1D550, 0x1D56C, 0x1D56D, 0x1D56E, 0x1D56F,
            0x1D570, 0x1D571, 0x1D572, 0x1D573, 0x1D574, 0x1D575, 0x1D576,
            0x1D577, 0x1D578, 0x1D579, 0x1D57A, 0x1D57B, 0x1D57C, 0x1D57D,
            0x1D57E, 0x1D57F, 0x1D580, 0x1D581, 0x1D582, 0x1D583, 0x1D584,
            0x1D585, 0x1D5A0, 0x1D5A1, 0x1D5A2, 0x1D5A3, 0x1D5A4, 0x1D5A5,
            0x1D5A6, 0x1D5A7, 0x1D5A8, 0x1D5A9, 0x1D5AA, 0x1D5AB, 0x1D5AC,
            0x1D5AD, 0x1D5AE, 0x1D5AF, 0x1D5B0, 0x1D5B1, 0x1D5B2, 0x1D5B3,
            0x1D5B4, 0x1D5B5, 0x1D5B6, 0x1D5B7, 0x1D5B8, 0x1D5B9, 0x1D5D4,
            0x1D5D5, 0x1D5D6, 0x1D5D7, 0x1D5D8, 0x1D5D9, 0x1D5DA, 0x1D5DB,
            0x1D5DC, 0x1D5DD, 0x1D5DE, 0x1D5DF, 0x1D5E0, 0x1D5E1, 0x1D5E2,
            0x1D5E3, 0x1D5E4, 0x1D5E5, 0x1D5E6, 0x1D5E7, 0x1D5E8, 0x1D5E9,
            0x1D5EA, 0x1D5EB, 0x1D5EC, 0x1D5ED, 0x1D608, 0x1D609, 0x1D60A,
            0x1D60B, 0x1D60C, 0x1D60D, 0x1D60E, 0x1D60F, 0x1D610, 0x1D611,
            0x1D612, 0x1D613, 0x1D614, 0x1D615, 0x1D616, 0x1D617, 0x1D618,
            0x1D619, 0x1D61A, 0x1D61B, 0x1D61C, 0x1D61D, 0x1D61E, 0x1D61F,
            0x1D620, 0x1D621, 0x1D63C, 0x1D63D, 0x1D63E, 0x1D63F, 0x1D640,
            0x1D641, 0x1D642, 0x1D643, 0x1D644, 0x1D645, 0x1D646, 0x1D647,
            0x1D648, 0x1D649, 0x1D64A, 0x1D64B, 0x1D64C, 0x1D64D, 0x1D64E,
            0x1D64F, 0x1D650, 0x1D651, 0x1D652, 0x1D653, 0x1D654, 0x1D655,
            0x1D670, 0x1D671, 0x1D672, 0x1D673, 0x1D674, 0x1D675, 0x1D676,
            0x1D677, 0x1D678, 0x1D679, 0x1D67A, 0x1D67B, 0x1D67C, 0x1D67D,
            0x1D67E, 0x1D67F, 0x1D680, 0x1D681, 0x1D682, 0x1D683, 0x1D684,
            0x1D685, 0x1D686, 0x1D687, 0x1D688, 0x1D689, 0x1D6A8, 0x1D6A9,
            0x1D6AA, 0x1D6AB, 0x1D6AC, 0x1D6AD, 0x1D6AE, 0x1D6AF, 0x1D6B0,
            0x1D6B1, 0x1D6B2, 0x1D6B3, 0x1D6B4, 0x1D6B5, 0x1D6B6, 0x1D6B7,
            0x1D6B8, 0x1D6B9, 0x1D6BA, 0x1D6BB, 0x1D6BC, 0x1D6BD, 0x1D6BE,
            0x1D6BF, 0x1D6C0, 0x1D6E2, 0x1D6E3, 0x1D6E4, 0x1D6E5, 0x1D6E6,
            0x1D6E7, 0x1D6E8, 0x1D6E9, 0x1D6EA, 0x1D6EB, 0x1D6EC, 0x1D6ED,
            0x1D6EE, 0x1D6EF, 0x1D6F0, 0x1D6F1, 0x1D6F2, 0x1D6F3, 0x1D6F4,
            0x1D6F5, 0x1D6F6, 0x1D6F7, 0x1D6F8, 0x1D6F9, 0x1D6FA, 0x1D71C,
            0x1D71D, 0x1D71E, 0x1D71F, 0x1D720, 0x1D721, 0x1D722, 0x1D723,
            0x1D724, 0x1D725, 0x1D726, 0x1D727, 0x1D728, 0x1D729, 0x1D72A,
            0x1D72B, 0x1D72C, 0x1D72D, 0x1D72E, 0x1D72F, 0x1D730, 0x1D731,
            0x1D732, 0x1D733, 0x1D734, 0x1D756, 0x1D757, 0x1D758, 0x1D759,
            0x1D75A, 0x1D75B, 0x1D75C, 0x1D75D, 0x1D75E, 0x1D75F, 0x1D760,
            0x1D761, 0x1D762, 0x1D763, 0x1D764, 0x1D765, 0x1D766, 0x1D767,
            0x1D768, 0x1D769, 0x1D76A, 0x1D76B, 0x1D76C, 0x1D76D, 0x1D76E,
            0x1D790, 0x1D791, 0x1D792, 0x1D793, 0x1D794, 0x1D795, 0x1D796,
            0x1D797, 0x1D798, 0x1D799, 0x1D79A, 0x1D79B, 0x1D79C, 0x1D79D,
            0x1D79E, 0x1D79F, 0x1D7A0, 0x1D7A1, 0x1D7A2, 0x1D7A3, 0x1D7A4,
            0x1D7A5, 0x1D7A6, 0x1D7A7, 0x1D7A8, 0x1D7CA, 0x1E900, 0x1E901,
            0x1E902, 0x1E903, 0x1E904, 0x1E905, 0x1E906, 0x1E907, 0x1E908,
            0x1E909, 0x1E90A, 0x1E90B, 0x1E90C, 0x1E90D, 0x1E90E, 0x1E90F,
            0x1E910, 0x1E911, 0x1E912, 0x1E913, 0x1E914, 0x1E915, 0x1E916,
            0x1E917, 0x1E918, 0x1E919, 0x1E91A, 0x1E91B, 0x1E91C, 0x1E91D,
            0x1E91E, 0x1E91F, 0x1E920, 0x1E921};
    };
}

#endif
